#pragma once
/*
 * This file is part of the libCEC(R) library.
 *
 * libCEC(R) is Copyright (C) 2011 Pulse-Eight Limited.  All rights reserved.
 * libCEC(R) is an original work, containing original code.
 *
 * libCEC(R) is a trademark of Pulse-Eight Limited.
 *
 * This program is dual-licensed; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * Alternatively, you can license this library under a commercial license,
 * please contact Pulse-Eight Licensing for more information.
 *
 * For more information contact:
 * Pulse-Eight Licensing       <license@pulse-eight.com>
 *     http://www.pulse-eight.com/
 *     http://www.pulse-eight.net/
 */

namespace PLATFORM
{
  inline pthread_mutexattr_t *GetRecursiveMutexAttribute(void)
  {
    static pthread_mutexattr_t g_mutexAttr;
    static bool bAttributeInitialised = false;
    if (!bAttributeInitialised)
    {
      pthread_mutexattr_init(&g_mutexAttr);
      pthread_mutexattr_settype(&g_mutexAttr, PTHREAD_MUTEX_RECURSIVE);
      bAttributeInitialised = true;
    }
    return &g_mutexAttr;
  }

  inline struct timespec GetAbsTime(uint64_t iIncreaseBy = 0)
  {
    struct timespec abstime;
    struct timeval now;
    gettimeofday(&now, NULL);
    iIncreaseBy += now.tv_usec / 1000;
    abstime.tv_sec = now.tv_sec + (time_t)(iIncreaseBy / 1000);
    abstime.tv_nsec = (int32_t)((iIncreaseBy % (uint32_t)1000) * (uint32_t)1000000);
    return abstime;
  }

  typedef pthread_t thread_t;

  #define ThreadsCreate(thread, func, arg)         (pthread_create(&thread, NULL, (void *(*) (void *))func, (void *)arg) == 0)
  #define ThreadsWait(thread, retval)              (pthread_join(thread, retval) == 0)

  typedef pthread_mutex_t mutex_t;
  #define MutexCreate(mutex)                       pthread_mutex_init(&mutex, GetRecursiveMutexAttribute());
  #define MutexDelete(mutex)                       pthread_mutex_destroy(&mutex);
  #define MutexLock(mutex)                         (pthread_mutex_lock(&mutex) == 0)
  #define MutexTryLock(mutex)                      (pthread_mutex_trylock(&mutex) == 0)
  #define MutexUnlock(mutex)                       pthread_mutex_unlock(&mutex)

  typedef pthread_cond_t condition_t;
  #define ConditionCreate(cond)                    pthread_cond_init(&cond, NULL)
  #define ConditionDelete(cond)                    pthread_cond_destroy(&cond)
  #define ConditionSignal(cond)                    pthread_cond_signal(&cond)
  #define ConditionBroadcast(cond)                 pthread_cond_broadcast(&cond)
  inline bool ConditionWait(condition_t &cond, mutex_t &mutex, uint32_t iTimeout)
  {
    sched_yield();
    if (iTimeout > 0)
    {
      struct timespec timeout = GetAbsTime(iTimeout);
      return (pthread_cond_timedwait(&cond, &mutex, &timeout) == 0);
    }
    return (pthread_cond_wait(&cond, &mutex) == 0);
  }
}
