// This module implements the TQextScintillaLexerRuby class.
//
// Copyright (c) 2006
// 	Riverbank Computing Limited <info@riverbankcomputing.co.uk>
// 
// This file is part of TQScintilla.
// 
// This copy of TQScintilla is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option) any
// later version.
// 
// TQScintilla is supplied in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
// details.
// 
// You should have received a copy of the GNU General Public License along with
// TQScintilla; see the file LICENSE.  If not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <tqcolor.h>
#include <tqfont.h>
#include <tqsettings.h>

#include "tqextscintillalexerruby.h"


// The ctor.
TQextScintillaLexerRuby::TQextScintillaLexerRuby(TQObject *parent,
		const char *name)
	: TQextScintillaLexer(parent, name)
{
}


// The dtor.
TQextScintillaLexerRuby::~TQextScintillaLexerRuby()
{
}


// Returns the language name.
const char *TQextScintillaLexerRuby::language() const
{
	return "Ruby";
}


// Returns the lexer name.
const char *TQextScintillaLexerRuby::lexer() const
{
	return "ruby";
}


// Return the list of words that can start a block.
const char *TQextScintillaLexerRuby::blockStart(int *style) const
{
	if (style)
		*style = Keyword;

	return "do";
}


// Return the list of words that can start end a block.
const char *TQextScintillaLexerRuby::blockEnd(int *style) const
{
	if (style)
		*style = Keyword;

	return "end";
}


// Return the list of words that can start end a block.
const char *TQextScintillaLexerRuby::blockStartKeyword(int *style) const
{
	if (style)
		*style = Keyword;

	return "def class if do elsif else case while for";
}


// Return the style used for braces.
int TQextScintillaLexerRuby::braceStyle() const
{
	return Operator;
}


// Returns the foreground colour of the text for a style.
TQColor TQextScintillaLexerRuby::color(int style) const
{
	switch (style)
	{
	case Default:
		return TQColor(0x80,0x80,0x80);

	case Comment:
		return TQColor(0x00,0x7f,0x00);

	case POD:
		return TQColor(0x00,0x40,0x00);

	case Number:
	case FunctionMethodName:
		return TQColor(0x00,0x7f,0x7f);

	case Keyword:
	case DemotedKeyword:
		return TQColor(0x00,0x00,0x7f);

	case DoubleQuotedString:
	case SingleQuotedString:
	case HereDocument:
	case PercentStringq:
	case PercentStringQ:
		return TQColor(0x7f,0x00,0x7f);

	case ClassName:
		return TQColor(0x00,0x00,0xff);

	case Regex:
	case HereDocumentDelimiter:
	case PercentStringr:
	case PercentStringw:
		return TQColor(0x00,0x00,0x00);

	case Global:
		return TQColor(0x80,0x00,0x80);

	case Symbol:
		return TQColor(0xc0,0xa0,0x30);

	case ModuleName:
		return TQColor(0xa0,0x00,0xa0);

	case InstanceVariable:
		return TQColor(0xb0,0x00,0x80);

	case ClassVariable:
		return TQColor(0x80,0x00,0xb0);

	case Backticks:
	case PercentStringx:
		return TQColor(0xff,0xff,0x00);

	case DataSection:
		return TQColor(0x60,0x00,0x00);
	}

	return TQextScintillaLexer::color(style);
}


// Returns the end-of-line fill for a style.
bool TQextScintillaLexerRuby::eolFill(int style) const
{
	bool fill;

	switch (style)
	{
	case POD:
	case DataSection:
	case HereDocument:
		fill = true;
		break;

	default:
		fill = false;
	}

	return fill;
}


// Returns the font of the text for a style.
TQFont TQextScintillaLexerRuby::font(int style) const
{
	TQFont f;

	switch (style)
	{
	case Comment:
#if defined(Q_OS_WIN)
		f = TQFont("Comic Sans MS",9);
#else
		f = TQFont("Bitstream Vera Serif",9);
#endif
		break;

	case POD:
	case DoubleQuotedString:
	case SingleQuotedString:
	case PercentStringq:
	case PercentStringQ:
#if defined(Q_OS_WIN)
		f = TQFont("Courier New",10);
#else
		f = TQFont("Bitstream Vera Sans Mono",9);
#endif
		break;

	case Keyword:
	case ClassName:
	case FunctionMethodName:
	case Operator:
	case ModuleName:
	case DemotedKeyword:
		f = TQextScintillaLexer::font(style);
		f.setBold(true);
		break;

	default:
		f = TQextScintillaLexer::font(style);
	}

	return f;
}


// Returns the set of keywords.
const char *TQextScintillaLexerRuby::keywords(int set) const
{
	if (set == 1)
		return
			"__FILE__ and def end in or self unless __LINE__ "
			"begin defined? ensure module redo super until BEGIN "
			"break do false next rescue then when END case else "
			"for nil require retry true while alias class elsif "
			"if not return undef yield";

	return 0;
}


// Returns the user name of a style.
TQString TQextScintillaLexerRuby::description(int style) const
{
	switch (style)
	{
	case Default:
		return tr("Default");

	case Error:
		return tr("Error");

	case Comment:
		return tr("Comment");

	case POD:
		return tr("POD");

	case Number:
		return tr("Number");

	case Keyword:
		return tr("Keyword");

	case DoubleQuotedString:
		return tr("Double-quoted string");

	case SingleQuotedString:
		return tr("Single-quoted string");

	case ClassName:
		return tr("Class name");

	case FunctionMethodName:
		return tr("Function or method name");

	case Operator:
		return tr("Operator");

	case Identifier:
		return tr("Identifier");

	case Regex:
		return tr("Regular expression");

	case Global:
		return tr("Global");

	case Symbol:
		return tr("Symbol");

	case ModuleName:
		return tr("Module name");

	case InstanceVariable:
		return tr("Instance variable");

	case ClassVariable:
		return tr("Class variable");

	case Backticks:
		return tr("Backticks");

	case DataSection:
		return tr("Data section");

	case HereDocumentDelimiter:
		return tr("Here document delimiter");

	case HereDocument:
		return tr("Here document");

	case PercentStringq:
		return tr("%q string");

	case PercentStringQ:
		return tr("%Q string");

	case PercentStringx:
		return tr("%x string");

	case PercentStringr:
		return tr("%r string");

	case PercentStringw:
		return tr("%w string");

	case DemotedKeyword:
		return tr("Demoted keyword");

	case Stdin:
		return tr("stdin");

	case Stdout:
		return tr("stdout");

	case Stderr:
		return tr("stderr");
	}

	return TQString();
}


// Returns the background colour of the text for a style.
TQColor TQextScintillaLexerRuby::paper(int style) const
{
	switch (style)
	{
	case Error:
		return TQColor(0xff,0x00,0x00);

	case POD:
		return TQColor(0xc0,0xff,0xc0);

	case Regex:
	case PercentStringr:
		return TQColor(0xa0,0xff,0xa0);

	case Backticks:
	case PercentStringx:
		return TQColor(0xa0,0x80,0x80);

	case DataSection:
		return TQColor(0xff,0xf0,0xd8);

	case HereDocumentDelimiter:
	case HereDocument:
		return TQColor(0xdd,0xd0,0xdd);

	case PercentStringw:
		return TQColor(0xff,0xff,0xe0);

	case Stdin:
	case Stdout:
	case Stderr:
		return TQColor(0xff,0x80,0x80);
	}

	return TQextScintillaLexer::paper(style);
}

#include "tqextscintillalexerruby.moc"
