/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2006-11-20
 * Description : a dialog to batch edit comments
 *
 * Copyright (C) 2006-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.

#include <tqlabel.h>
#include <tqframe.h>
#include <tqlayout.h>
#include <tqpushbutton.h>
#include <tqcheckbox.h>
#include <tqwhatsthis.h>

// KDE includes.

#include <tdelocale.h>
#include <tdeconfig.h>
#include <kdebug.h>
#include <ktextedit.h>
#include <kiconloader.h>
#include <tdeapplication.h>
#include <khelpmenu.h>
#include <tdepopupmenu.h>

// Local includes.

#include "pluginsversion.h"
#include "kpaboutdata.h"
#include "commenteditdialog.h"
#include "commenteditdialog.moc"

namespace KIPIMetadataEditPlugin
{

class CommentEditDialogDialogPrivate
{

public:

    CommentEditDialogDialogPrivate()
    {
        userCommentEdit      = 0;
        about                = 0;
        syncJFIFCommentCheck = 0;
        syncEXIFCommentCheck = 0;
        syncIPTCCaptionCheck = 0;
    }

    TQCheckBox                *syncJFIFCommentCheck;
    TQCheckBox                *syncEXIFCommentCheck;
    TQCheckBox                *syncIPTCCaptionCheck;

    KTextEdit                *userCommentEdit;

    KIPIPlugins::KPAboutData *about;
};

CommentEditDialog::CommentEditDialog(TQWidget* parent)
                 : KDialogBase(Plain, i18n("Edit Image Caption"),
                               Help|Ok|Cancel, Ok,
                               parent, 0, true, true)
{
    d = new CommentEditDialogDialogPrivate;

    // ---------------------------------------------------------------
    // About data and help button.

    d->about = new KIPIPlugins::KPAboutData(I18N_NOOP("Edit Metadata"),
                                            0,
                                            TDEAboutData::License_GPL,
                                            I18N_NOOP("A Plugin to edit images' metadata"),
                                            "(c) 2006-2008, Gilles Caulier");

    d->about->addAuthor("Gilles Caulier", I18N_NOOP("Author and Maintainer"),
                        "caulier dot gilles at gmail dot com");

    KHelpMenu* helpMenu = new KHelpMenu(this, d->about, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("Plugin Handbook"),
                                 this, TQ_SLOT(slotHelp()), 0, -1, 0);
    actionButton(Help)->setPopup( helpMenu->menu() );

    // ------------------------------------------------------------

    TQVBoxLayout *vlay = new TQVBoxLayout(plainPage(), 0, KDialog::spacingHint());

    TQLabel *title = new TQLabel(i18n("<p>Enter the image caption hosted by <b>%1</b>. "
                                    "This field is not limited (excepted with IPTC). UTF-8 encoding "
                                    "will be used to save text.")
                                    .arg(TDEApplication::kApplication()->aboutData()->appName()),
                               plainPage());

    d->userCommentEdit  = new KTextEdit(plainPage());

    d->syncJFIFCommentCheck = new TQCheckBox(i18n("Sync JFIF Comment section"), plainPage());
    d->syncEXIFCommentCheck = new TQCheckBox(i18n("Sync EXIF Comment"), plainPage());
    d->syncIPTCCaptionCheck = new TQCheckBox(i18n("Sync IPTC caption (warning: limited to 2000 printable "
                                                 "Ascii characters set)"), plainPage());

    TQLabel *note = new TQLabel(i18n("<b>Note: captions from currently selected images "
                                   "will be permanently replaced.</b>"), plainPage());
    
    // ------------------------------------------------------------

    vlay->addWidget(title);
    vlay->addWidget(d->userCommentEdit);
    vlay->addWidget(d->syncJFIFCommentCheck);
    vlay->addWidget(d->syncEXIFCommentCheck);
    vlay->addWidget(d->syncIPTCCaptionCheck);
    vlay->addWidget(note);

    // ------------------------------------------------------------

    readSettings();
}

CommentEditDialog::~CommentEditDialog()
{
    delete d->about;
    delete d;
}

void CommentEditDialog::slotHelp()
{
    TDEApplication::kApplication()->invokeHelp("metadataedit", "kipi-plugins");
}

void CommentEditDialog::closeEvent(TQCloseEvent *e)
{
    if (!e) return;
    saveSettings();
    e->accept();
}

void CommentEditDialog::slotCancel()
{
    saveSettings();
    KDialogBase::slotCancel();
}

void CommentEditDialog::readSettings()
{
    TDEConfig config("kipirc");
    config.setGroup("Comments Edit Settings");
    setCheckedSyncJFIFComment(config.readBoolEntry("Sync JFIF Comment", true));
    setCheckedSyncEXIFComment(config.readBoolEntry("Sync EXIF Comment", true));
    setCheckedSyncIPTCCaption(config.readBoolEntry("Sync IPTC Caption", true));
    resize(configDialogSize(config, TQString("Comments Edit Dialog")));
}

void CommentEditDialog::saveSettings()
{
    TDEConfig config("kipirc");
    config.setGroup("Comments Edit Settings");
    config.writeEntry("Sync JFIF Comment", syncJFIFCommentIsChecked());
    config.writeEntry("Sync EXIF Comment", syncEXIFCommentIsChecked());
    config.writeEntry("Sync IPTC Caption", syncIPTCCaptionIsChecked());
    saveDialogSize(config, TQString("Comments Edit Dialog"));
    config.sync();
}

void CommentEditDialog::slotOk()
{
    saveSettings();
    accept();
}

bool CommentEditDialog::syncJFIFCommentIsChecked()
{
    return d->syncJFIFCommentCheck->isChecked();
}

bool CommentEditDialog::syncEXIFCommentIsChecked()
{
    return d->syncEXIFCommentCheck->isChecked();
}

bool CommentEditDialog::syncIPTCCaptionIsChecked()
{
    return d->syncIPTCCaptionCheck->isChecked();
}

TQString CommentEditDialog::getComments()
{
    return d->userCommentEdit->text();
}

void CommentEditDialog::setCheckedSyncJFIFComment(bool c)
{
    d->syncJFIFCommentCheck->setChecked(c);
}

void CommentEditDialog::setCheckedSyncEXIFComment(bool c)
{
    d->syncEXIFCommentCheck->setChecked(c);
}

void CommentEditDialog::setCheckedSyncIPTCCaption(bool c)
{
    d->syncIPTCCaptionCheck->setChecked(c);
}

}  // namespace KIPIMetadataEditPlugin
