/**********************************************************************
** Copyright (C) 2000 Trolltech AS.  All rights reserved.
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid TQt Enterprise Edition or TQt Professional Edition
** licenses may use this file in accordance with the TQt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about TQt Commercial License Agreements.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include <tqvariant.h>  // HP-UX compiler needs this here
#include "styledbutton.h"
#include "formwindow.h"
#include "pixmapchooser.h"
#include <tqcolordialog.h>
#include <tqpalette.h>
#include <tqlabel.h>
#include <tqpainter.h>
#include <tqimage.h>
#include <tqpixmap.h>
#include <tqapplication.h>
#include <tqdragobject.h>
#include <tqstyle.h>

StyledButton::StyledButton(TQWidget* parent, const char* name)
    : TQButton( parent, name ), pix( 0 ), spix( 0 ), s( 0 ), formWindow( 0 ), mousePressed( FALSE )
{
    setMinimumSize( minimumSizeHint() );
    setAcceptDrops( TRUE );

    connect( this, TQ_SIGNAL(clicked()), TQ_SLOT(onEditor()));

    setEditor( ColorEditor );
}

StyledButton::StyledButton( const TQBrush& b, TQWidget* parent, const char* name, WFlags f )
    : TQButton( parent, name, f ), spix( 0 ), s( 0 ), formWindow( 0 )
{
    col = b.color();
    pix = b.pixmap();
    setMinimumSize( minimumSizeHint() );
}

StyledButton::~StyledButton()
{
}

void StyledButton::setEditor( EditorType e )
{
    if ( edit == e )
	return;

    edit = e;
    update();
}

StyledButton::EditorType StyledButton::editor() const
{
    return edit;
}

void StyledButton::setColor( const TQColor& c )
{
    col = c;
    update();
}

void StyledButton::setPixmap( const TQPixmap & pm )
{
    if ( !pm.isNull() ) {
	delete pix;
	pix = new TQPixmap( pm );
    } else {
	delete pix;
	pix = 0;
    }
    scalePixmap();
}

TQColor StyledButton::color() const
{
    return col;
}

TQPixmap* StyledButton::pixmap() const
{
    return pix;
}

bool StyledButton::scale() const
{
    return s;
}

void StyledButton::setScale( bool on )
{
    if ( s == on )
	return;

    s = on;
    scalePixmap();
}

TQSize StyledButton::sizeHint() const
{
    return TQSize( 50, 25 );
}

TQSize StyledButton::minimumSizeHint() const
{
    return TQSize( 50, 25 );
}

void StyledButton::scalePixmap()
{
    delete spix;

    if ( pix ) {
	spix = new TQPixmap( 6*width()/8, 6*height()/8 );
	TQImage img = pix->convertToImage();

	spix->convertFromImage( s? img.smoothScale( 6*width()/8, 6*height()/8 ) : img );
    } else {
	spix = 0;
    }

    update();
}

void StyledButton::resizeEvent( TQResizeEvent* e )
{
    scalePixmap();
    TQButton::resizeEvent( e );
}

void StyledButton::drawButton( TQPainter *paint )
{
    style().drawPrimitive(TQStyle::PE_ButtonBevel, paint, rect(), colorGroup(),
			  isDown() ? TQStyle::Style_Sunken : TQStyle::Style_Raised);
    drawButtonLabel(paint);

    if (hasFocus())
	style().drawPrimitive(TQStyle::PE_FocusRect, paint,
			      style().subRect(TQStyle::SR_PushButtonFocusRect, this),
			      colorGroup(), TQStyle::Style_Default);
}

void StyledButton::drawButtonLabel( TQPainter *paint )
{
    TQColor pen = isEnabled() ?
		 hasFocus() ? palette().active().buttonText() : palette().inactive().buttonText()
		 : palette().disabled().buttonText();
    paint->setPen( pen );

    if(!isEnabled()) {
	paint->setBrush( TQBrush( colorGroup().button() ) );
    }
    else if ( edit == PixmapEditor && spix ) {
	paint->setBrush( TQBrush( col, *spix ) );
	paint->setBrushOrigin( width()/8, height()/8 );
    } else
	paint->setBrush( TQBrush( col ) );

    paint->drawRect( width()/8, height()/8, 6*width()/8, 6*height()/8 );
}

void StyledButton::onEditor()
{
    switch (edit) {
    case ColorEditor: {
	TQColor c = TQColorDialog::getColor( palette().active().background(), this );
	if ( c.isValid() ) {
	    setColor( c );
	    emit changed();
	}
    } break;
    case PixmapEditor: {
	TQPixmap p;
        if ( pixmap() )
		p = qChoosePixmap( this, formWindow, *pixmap() );
        else
		p = qChoosePixmap( this, formWindow, TQPixmap() );
	if ( !p.isNull() ) {
	    setPixmap( p );
	    emit changed();
	}
    } break;
    default:
	break;
    }
}

void StyledButton::mousePressEvent(TQMouseEvent* e)
{
    TQButton::mousePressEvent(e);
    mousePressed = TRUE;
    pressPos = e->pos();
}

void StyledButton::mouseMoveEvent(TQMouseEvent* e)
{
    TQButton::mouseMoveEvent( e );
#ifndef TQT_NO_DRAGANDDROP
    if ( !mousePressed )
	return;
    if ( ( pressPos - e->pos() ).manhattanLength() > TQApplication::startDragDistance() ) {
	if ( edit == ColorEditor ) {
	    TQColorDrag *drg = new TQColorDrag( col, this );
	    TQPixmap pix( 25, 25 );
	    pix.fill( col );
	    TQPainter p( &pix );
	    p.drawRect( 0, 0, pix.width(), pix.height() );
	    p.end();
	    drg->setPixmap( pix );
	    mousePressed = FALSE;
	    drg->dragCopy();
	}
	else if ( edit == PixmapEditor && pix && !pix->isNull() ) {
	    TQImage img = pix->convertToImage();
	    TQImageDrag *drg = new TQImageDrag( img, this );
	    if(spix)
		drg->setPixmap( *spix );
	    mousePressed = FALSE;
	    drg->dragCopy();
	}
    }
#endif
}

#ifndef TQT_NO_DRAGANDDROP
void StyledButton::dragEnterEvent( TQDragEnterEvent *e )
{
    setFocus();
    if ( edit == ColorEditor && TQColorDrag::canDecode( e ) )
	e->accept();
    else if ( edit == PixmapEditor && TQImageDrag::canDecode( e ) )
	e->accept();
    else
	e->ignore();
}

void StyledButton::dragLeaveEvent( TQDragLeaveEvent * )
{
    if ( hasFocus() )
	parentWidget()->setFocus();
}

void StyledButton::dragMoveEvent( TQDragMoveEvent *e )
{
    if ( edit == ColorEditor && TQColorDrag::canDecode( e ) )
	e->accept();
    else if ( edit == PixmapEditor && TQImageDrag::canDecode( e ) )
	e->accept();
    else
	e->ignore();
}

void StyledButton::dropEvent( TQDropEvent *e )
{
    if ( edit == ColorEditor && TQColorDrag::canDecode( e ) ) {
	TQColor color;
	TQColorDrag::decode( e, color );
	setColor(color);
	emit changed();
	e->accept();
    }
    else if ( edit == PixmapEditor && TQImageDrag::canDecode( e ) ) {
	TQImage img;
	TQImageDrag::decode( e, img );
	TQPixmap pm;
	pm.convertFromImage(img);
	setPixmap(pm);
	emit changed();
	e->accept();
    } else {
	e->ignore();
    }
}
#endif // TQT_NO_DRAGANDDROP
