// Copyright (c) 2003,2004 Charles Samuels <charles@kde.org>
// See the file COPYING for redistribution terms.

#include "menu.h"
#include <kiconloader.h>
#include <tdelocale.h>

#include "tree.h"

FileMenu::FileMenu(TQWidget *parent, Oblique *oblique, File file)
	: TDEPopupMenu(parent)
{
	if (file)
		mFiles.append(file);
	insertItem(
			BarIconSet("delete"), i18n("&Remove From Playlist"),
			this, TQ_SLOT(removeFromList())
		);
	insertItem(i18n("&Properties"), this, TQ_SLOT(properties()));

	(new SliceListAction(
			i18n("&Slices"), oblique,
			this, TQ_SLOT(toggleInSlice(Slice *)),
			mFiles, this
		))->plug(this);
}

static void addTo(TQValueList<File> &files, TreeItem *item)
{
	File f = item->file();
	if (f) files.append(f);

	item = item->firstChild();

	while (item)
	{
		addTo(files, item);
		item = item->nextSibling();
	}
}

FileMenu::FileMenu(TQWidget *parent, Oblique *oblique, TreeItem *items)
	: TDEPopupMenu(parent)
{
	addTo(mFiles, items);

	insertItem(
			BarIconSet("delete"), i18n("&Remove From Playlist"),
			this, TQ_SLOT(removeFromList())
		);
	insertItem(i18n("&Properties"), this, TQ_SLOT(properties()));

	(new SliceListAction(
			i18n("&Slices"), oblique,
			this, TQ_SLOT(toggleInSlice(Slice *)),
			mFiles,this
		))->plug(this);
}

void FileMenu::removeFromList()
{
	for (TQValueList<File>::Iterator i(mFiles.begin()); i != mFiles.end(); ++i)
	{
		(*i).remove();
	}
}

void FileMenu::properties()
{
	new ObliquePropertiesDialog(mFiles, parentWidget());
}

void FileMenu::toggleInSlice(Slice *slice)
{
	void (File::*task)(Slice*)=0;
	for (TQValueList<File>::Iterator i(mFiles.begin()); i != mFiles.end(); ++i)
	{
		if (!task)
		{ // we check with the first one
			if ((*i).isIn(slice))
				task = &File::removeFrom;
			else
				task = &File::addTo;
		}

		((*i).*task)(slice);
	}
}



SliceListAction::SliceListAction(
		const TQString &text, Oblique *oblique,
		TQObject *reciever, const char *slot,
		const TQValueList<File> &files, TQObject *parent, const char *name
	) : TDEActionMenu(text, parent, name)
{
	mFiles = files;
	mOblique = oblique;
	slicesModified();
	if (reciever)
		connect(this, TQ_SIGNAL(activated(Slice*)), reciever, slot);
	connect(popupMenu(), TQ_SIGNAL(activated(int)), TQ_SLOT(hit(int)));
	connect(oblique->base(), TQ_SIGNAL(slicesModified()), TQ_SLOT(slicesModified()));
}

void SliceListAction::slicesModified()
{
	mIndexToSlices.clear();
	TDEPopupMenu *menu = popupMenu();
	menu->clear();

	TQPtrList<Slice> slices = mOblique->base()->slices();
	int id=1;



	for (TQPtrListIterator<Slice> i(slices); *i; ++i)
	{
		Slice *s = *i;
		if (s->id()==0 && mFiles.count())
		{
			continue;
		}

		menu->insertItem(s->name(), id);
		if (mFiles.count())
		{
			menu->setItemChecked(id, mFiles.first().isIn(s));
		}
//		else if (mOblique->slice() == s) // TODO: show the selected one
//		{
//			menu->setItemChecked(id, true);
//		}

		if (mFiles.count() && s->id() == 0)
		{
			menu->setItemEnabled(id, false);
		}

		mIndexToSlices.insert(id, s);
		id++;
	}
}

void SliceListAction::hit(int index)
{
	emit activated(mIndexToSlices[index]);
}



SchemaListAction::SchemaListAction(
		const TQString &text,
		TQObject *reciever, const char *slot,
		TQObject *parent, const char *name
	) : TDEActionMenu(text, parent, name)
{
	mTree = 0;
	if (reciever)
		connect(this, TQ_SIGNAL(activated(const TQString&)), reciever, slot);
	connect(popupMenu(), TQ_SIGNAL(aboutToShow()), TQ_SLOT(prepare()));
	connect(popupMenu(), TQ_SIGNAL(activated(int)), TQ_SLOT(hit(int)));
}

void SchemaListAction::prepare()
{
	assert(mTree);
	mIndexToSchemas.clear();
	TDEPopupMenu *menu = popupMenu();
	menu->clear();
	
	if (!mTree) return;

	int id=1;

	TQStringList names = mTree->oblique()->schemaNames();

	for (TQStringList::Iterator i(names.begin()); i != names.end(); ++i)
	{
		Query q;
		TQString title = mTree->oblique()->loadSchema(q, *i);

		menu->insertItem(title, id);

		menu->setItemChecked(id, mTree->fileOfQuery() == *i);

		mIndexToSchemas.insert(id, *i);
		id++;
	}
}

void SchemaListAction::hit(int index)
{
	emit activated(mIndexToSchemas[index]);
}



ObliquePropertiesDialog::ObliquePropertiesDialog(const TQValueList<File> &files, TQWidget *parent)
	: KPropertiesDialog(makeItems(files), parent), mFiles(files)
{
	connect(this, TQ_SIGNAL(propertiesClosed()), TQ_SLOT(deleteLater()));
	connect(this, TQ_SIGNAL(applied()), TQ_SLOT(modified()));

	show();
}

void ObliquePropertiesDialog::modified()
{
	// TODO reload the file's info
	for (TQValueList<File>::Iterator i(mFiles.begin()); i != mFiles.end(); ++i)
	{
		(*i).makeCache();
		(*i).base()->notifyChanged(*i);
	}
}

KFileItemList ObliquePropertiesDialog::makeItems(const TQValueList<File> &files)
{
	KFileItemList kl;
	for (TQValueList<File>::ConstIterator i(files.begin()); i != files.end(); ++i)
	{
		File f = *i;
		kl.append(new KFileItem(f.url(), f.property("mimetype"), KFileItem::Unknown));
	}
	return kl;
}

#include "menu.moc"

