/***************************************************************************
    copyright            : (C) 2005-2006 by Robby Stephenson
    email                : robby@periapsis.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License as  *
 *   published by the Free Software Foundation;                            *
 *                                                                         *
 ***************************************************************************/

#include "urlfieldwidget.h"
#include "../field.h"
#include "../latin1literal.h"
#include "../tellico_kernel.h"

#include <klineedit.h>
#include <kurlrequester.h>
#include <kurllabel.h>

using Tellico::GUI::URLFieldWidget;

// subclass of KURLCompletion is needed so the KURLLabel
// can open relative links. I don't want to have to have to update
// the base directory of the completion every time a new document is opened
TQString URLFieldWidget::URLCompletion::makeCompletion(const TQString& text_) {
  // KURLCompletion::makeCompletion() uses an internal variable instead
  // of calling KURLCompletion::dir() so need to set the base dir before completing
  setDir(Kernel::self()->URL().directory());
  return KURLCompletion::makeCompletion(text_);
}

URLFieldWidget::URLFieldWidget(Data::FieldPtr field_, TQWidget* parent_, const char* name_/*=0*/)
    : FieldWidget(field_, parent_, name_), m_run(0) {

  m_requester = new KURLRequester(this);
  m_requester->lineEdit()->setCompletionObject(new URLCompletion());
  m_requester->lineEdit()->setAutoDeleteCompletionObject(true);
  connect(m_requester, TQT_SIGNAL(textChanged(const TQString&)), TQT_SIGNAL(modified()));
  connect(m_requester, TQT_SIGNAL(textChanged(const TQString&)), label(), TQT_SLOT(setURL(const TQString&)));
  connect(label(), TQT_SIGNAL(leftClickedURL(const TQString&)), TQT_SLOT(slotOpenURL(const TQString&)));
  registerWidget();

  // special case, remember if it's a relative url
  m_isRelative = field_->property(TQString::fromLatin1("relative")) == Latin1Literal("true");
}

URLFieldWidget::~URLFieldWidget() {
  if(m_run) {
    m_run->abort();
  }
}

TQString URLFieldWidget::text() const {
  if(m_isRelative) {
    return KURL::relativeURL(Kernel::self()->URL(), m_requester->url());
  }
  // for comparison purposes and to be consistent with the file listing importer
  // I want the full url here, including the protocol
  // the requester only returns the path, so create a KURL
  return KURL(m_requester->url()).url();
}

void URLFieldWidget::setText(const TQString& text_) {
  blockSignals(true);

  m_requester->blockSignals(true);
  m_requester->setURL(text_);
  m_requester->blockSignals(false);
  static_cast<KURLLabel*>(label())->setURL(text_);

  blockSignals(false);
}

void URLFieldWidget::clear() {
  m_requester->clear();
  editMultiple(false);
}

void URLFieldWidget::updateFieldHook(Data::FieldPtr, Data::FieldPtr newField_) {
  m_isRelative = newField_->property(TQString::fromLatin1("relative")) == Latin1Literal("true");
}

void URLFieldWidget::slotOpenURL(const TQString& url_) {
  if(url_.isEmpty()) {
    return;
  }
  // just in case, interpret string relative to document url
  m_run = new KRun(KURL(Kernel::self()->URL(), url_));
}

TQWidget* URLFieldWidget::widget() {
  return m_requester;
}

#include "urlfieldwidget.moc"
