/***************************************************************************
 *
 * tdenetman-pptp.cpp - A NetworkManager frontend for TDE
 *
 * Copyright (C) 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <Helmut.Schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <tdelocale.h>
#include <tqmessagebox.h>
#include <tqbutton.h>
#include <kcombobox.h>
#include <klineedit.h>
#include <kurlrequester.h>
#include <tqobjectlist.h>
#include <tqobject.h>
#include <tqcheckbox.h>
#include <kpassdlg.h>
#include <kgenericfactory.h>
#include <tqlabel.h>

#include "tdenetman-pptp.h"

typedef KGenericFactory<PPTPPlugin> PPTPPluginFactory;
K_EXPORT_COMPONENT_FACTORY(tdenetman_pptp, PPTPPluginFactory("tdenetman_pptp"));


PPTPPlugin::PPTPPlugin(TQObject* parent, const char* name, const TQStringList& args)
	: VPNPlugin(parent, name, args)
{
	TDELocale* loc = TDEGlobal::locale();
	loc->insertCatalogue("NetworkManager-pptp");
}

PPTPPlugin::~PPTPPlugin()
{

}

VPNConfigWidget* PPTPPlugin::CreateConfigWidget(TQWidget* parent)
{
	return new PPTPConfig(parent);
}

VPNAuthenticationWidget* PPTPPlugin::CreateAuthenticationWidget(TQWidget* parent)
{
	return new PPTPAuthentication(parent);
}


PPTPConfig::PPTPConfig(TQWidget* parent)
	: VPNConfigWidget(parent)
{
	TQVBoxLayout* layout = new TQVBoxLayout(this, 1, 1);
	_pptpWidget = new PPTPConfigWidget(this);
	layout->addWidget(_pptpWidget);

	connect(_pptpWidget->chkIPAdresses, TQT_SIGNAL(toggled(bool)), _pptpWidget->routes, TQT_SLOT(setEnabled(bool)));

	this->languageChange();
}

PPTPConfig::~PPTPConfig()
{

}

void PPTPConfig::languageChange()
{

}

void PPTPConfig::setVPNData(TDENetworkSingleRouteConfigurationList& routes, TDENetworkSettingsMap& properties, TDENetworkSettingsMap& secrets) {
	m_vpnProperties = properties;
	m_vpnSecrets = secrets;

	// fill up our inputfields (only textfields atm)
	for(TQMap<TQString, TQString>::ConstIterator it = properties.begin(); it != properties.end(); ++it)
	{
		TQString entry = it.key();
		TQString value = it.data();

		if (entry == "gateway")
		{
			_pptpWidget->gateway->setText(value);
		}
		else if (entry == "refuse-eap")
		{
			_pptpWidget->chk_refuseeap->setChecked(value == "yes" || value == "true");
		}
		else if (entry == "refuse-pap")
		{
			_pptpWidget->chk_refusepap->setChecked(value == "yes" || value == "true");
		}
		else if (entry == "refuse-chap")
		{
			_pptpWidget->chk_refusechap->setChecked(value == "yes" || value == "true");
		}
		else if (entry == "refuse-mschap")
		{
			_pptpWidget->chk_refusemschap->setChecked(value == "yes" || value == "true");
		}
		else if (entry == "refuse-mschapv2")
		{
			_pptpWidget->chk_refusemschapv2->setChecked(value == "yes" || value == "true");
		}
		else if (entry == "require-mppe")
		{
			_pptpWidget->chk_requiremppe->setChecked(value == "yes" || value == "true");
		}
		else if (entry == "require-mppe-40")
		{
			_pptpWidget->chk_requiremppe40->setChecked(value == "yes" || value == "true");
		}
		else if (entry == "require-mppe-128")
		{
			_pptpWidget->chk_requiremppe128->setChecked(value == "yes" || value == "true");
		}
		else if (entry == "mppe-stateful")
		{
			_pptpWidget->chk_mppestateful->setChecked(value == "yes" || value == "true");
		}
		else if (entry == "nodeflate")
		{
			_pptpWidget->chk_nodeflate->setChecked(value == "yes" || value == "true");
		}
	}

	// set routes
	if (!routes.empty())
	{
		_pptpWidget->chkIPAdresses->setChecked(true);
		TQStringList routesText;
		for (TDENetworkSingleRouteConfigurationList::Iterator it = routes.begin(); it != routes.end(); ++it) {
			routesText.append(TQString("%1/%2").arg((*it).ipAddress.toString()).arg((*it).networkMask.toCIDRMask()));
		}
		_pptpWidget->routes->setText(routesText.join(" "));
	}
}

TDENetworkSettingsMap PPTPConfig::getVPNProperties() {
	// Build a list of properties
	m_vpnProperties.insert("gateway", TQString(_pptpWidget->gateway->text()));
	m_vpnProperties.insert("refuse-eap", TQString(_pptpWidget->chk_refuseeap->isChecked() ? "yes" : "no"));
	m_vpnProperties.insert("refuse-pap", TQString(_pptpWidget->chk_refusepap->isChecked() ? "yes" : "no"));
	m_vpnProperties.insert("refuse-chap", TQString(_pptpWidget->chk_refusechap->isChecked() ? "yes" : "no"));
	m_vpnProperties.insert("refuse-mschap", TQString(_pptpWidget->chk_refusemschap->isChecked() ? "yes" : "no"));
	m_vpnProperties.insert("refuse-mschapv2", TQString(_pptpWidget->chk_refusemschapv2->isChecked() ? "yes" : "no"));
	m_vpnProperties.insert("require-mppe", TQString(_pptpWidget->chk_requiremppe->isChecked() ? "yes" : "no"));
	m_vpnProperties.insert("require-mppe-40", TQString(_pptpWidget->chk_requiremppe40->isChecked() ? "yes" : "no"));
	m_vpnProperties.insert("require-mppe-128", TQString(_pptpWidget->chk_requiremppe128->isChecked() ? "yes" : "no"));
	m_vpnProperties.insert("mppe-stateful", TQString(_pptpWidget->chk_mppestateful->isChecked() ? "yes" : "no"));
	m_vpnProperties.insert("nodeflate", TQString(_pptpWidget->chk_nodeflate->isChecked() ? "yes" : "no"));

	// Current network-manager-pptp plugin supports bluetooth-gprs,dialup and pptp.
	// We want a pptp connection.
	//m_vpnProperties.insert("ppp-connection-type", "pptp");

	return m_vpnProperties;
}

TDENetworkSettingsMap PPTPConfig::getVPNSecrets() {
	// Build a list of secrets
	// FIXME

	return m_vpnSecrets;
}

TDENetworkSingleRouteConfigurationList PPTPConfig::getVPNRoutes()
{
	TDENetworkSingleRouteConfigurationList ret;
	TQStringList strlist;
	if(_pptpWidget->chkIPAdresses->isChecked()) {
		strlist = TQStringList::split(" ", _pptpWidget->routes->text());
	}

	for (TQStringList::Iterator it = strlist.begin(); it != strlist.end(); ++it) {
		TQStringList pieces = TQStringList::split("/", (*it));
		TDENetworkSingleRouteConfiguration routeconfig;
		routeconfig.ipAddress.setAddress(pieces[0]);
		if (pieces.count() > 1) {
			routeconfig.networkMask.fromCIDRMask(pieces[1].toUInt());
		}
		ret.append(routeconfig);
	}

	return ret;
}

bool PPTPConfig::hasChanged()
{
	return true;
}

bool PPTPConfig::isValid(TQStringList& err_msg)
{
	bool retval = true;
	if(_pptpWidget->gateway->text() == "")
	{
		err_msg.append(i18n("At least the gateway has to be supplied."));
		retval = false;
	}
	return retval;
}

PPTPAuthentication::PPTPAuthentication(TQWidget* parent, char* name)
	: VPNAuthenticationWidget(parent, name)
{
	TQVBoxLayout* layout = new TQVBoxLayout(this, 1, 1);
	_pptpAuth = new PPTPAuthenticationWidget(this);
	layout->addWidget(_pptpAuth);
}

PPTPAuthentication::~PPTPAuthentication()
{

}

TDENetworkSettingsMap PPTPAuthentication::getPasswords()
{
	// network-manager-pptp will fail hard if "CHAP" is not the
	// first element in the username&password list.
	TDENetworkSettingsMap pwds;
	//pwds.insert("CHAP", "CHAP");
	pwds.insert("user", TQString(_pptpAuth->username->text()));
	pwds.insert("password", _pptpAuth->password->password());
	pwds.insert("domain", TQString(_pptpAuth->domain->text()));
	return pwds;
}

void PPTPAuthentication::setPasswords(TDENetworkSettingsMap secrets) {
	if (secrets.contains("password")) {
		_pptpAuth->password->erase();
		_pptpAuth->password->insert(secrets["password"]);
	}
}

#include "tdenetman-pptp.moc"
