/***************************************************************************
 *
 * tdenetman-connection_setting_wireless_security_widget.cpp
 * - A NetworkManager frontend for TDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <Helmut.Schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

/* qt headers */
#include <tqwidget.h>
#include <tqlineedit.h>
#include <tqgroupbox.h>
#include <tqbuttongroup.h>
#include <tqwidgetstack.h>
#include <tqcombobox.h>
#include <tqradiobutton.h>
#include <tqlayout.h>
#include <tqcheckbox.h>

/* kde headers */
#include <kurlrequester.h>
#include <tdelocale.h>
#include <kdebug.h>
#include <kpassdlg.h>
#include <kmdcodec.h>

/* tdenetman headers */
#include "tdenetman-connection_setting_wireless_security_widget.h"
#include "sha1.h"
#include "tdenetman-wireless_manager.h"

#define WPA_PMK_LEN 32

#define SET_FLAG_IF_TRUE_CLEAR_IF_FALSE(x,y,z)		{					\
								if (z) {			\
									x |= y;			\
								}				\
								else {				\
									x &= (~y);		\
								}				\
							}

using namespace ConnectionSettings;

/*
	class WirelessSecurityWEPImpl
*/
WirelessSecurityWEPImpl::WirelessSecurityWEPImpl(TDEWiFiConnection* sec, TQWidget* parent, ConnectionSettingsDialogImpl* connsettings, const char* name, WFlags fl)
	: ConnectionSettingWirelessSecurityWEP(parent, name, fl)
	, _security_setting(sec)
	, _wepKeyType( WEPKEY_TYPE_HEX )
{
	_parentdialog = connsettings;

	cboAuthentication->insertItem(i18n("Open System"), 0);
	cboAuthentication->insertItem(i18n("Shared Key"), 1);
	if (_security_setting->securitySettings.authType == TDENetworkWiFiAuthType::Open) {
		cboAuthentication->setCurrentItem(0);
	}
	else if (_security_setting->securitySettings.authType == TDENetworkWiFiAuthType::Shared) {
		cboAuthentication->setCurrentItem(1);
	}
	else {
		// Set up defaults
		_security_setting->securitySettings.authType = TDENetworkWiFiAuthType::Open;
		cboAuthentication->setCurrentItem(0);
	}

	cbKeyType->insertItem(i18n("WEP 40/128-bit ASCII"), WEPKEY_TYPE_ASCII);
	cbKeyType->insertItem(i18n("WEP 40/128-bit Hexadecimal"), WEPKEY_TYPE_HEX);
	cbKeyType->insertItem(i18n("WEP 128-bit passphrase"), WEPKEY_TYPE_PASSPHRASE);
	cbKeyType->setCurrentItem( _wepKeyType );

	if (_security_setting->securitySettings.authType == TDENetworkWiFiAuthType::Other) {
		// Set up defaults
		_security_setting->securitySettings.wepKeyType = TDENetworkWepKeyType::Hexadecimal;
		cbKeyType->setCurrentItem(WEPKEY_TYPE_HEX);
	}

	txtWEPKey0->setText(_security_setting->securitySettings.wepKey0);
	txtWEPKey1->setText(_security_setting->securitySettings.wepKey1);
	txtWEPKey2->setText(_security_setting->securitySettings.wepKey2);
	txtWEPKey3->setText(_security_setting->securitySettings.wepKey3);

	switch(_security_setting->securitySettings.wepKeyIndex)
	{
		case 0:
			rbKeyIdx0->setChecked(true);
			break;
		case 1:
			rbKeyIdx1->setChecked(true);
			break;
		case 2:
			rbKeyIdx2->setChecked(true);
			break;
		case 3:
			rbKeyIdx3->setChecked(true);
			break;
		default:
			rbKeyIdx0->setChecked(true);
			break;
	}

	connect(cboAuthentication, TQT_SIGNAL(activated(int)), this, TQT_SLOT(slotAuthAlgChanged(int)));
	connect(cbKeyType, TQT_SIGNAL(activated(int)), this, TQT_SLOT(slotKeyTypeChanged(int)));
	connect(txtWEPKey0, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotWepKey0Changed(const TQString&)));
	connect(txtWEPKey1, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotWepKey1Changed(const TQString&)));
	connect(txtWEPKey2, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotWepKey2Changed(const TQString&)));
	connect(txtWEPKey3, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotWepKey3Changed(const TQString&)));

	connect(rbKeyIdx0, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotWepIdx0Checked(bool)));
	connect(rbKeyIdx1, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotWepIdx1Checked(bool)));
	connect(rbKeyIdx2, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotWepIdx2Checked(bool)));
	connect(rbKeyIdx3, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotWepIdx3Checked(bool)));
}

void WirelessSecurityWEPImpl::slotAuthAlgChanged(int index)
{
	if (index == 0) {
		_security_setting->securitySettings.authType = TDENetworkWiFiAuthType::Open;
	}
	else if (index == 1) {
		_security_setting->securitySettings.authType = TDENetworkWiFiAuthType::Shared;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityWEPImpl::slotKeyTypeChanged(int index)
{
	_wepKeyType = (WEPKEY_TYPE)index;

	if (index == WEPKEY_TYPE_ASCII) {
		_security_setting->securitySettings.wepKeyType = TDENetworkWepKeyType::Ascii;
	}
	else if (index == WEPKEY_TYPE_HEX) {
		_security_setting->securitySettings.wepKeyType = TDENetworkWepKeyType::Hexadecimal;
	}
	else if (index == WEPKEY_TYPE_PASSPHRASE) {
		_security_setting->securitySettings.wepKeyType = TDENetworkWepKeyType::Passphrase;
	}
	_security_setting->securitySettings.secretsValid = true;

	// update all WEP-Keys here due to the new key_type

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityWEPImpl::slotWepKey0Changed(const TQString &key)
{
	TQCString hashed = getHashedWEPKey(key, _wepKeyType);
	_security_setting->securitySettings.wepKey0 = hashed;
	_security_setting->securitySettings.secretsValid = true;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityWEPImpl::slotWepKey1Changed(const TQString &key)
{
	TQCString hashed = getHashedWEPKey(key, _wepKeyType);
	_security_setting->securitySettings.wepKey1 = hashed;
	_security_setting->securitySettings.secretsValid = true;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityWEPImpl::slotWepKey2Changed(const TQString &key)
{
	TQCString hashed = getHashedWEPKey(key, _wepKeyType);
	_security_setting->securitySettings.wepKey2 = hashed;
	_security_setting->securitySettings.secretsValid = true;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityWEPImpl::slotWepKey3Changed(const TQString& key)
{
	TQCString hashed = getHashedWEPKey(key, _wepKeyType);
	_security_setting->securitySettings.wepKey3 = hashed;
	_security_setting->securitySettings.secretsValid = true;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityWEPImpl::slotWepIdx0Checked(bool check)
{
	if (check) {
		_security_setting->securitySettings.wepKeyIndex = 0;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityWEPImpl::slotWepIdx1Checked(bool check)
{
	if (check) {
		_security_setting->securitySettings.wepKeyIndex = 1;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityWEPImpl::slotWepIdx2Checked(bool check)
{
	if (check) {
		_security_setting->securitySettings.wepKeyIndex = 2;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityWEPImpl::slotWepIdx3Checked(bool check)
{
	if (check) {
		_security_setting->securitySettings.wepKeyIndex = 3;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

TQCString
WirelessSecurityWEPImpl::getHashedWEPKey(TQString key, WEPKEY_TYPE type) const
{
	TQCString hashed;
	switch(type)
	{
		case WEPKEY_TYPE_HEX:
			return TQCString(key.latin1());
			break;
		case WEPKEY_TYPE_ASCII:
			hashed = String2Hex(TQCString(key.latin1()), key.length() * 2);
			return hashed;
			break;
		case WEPKEY_TYPE_PASSPHRASE:
			return key.utf8();
			break;
	}
	return hashed;
}

TQCString
WirelessSecurityWEPImpl::getWEP128PassphraseHash(TQCString input) const
{
	char md5_data[65];
	int input_len;
	int i;

	if (input.isNull()) return input;

	input_len = input.length();
	if (input_len < 1)
		return TQCString();

	/* Get at least 64 bytes */
	for (i = 0; i < 64; i++)
		md5_data [i] = input [i % input_len];

	/* Null terminate md5 seed data and hash it */
	md5_data[64] = 0;
	KMD5 context(md5_data);
	return (String2Hex(context.hexDigest(), 26));

}

TQCString
WirelessSecurityWEPImpl::String2Hex(TQByteArray bytes, int final_len) const
{
	TQCString result(final_len+1);
	static char hex_digits[] = "0123456789abcdef";
	result.resize(bytes.size() * 2 + 1);
	for (uint i = 0; i < bytes.size(); i++)
	{
		result[2*i] = hex_digits[(bytes[i] >> 4) & 0xf];
		result[2*i+1] = hex_digits[bytes[i] & 0xf];
	}
	/* Cut converted key off at the correct length for this cipher type */
	if (final_len > -1)
		result[final_len] = '\0';
	return result;
}


/*
	class WirelessSecurityWEPEncryptionImpl
*/
WirelessSecurityWEPEncryptionImpl::WirelessSecurityWEPEncryptionImpl(TDEWiFiConnection* security_setting, TQWidget* parent, ConnectionSettingsDialogImpl* connsettings, const char* name, WFlags fl)
	: ConnectionSettingWirelessSecurityWEPEncryption(parent, name, fl)
	, _security_setting(security_setting)
{
	_parentdialog = connsettings;

	cboEncryption->insertItem(i18n("None"));
	cboEncryption->insertItem(i18n("Dynamic WEP"));
}


/*
	class WirelessSecurityWPAVersionImpl
*/
WirelessSecurityWPAVersionImpl::WirelessSecurityWPAVersionImpl(TDEWiFiConnection* security_setting, TQWidget* parent, ConnectionSettingsDialogImpl* connsettings, const char* name, WFlags fl)
	: ConnectionSettingWirelessSecurityWPAVersion(parent, name, fl)
	, _security_setting(security_setting)
{
	_parentdialog = connsettings;

	cbWPA->setChecked(_security_setting->securitySettings.wpaVersion & TDENetworkWiFiWPAVersionFlags::WPA);
	cbRSN->setChecked(_security_setting->securitySettings.wpaVersion & TDENetworkWiFiWPAVersionFlags::RSN);

	connect(cbWPA, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotWPA1(bool)));
	connect(cbRSN, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotWPA2(bool)));

	connect(grpUseWPAVersion, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotAuto(bool)));
}

void
WirelessSecurityWPAVersionImpl::slotAuto(bool on)
{
	if (!on) {
		// auto-select proto
		_security_setting->securitySettings.wpaVersion = TDENetworkWiFiWPAVersionFlags::Any;
	}
	else {
		// use selected wpa-version
		TDENetworkWiFiWPAVersionFlags::TDENetworkWiFiWPAVersionFlags proto = TDENetworkWiFiWPAVersionFlags::None;
		if (cbWPA->isChecked()) {
			proto |= TDENetworkWiFiWPAVersionFlags::WPA;
		}
		if (cbRSN->isChecked()) {
			proto |= TDENetworkWiFiWPAVersionFlags::RSN;
		}
		_security_setting->securitySettings.wpaVersion = proto;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWPAVersionImpl::slotWPA1(bool on)
{
	SET_FLAG_IF_TRUE_CLEAR_IF_FALSE(_security_setting->securitySettings.wpaVersion, TDENetworkWiFiWPAVersionFlags::WPA, on);

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWPAVersionImpl::slotWPA2(bool on)
{
	SET_FLAG_IF_TRUE_CLEAR_IF_FALSE(_security_setting->securitySettings.wpaVersion, TDENetworkWiFiWPAVersionFlags::RSN, on);

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

/*
	class WirelessSecurityWPACipherImpl
*/
WirelessSecurityWPACipherImpl::WirelessSecurityWPACipherImpl(TDEWiFiConnection* security_setting, TQWidget* parent, ConnectionSettingsDialogImpl* connsettings, const char* name, WFlags fl)
	: ConnectionSettingWirelessSecurityWPACipher(parent, name, fl)
	, _security_setting(security_setting)
{
	_parentdialog = connsettings;

	connect(grpUseCipher, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotCipherChangedAuto(bool)));

	connect(chkGroupCipherTKIP, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotGroupCipherChangedTKIP(bool)));
	connect(chkGroupCipherCCMP, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotGroupCipherChangedCCMP(bool)));
	connect(chkGroupCipherWEP40, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotGroupCipherChangedWEP40(bool)));
	connect(chkGroupCipherWEP104, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotGroupCipherChangedWEP104(bool)));

	connect(chkPairwiseCipherTKIP, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotPairwiseCipherChangedTKIP(bool)));
	connect(chkPairwiseCipherCCMP, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotPairwiseCipherChangedCCMP(bool)));

	chkPairwiseCipherCCMP->setChecked(_security_setting->securitySettings.allowedPairWiseCiphers.contains(TDENetworkWiFiConnectionCipher::Any) || _security_setting->securitySettings.allowedPairWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherCCMP));
	chkPairwiseCipherTKIP->setChecked(_security_setting->securitySettings.allowedPairWiseCiphers.contains(TDENetworkWiFiConnectionCipher::Any) || _security_setting->securitySettings.allowedPairWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherTKIP));

	chkGroupCipherCCMP->setChecked(_security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::Any) || _security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherCCMP));
	chkGroupCipherTKIP->setChecked(_security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::Any) || _security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherTKIP));
	chkGroupCipherWEP40->setChecked(_security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::Any) || _security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherWEP40));
	chkGroupCipherWEP104->setChecked(_security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::Any) || _security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherWEP104));
}

void
WirelessSecurityWPACipherImpl::slotCipherChangedAuto(bool checked)
{
	if (!checked)
	{
		// select auto for both ciphers
		_security_setting->securitySettings.allowedGroupWiseCiphers.clear();
		_security_setting->securitySettings.allowedGroupWiseCiphers.append(TDENetworkWiFiConnectionCipher::Any);
		_security_setting->securitySettings.allowedPairWiseCiphers.clear();
		_security_setting->securitySettings.allowedPairWiseCiphers.append(TDENetworkWiFiConnectionCipher::Any);
	}
	else
	{
		// use the already selected ciphers

		// group cipher
		TDENetworkWiFiConnectionCipherList cipher;
		if (chkGroupCipherTKIP->isChecked()) {
			cipher.append(TDENetworkWiFiConnectionCipher::CipherTKIP);
		}
		if (chkGroupCipherCCMP->isChecked()) {
			cipher.append(TDENetworkWiFiConnectionCipher::CipherCCMP);
		}
		if (chkGroupCipherWEP40->isChecked()) {
			cipher.append(TDENetworkWiFiConnectionCipher::CipherWEP40);
		}
		if (chkGroupCipherWEP104->isChecked()) {
			cipher.append(TDENetworkWiFiConnectionCipher::CipherWEP104);
		}
		_security_setting->securitySettings.allowedGroupWiseCiphers = cipher;

		// pairwise cipher
		cipher.clear();
		if (chkPairwiseCipherTKIP->isChecked()) {
			cipher.append(TDENetworkWiFiConnectionCipher::CipherTKIP);
		}
		if (chkPairwiseCipherCCMP->isChecked()) {
			cipher.append(TDENetworkWiFiConnectionCipher::CipherCCMP);
		}
		_security_setting->securitySettings.allowedPairWiseCiphers = cipher;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWPACipherImpl::slotGroupCipherChangedTKIP(bool checked)
{
	if (checked) {
		if (!_security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherTKIP)) _security_setting->securitySettings.allowedGroupWiseCiphers.append(TDENetworkWiFiConnectionCipher::CipherTKIP);
	}
	else {
		_security_setting->securitySettings.allowedGroupWiseCiphers.remove(TDENetworkWiFiConnectionCipher::CipherTKIP);
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWPACipherImpl::slotGroupCipherChangedCCMP(bool checked)
{
	if (checked) {
		if (!_security_setting->securitySettings.allowedGroupWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherCCMP)) _security_setting->securitySettings.allowedGroupWiseCiphers.append(TDENetworkWiFiConnectionCipher::CipherCCMP);
	}
	else {
		_security_setting->securitySettings.allowedGroupWiseCiphers.remove(TDENetworkWiFiConnectionCipher::CipherCCMP);
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWPACipherImpl::slotGroupCipherChangedWEP40(bool checked)
{
	if (checked) {
		if (!_security_setting->securitySettings.allowedPairWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherWEP40)) _security_setting->securitySettings.allowedPairWiseCiphers.append(TDENetworkWiFiConnectionCipher::CipherWEP40);
	}
	else {
		_security_setting->securitySettings.allowedPairWiseCiphers.remove(TDENetworkWiFiConnectionCipher::CipherWEP40);
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWPACipherImpl::slotGroupCipherChangedWEP104(bool checked)
{
	if (checked) {
		if (!_security_setting->securitySettings.allowedPairWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherWEP104)) _security_setting->securitySettings.allowedPairWiseCiphers.append(TDENetworkWiFiConnectionCipher::CipherWEP104);
	}
	else {
		_security_setting->securitySettings.allowedPairWiseCiphers.remove(TDENetworkWiFiConnectionCipher::CipherWEP104);
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWPACipherImpl::slotPairwiseCipherChangedTKIP(bool checked)
{
	if (checked) {
		if (!_security_setting->securitySettings.allowedPairWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherTKIP)) _security_setting->securitySettings.allowedPairWiseCiphers.append(TDENetworkWiFiConnectionCipher::CipherTKIP);
	}
	else {
		_security_setting->securitySettings.allowedPairWiseCiphers.remove(TDENetworkWiFiConnectionCipher::CipherTKIP);
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWPACipherImpl::slotPairwiseCipherChangedCCMP(bool checked)
{
	if (checked) {
		if (!_security_setting->securitySettings.allowedPairWiseCiphers.contains(TDENetworkWiFiConnectionCipher::CipherCCMP)) _security_setting->securitySettings.allowedPairWiseCiphers.append(TDENetworkWiFiConnectionCipher::CipherCCMP);
	}
	else {
		_security_setting->securitySettings.allowedPairWiseCiphers.remove(TDENetworkWiFiConnectionCipher::CipherCCMP);
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

/*
	class WirelessSecurityWPAPSK
*/
WirelessSecurityWPAPSKImpl::WirelessSecurityWPAPSKImpl(TDEWiFiConnection* security_setting, TDEWiFiConnection* wireless_setting, TQWidget* parent, ConnectionSettingsDialogImpl* connsettings, const char* name, WFlags fl)
	: ConnectionSettingWirelessSecurityWPAPSK(parent, name, fl)
	, _security_setting(security_setting)
	, _wireless_setting(wireless_setting)
{
	_parentdialog = connsettings;

	if (_security_setting->securitySettings.secretsValid) {
		txtPSK->setText(_security_setting->securitySettings.psk);
	}
	connect(txtPSK, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotPSKChanged(const TQString&)));
}

void
WirelessSecurityWPAPSKImpl::slotPSKChanged(const TQString& psk)
{
	if (_wireless_setting)
	{
		_security_setting->securitySettings.psk = psk;
		_security_setting->securitySettings.secretsValid = true;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

TQCString
WirelessSecurityWPAPSKImpl::String2Hex(TQByteArray bytes, int final_len) const
{
	TQCString result(final_len+1);
	static char hex_digits[] = "0123456789abcdef";
	result.resize(bytes.size() * 2 + 1);
	for (uint i = 0; i < bytes.size(); i++)
	{
		result[2*i] = hex_digits[(bytes[i] >> 4) & 0xf];
		result[2*i+1] = hex_digits[bytes[i] & 0xf];
	}
	/* Cut converted key off at the correct length for this cipher type */
	if (final_len > -1)
		result[final_len] = '\0';
	return result;
}

/*
	class WirelessSecurityEAPImpl
*/
WirelessSecurityEAPImpl::WirelessSecurityEAPImpl(TDEWiFiConnection* security_setting, WirelessSecurityPhase2Impl* phase2_widget, TQWidget* parent, ConnectionSettingsDialogImpl* connsettings, const char* name, WFlags fl)
	: ConnectionSettingWirelessSecurityEAP(parent, name, fl)
	, _security_setting(security_setting)
	, _phase2_widget(phase2_widget)
{
	_parentdialog = connsettings;

	// insert all EAP-Methods
	int index = 0;
	cboMethod->insertItem(i18n("None"), index);
	_eapIndexMap[index] = TDENetworkIEEE8021xType::None;

	cboMethod->insertItem(i18n("TTLS"), ++index);
	_eapIndexMap[index] = TDENetworkIEEE8021xType::TTLS;

	cboMethod->insertItem(i18n("PEAP"), ++index);
	_eapIndexMap[index] = TDENetworkIEEE8021xType::PEAP;

	cboMethod->insertItem(i18n("TLS"), ++index);
	_eapIndexMap[index] = TDENetworkIEEE8021xType::TLS;

	cboMethod->insertItem(i18n("Leap"), ++index);
	_eapIndexMap[index] = TDENetworkIEEE8021xType::LEAP;

	cboMethod->insertItem(i18n("MD5"), ++index);
	_eapIndexMap[index] = TDENetworkIEEE8021xType::MD5;

	cboMethod->insertItem(i18n("FAST"), ++index);
	_eapIndexMap[index] = TDENetworkIEEE8021xType::Fast;

	cboMethod->insertItem(i18n("SIM"), ++index);
	_eapIndexMap[index] = TDENetworkIEEE8021xType::SIM;

	// preselect the correct method
	TQBiDirectionalMap<int, TDENetworkIEEE8021xType::TDENetworkIEEE8021xType>::Iterator it = _eapIndexMap.findData(_security_setting->eapConfig.type);
	if (it != _eapIndexMap.end()) {
		cboMethod->setCurrentItem(it.key());
	}

	// update phase2 combobox
	_phase2_widget->setAllowedPhase2Methods(_security_setting->eapConfig.allowedPhase2EAPMethods);

	txtIdentity->setText(_security_setting->eapConfig.userName);
	txtAnonIdentity->setText(_security_setting->eapConfig.anonymousUserName);
	if (_security_setting->eapConfig.secretsValid) {
		txtPassword->setText(_security_setting->eapConfig.password);
	}

	chkCAStore->setChecked(_security_setting->eapConfig.forceSystemCaCertificates);
	kURLCACert->setEnabled(!_security_setting->eapConfig.forceSystemCaCertificates);

	// get notified if the method changes
	connect(cboMethod, TQT_SIGNAL(activated(int)), this, TQT_SLOT(slotMethodChanged(int)));
	connect(txtIdentity, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotIdentityChanged(const TQString&)));
	connect(txtAnonIdentity, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotAnonIdentityChanged(const TQString&)));
	connect(txtPassword, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotPasswordChanged(const TQString&)));
	connect(chkCAStore, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotUseSystemCaCertChanged(bool)));
}

void WirelessSecurityEAPImpl::slotUseSystemCaCertChanged(bool on)
{
	_security_setting->eapConfig.forceSystemCaCertificates = on;
	kURLCACert->setEnabled(!on);

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityEAPImpl::slotMethodChanged(int index)
{
	// new method choosen
	TDENetworkIEEE8021xType::TDENetworkIEEE8021xType eap = _eapIndexMap[index];
	_security_setting->eapConfig.type = eap;

	_phase2_widget->setAllowedPhase2Methods(_security_setting->eapConfig.allowedPhase2EAPMethods);

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityEAPImpl::slotIdentityChanged(const TQString& identity)
{
	_security_setting->eapConfig.userName = identity;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityEAPImpl::slotAnonIdentityChanged(const TQString& identity)
{
	_security_setting->eapConfig.anonymousUserName = identity;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void WirelessSecurityEAPImpl::slotPasswordChanged(const TQString& pwd)
{
	_security_setting->eapConfig.password = txtPassword->password();
	_security_setting->eapConfig.secretsValid = true;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

/*
	class WirelessSecurityPhase2Impl
*/
WirelessSecurityPhase2Impl::WirelessSecurityPhase2Impl(TDEWiFiConnection* security_setting, TQWidget* parent, ConnectionSettingsDialogImpl* connsettings, const char* name, WFlags fl)
	: ConnectionSettingWirelessSecurityPhase2(parent, name, fl)
	, _security_setting(security_setting)
{
	_parentdialog = connsettings;

	_allowed_methods.append(TDENetworkIEEE8021xType::None);
	updateMethodComboBox();

	connect(cboPhase2Method, TQT_SIGNAL(activated(int)), this, TQT_SLOT(slotPhase2MethodChanged(int)));
}

void WirelessSecurityPhase2Impl::updateMethodComboBox()
{
	// insert all phase2 EAP-Methods
	int index = 0;
	cboPhase2Method->clear();
	_eapIndexMap.clear();

	for (TQValueList<TDENetworkIEEE8021xType::TDENetworkIEEE8021xType>::Iterator it = _allowed_methods.begin(); it != _allowed_methods.end(); ++it)
	{
		if ((*it) == TDENetworkIEEE8021xType::None)
		{
			cboPhase2Method->insertItem(i18n("None"), index);
			_eapIndexMap[index] = TDENetworkIEEE8021xType::None;
			index++;
		}
		else if ((*it) == TDENetworkIEEE8021xType::MSCHAPV2)
		{
			cboPhase2Method->insertItem(i18n("MSCHAPv2"), index);
			_eapIndexMap[index] = TDENetworkIEEE8021xType::MSCHAPV2;
			index++;
		}
		else if ((*it) == TDENetworkIEEE8021xType::PAP)
		{	
			cboPhase2Method->insertItem(i18n("PAP"), index);
			_eapIndexMap[index] = TDENetworkIEEE8021xType::PAP;
			index++;
		}
		else if ((*it) == TDENetworkIEEE8021xType::CHAP)
		{
			cboPhase2Method->insertItem(i18n("CHAP"), index);
			_eapIndexMap[index] = TDENetworkIEEE8021xType::CHAP;
			index++;
		}
		else if ((*it) == TDENetworkIEEE8021xType::MSCHAP)
		{
			cboPhase2Method->insertItem(i18n("MSCHAP"), index);
			_eapIndexMap[index] = TDENetworkIEEE8021xType::MSCHAP;
			index++;
		}
		else if ((*it) == TDENetworkIEEE8021xType::GTC)
		{
			cboPhase2Method->insertItem(i18n("GTC"), index);
			_eapIndexMap[index] = TDENetworkIEEE8021xType::GTC;
			index++;
		}
		else if ((*it) == TDENetworkIEEE8021xType::OTP)
		{
			cboPhase2Method->insertItem(i18n("OTP"), index);
			_eapIndexMap[index] = TDENetworkIEEE8021xType::OTP;
			index++;
		}
		else if ((*it) == TDENetworkIEEE8021xType::MD5)
		{
			cboPhase2Method->insertItem(i18n("MD5"), index);
			_eapIndexMap[index] = TDENetworkIEEE8021xType::MD5;
			index++;
		}
		else if ((*it) == TDENetworkIEEE8021xType::TLS)
		{
			cboPhase2Method->insertItem(i18n("TLS"), index);
			_eapIndexMap[index] = TDENetworkIEEE8021xType::TLS;
			index++;
		}
	}

	// preselect the correct method
	TQBiDirectionalMap<int, TDENetworkIEEE8021xType::TDENetworkIEEE8021xType>::Iterator it = _eapIndexMap.findData(_security_setting->eapConfig.phase2EAPAuthMethod);
	if (it != _eapIndexMap.end())
	{
		cboPhase2Method->setCurrentItem(it.key());
		_security_setting->eapConfig.phase2EAPAuthMethod = it.data();
	}
	else
	{
		cboPhase2Method->setCurrentItem(0);
		_security_setting->eapConfig.phase2EAPAuthMethod = _eapIndexMap[0];
	}
}

void WirelessSecurityPhase2Impl::setAllowedPhase2Methods(const TQValueList<TDENetworkIEEE8021xType::TDENetworkIEEE8021xType>& list)
{
	_allowed_methods = list;
	updateMethodComboBox();
}

void WirelessSecurityPhase2Impl::slotPhase2MethodChanged(int index)
{
	// new method choosen
	TDENetworkIEEE8021xType::TDENetworkIEEE8021xType eap = _eapIndexMap[index];
	_security_setting->eapConfig.phase2EAPAuthMethod = eap;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

/*
	class WirelessSecurityWidgetImpl
*/

WirelessSecurityWidgetImpl::WirelessSecurityWidgetImpl(TDENetworkConnection* conn, bool new_conn, TQWidget* parent, const char* name, WFlags fl)
	: WidgetInterface(parent, name, fl)
{
	_parentdialog = dynamic_cast<ConnectionSettingsDialogImpl*>(parent);
	_security_setting = dynamic_cast<TDEWiFiConnection*> (conn);
	_wireless_setting = dynamic_cast<TDEWiFiConnection*> (conn);
	_ieee8021x_setting = dynamic_cast<TDEWiFiConnection*> (conn);
	_new_conn = new_conn;

	TQVBoxLayout* layout = new TQVBoxLayout(this, 1, 1);
	_mainWid = new ConnectionSettingWirelessSecurityWidget(this);
	layout->addWidget(_mainWid);

	TQTimer::singleShot(0, this, TQT_SLOT(slotInit()));	
}

void
WirelessSecurityWidgetImpl::slotInit()
{
	// create all security widgets...
	TQWidget* wep = new WirelessSecurityWEPImpl(_security_setting, _mainWid->groupUseEncryption, _parentdialog);
	TQWidget* phase2 = new WirelessSecurityPhase2Impl(_ieee8021x_setting, _mainWid->groupUseEncryption, _parentdialog);
	TQWidget* eap = new WirelessSecurityEAPImpl(_ieee8021x_setting, (WirelessSecurityPhase2Impl*)phase2, _mainWid->groupUseEncryption, _parentdialog);
	TQWidget* wpaversion = new WirelessSecurityWPAVersionImpl(_security_setting, _mainWid->groupUseEncryption, _parentdialog);
	TQWidget* wpacipher = new WirelessSecurityWPACipherImpl(_security_setting, _mainWid->groupUseEncryption, _parentdialog);
	TQWidget* wpapsk = new WirelessSecurityWPAPSKImpl(_security_setting, _wireless_setting, _mainWid->groupUseEncryption, _parentdialog);
	TQWidget* wepencryption = new WirelessSecurityWEPEncryptionImpl(_security_setting, _mainWid->groupUseEncryption, _parentdialog);

	wep->setHidden(true);
	eap->setHidden(true);
	wpaversion->setHidden(true);
	wpacipher->setHidden(true);
	phase2->setHidden(true);
	wpapsk->setHidden(true);
	wepencryption->setHidden(true);

	_widgets[SECURITY_WEP].clear();
	_widgets[SECURITY_WPA_PSK].clear();
	_widgets[SECURITY_WPA_EAP].clear();
	_widgets[SECURITY_IEEE8021X].clear();

	// create WEP widget list
	_widgets[SECURITY_WEP].append(wep);

	// create WPA PSK widget list
	_extra_widgets[SECURITY_WPA_PSK].append(wpaversion);
	_extra_widgets[SECURITY_WPA_PSK].append(wpacipher);
	_widgets[SECURITY_WPA_PSK].append(wpapsk);

	// create WPA EAP widget list
	_extra_widgets[SECURITY_WPA_EAP].append(wpaversion);
	_extra_widgets[SECURITY_WPA_EAP].append(wpacipher);
	_widgets[SECURITY_WPA_EAP].append(eap);
	_widgets[SECURITY_WPA_EAP].append(phase2);

	// create IEEE8021X widget list
	_widgets[SECURITY_IEEE8021X].append(wepencryption);
	_widgets[SECURITY_IEEE8021X].append(eap);

	connect(_mainWid->cboSecurity, TQT_SIGNAL(activated(int)), this, TQT_SLOT(slotComboSecurityActivated(int)));
	connect(_mainWid->groupUseEncryption, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotUseEncryptionToggled(bool)));
	connect(_mainWid->pbExtra, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotExtraSettingsToggled(bool)));

}

void
WirelessSecurityWidgetImpl::Activate()
{
	kdDebug() << "Activate " << (_new_conn ? "New" : "Edit") << endl;
	comboSecurityInit();
}

void
WirelessSecurityWidgetImpl::comboSecurityInit()
{
	int index = 0;

	TQValueList<SecurityMethods> allowed_methods;
	_mainWid->cboSecurity->clear();

	// TODO: Preselect the right security method
	// We should have an Essid already, fetch all possible APs
	TQValueList<TDENetworkWiFiAPInfo*> aps;
	if (_new_conn && !_wireless_setting->SSID.isEmpty()) {
		aps = WirelessManager::getAccessPointsForEssid(_wireless_setting->SSID);
#if 0
		kdDebug() << "Hugo " << TQString(TQCString(_wireless_setting->getEssid())).ascii() << endl;
#endif
	}
	if (!aps.isEmpty()) {
		// if at least one AP has this security setting show the entry in the combobox
		for (TQValueList<TDENetworkWiFiAPInfo*>::Iterator it = aps.begin(); it != aps.end(); ++it) {
			if (((*it)->wpaFlags & TDENetworkWiFiAPFlags::PrivacySupport) || ((*it)->rsnFlags & TDENetworkWiFiAPFlags::PrivacySupport)) {
				kdDebug() << "AP " << (*it)->friendlySSID().ascii() << " is encrypted" << endl;
				if ((((*it)->rsnFlags & TDENetworkWiFiAPFlags::EncryptionFlagsMask) != TDENetworkWiFiAPFlags::None) || (((*it)->wpaFlags & TDENetworkWiFiAPFlags::EncryptionFlagsMask) != TDENetworkWiFiAPFlags::None)) {
					// WPA or RSN
					if ((*it)->rsnFlags & TDENetworkWiFiAPFlags::KeyManagementPSK || (*it)->wpaFlags & TDENetworkWiFiAPFlags::KeyManagementPSK) {
						if (!allowed_methods.contains(SECURITY_WPA_PSK)) {
							allowed_methods.append(SECURITY_WPA_PSK);
						}
					}
			
					if ((*it)->rsnFlags & TDENetworkWiFiAPFlags::KeyManagement80211 || (*it)->wpaFlags & TDENetworkWiFiAPFlags::KeyManagement80211) {
						if (!allowed_methods.contains(SECURITY_WPA_EAP)) {
							allowed_methods.append(SECURITY_WPA_EAP);
						}
					}
				}

				// No WPA & RSN => WEP or dynamic WEP with 802.1x authentication
				// TODO: an AP can provide WEP in addition to WPA
				if (!allowed_methods.contains(SECURITY_WEP)) {
					allowed_methods.append(SECURITY_WEP);
				}
				if (!allowed_methods.contains(SECURITY_IEEE8021X)) {
					allowed_methods.append(SECURITY_IEEE8021X);
				}
			}
		}

		// insert only allowed security methods
		if (allowed_methods.contains(SECURITY_WPA_PSK))
		{
			_securityComboMap.insert(index, SECURITY_WPA_PSK);
			_mainWid->cboSecurity->insertItem(i18n("WPA Personal"), index++);
		}

		if (allowed_methods.contains(SECURITY_WPA_EAP))
		{
			_securityComboMap.insert(index, SECURITY_WPA_EAP);
			_mainWid->cboSecurity->insertItem(i18n("WPA Enterprise"), index++);
		}

		if (allowed_methods.contains(SECURITY_WEP))
		{
			_securityComboMap.insert(index, SECURITY_WEP);
			_mainWid->cboSecurity->insertItem(i18n("WEP"), index++);
		}

		if (allowed_methods.contains(SECURITY_IEEE8021X))
		{
			_securityComboMap.insert(index, SECURITY_IEEE8021X);
			_mainWid->cboSecurity->insertItem(i18n("IEEE 802.1X"), index++);
		}
	}
	else
	{
		// insert all possible authentication methods
		_mainWid->cboSecurity->insertItem(i18n("WEP"),SECURITY_WEP );
		_mainWid->cboSecurity->insertItem(i18n("WPA Personal"), SECURITY_WPA_PSK);
		_mainWid->cboSecurity->insertItem(i18n("WPA Enterprise"), SECURITY_WPA_EAP);
		_mainWid->cboSecurity->insertItem(i18n("IEEE 802.1X"), SECURITY_IEEE8021X);
		_securityComboMap.insert(SECURITY_WEP, SECURITY_WEP);
		_securityComboMap.insert(SECURITY_WPA_PSK, SECURITY_WPA_PSK);
		_securityComboMap.insert(SECURITY_WPA_EAP, SECURITY_WPA_EAP);
		_securityComboMap.insert(SECURITY_IEEE8021X, SECURITY_IEEE8021X);
	}

	if (!_new_conn)
	{
		switch(_security_setting->securitySettings.keyType)
		{
			case TDENetworkWiFiKeyType::WEP:
				if (_security_setting->securitySettings.authType == TDENetworkWiFiAuthType::Shared ||
				    !_security_setting->securitySettings.wepKey0.isEmpty() ||
				    !_security_setting->securitySettings.wepKey1.isEmpty() ||
				    !_security_setting->securitySettings.wepKey2.isEmpty() ||
				    !_security_setting->securitySettings.wepKey3.isEmpty() )
				{
					_mainWid->groupUseEncryption->setChecked(true);
					_mainWid->cboSecurity->setCurrentItem(SECURITY_WEP);
					slotComboSecurityActivated(_securityComboMap[SECURITY_WEP]);
				}
				else
					_mainWid->groupUseEncryption->setChecked(false);
				break;
			case TDENetworkWiFiKeyType::WPAInfrastructure:
				_mainWid->groupUseEncryption->setChecked(true);
				_mainWid->cboSecurity->setCurrentItem(SECURITY_WPA_PSK);
				slotComboSecurityActivated(_securityComboMap[SECURITY_WPA_PSK]);
				break;
			case TDENetworkWiFiKeyType::WPAEnterprise:
				_mainWid->groupUseEncryption->setChecked(true);
				_mainWid->cboSecurity->setCurrentItem(SECURITY_WPA_EAP);
				slotComboSecurityActivated(_securityComboMap[SECURITY_WPA_EAP]);
				break;
			case TDENetworkWiFiKeyType::DynamicWEP:
				_mainWid->groupUseEncryption->setChecked(true);
				_mainWid->cboSecurity->setCurrentItem(SECURITY_IEEE8021X);
				slotComboSecurityActivated(_securityComboMap[SECURITY_IEEE8021X]);
				break;

			default:
				break;
		}
	}
	else
	{
		// select first possible security method
		if (_mainWid->cboSecurity->count() > 0)
		{
			_mainWid->groupUseEncryption->setChecked(true);
			_mainWid->groupUseEncryption->setEnabled(true);
			_mainWid->cboSecurity->setCurrentItem(0);
			slotComboSecurityActivated(0);
		}
		else
		{
			_mainWid->groupUseEncryption->setChecked(false);
			_mainWid->groupUseEncryption->setEnabled(false);
		}
	}
}

void
WirelessSecurityWidgetImpl::slotUseEncryptionToggled(bool on)
{
	_wireless_setting->securityRequired = on;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWidgetImpl::slotComboSecurityActivated(int index)
{
	int i = _securityComboMap[index];

	// authentication switched, we have to show the appropriate widgets and hide some others
	switch(i)
	{
		case SECURITY_WEP:
			configureForWEP();
			break;

		case SECURITY_WPA_PSK:
			configureForWPAPSK();
			break;

		case SECURITY_WPA_EAP:
			configureForWPAEAP();
			break;

		case SECURITY_IEEE8021X:
			configureForIEEE8021X();
			break;

		default:
			// should not happen, something is broken...
			break;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWidgetImpl::configureWidgets(SecurityMethods method)
{
	// store selected method
	_currentMethod = method;

	for (int i = 0; i < SECURITY_COUNT; ++i)
	{
		// remove all current widgets that do not belong to the selected method
		if (i != method)
		{
			for (TQValueList<TQWidget*>::iterator it = _widgets[i].begin(); it != _widgets[i].end(); ++it)
			{
				_mainWid->groupUseEncryption->layout()->remove(*it);
				(*it)->hide();
			}
			// remove extra widgets too
			for (TQValueList<TQWidget*>::iterator it = _extra_widgets[i].begin(); it != _extra_widgets[i].end(); ++it)
			{
				_mainWid->groupUseEncryption->layout()->remove(*it);
				(*it)->hide();
			}
		}
	}

	// show all widgets widgets for the selected security method
	for (TQValueList<TQWidget*>::iterator it = _widgets[method].begin(); it != _widgets[method].end(); ++it)
	{
		_mainWid->groupUseEncryption->layout()->add(*it);
		(*it)->show();
	}

	if (_mainWid->pbExtra->isOn())
		for (TQValueList<TQWidget*>::iterator it = _extra_widgets[method].begin(); it != _extra_widgets[method].end(); ++it)
		{
			_mainWid->groupUseEncryption->layout()->add(*it);
			(*it)->show();
		}

	// deactivate button if no extra settings are available
	_mainWid->pbExtra->setEnabled(!(_extra_widgets[method].begin() == _extra_widgets[method].end()));
}

void
WirelessSecurityWidgetImpl::slotExtraSettingsToggled(bool on)
{
	if (on)
		for (TQValueList<TQWidget*>::iterator it = _extra_widgets[_currentMethod].begin(); it != _extra_widgets[_currentMethod].end(); ++it)
		{
			_mainWid->groupUseEncryption->layout()->add(*it);
			(*it)->show();
		}
	else
		for (TQValueList<TQWidget*>::iterator it = _extra_widgets[_currentMethod].begin(); it != _extra_widgets[_currentMethod].end(); ++it)
		{
			_mainWid->groupUseEncryption->layout()->remove(*it);
			(*it)->hide();
		}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
WirelessSecurityWidgetImpl::configureForWEP()
{
	_security_setting->securitySettings.keyType = TDENetworkWiFiKeyType::WEP;
	configureWidgets(SECURITY_WEP);
}

void
WirelessSecurityWidgetImpl::configureForWPAPSK()
{
	_security_setting->securitySettings.keyType = TDENetworkWiFiKeyType::WPAInfrastructure;
	configureWidgets(SECURITY_WPA_PSK);
}

void
WirelessSecurityWidgetImpl::configureForWPAEAP()
{
	_security_setting->securitySettings.keyType = TDENetworkWiFiKeyType::WPAEnterprise;
	configureWidgets(SECURITY_WPA_EAP);
}

void
WirelessSecurityWidgetImpl::configureForIEEE8021X()
{
	_security_setting->securitySettings.keyType = TDENetworkWiFiKeyType::DynamicWEP;
	configureWidgets(SECURITY_IEEE8021X);
}




#include "tdenetman-connection_setting_wireless_security_widget.moc"
