/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2006-10-15
 * Description : IPTC keywords settings page.
 *
 * Copyright (C) 2006-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// QT includes.

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqvalidator.h>
#include <tqcheckbox.h>
#include <tqpushbutton.h>

// KDE includes.

#include <tdelocale.h>
#include <kdialog.h>
#include <tdelistbox.h>
#include <klineedit.h>
#include <kactivelabel.h>
#include <kiconloader.h>

// LibKExiv2 includes. 

#include <libkexiv2/kexiv2.h>

// Local includes.

#include "iptckeywords.h"
#include "iptckeywords.moc"

namespace KIPIMetadataEditPlugin
{

class IPTCKeywordsPriv
{
public:

    IPTCKeywordsPriv()
    {
        addKeywordButton = 0;
        delKeywordButton = 0;
        keywordsBox      = 0;
        keywordsCheck    = 0;
        keywordEdit      = 0;
    }

    TQStringList  oldKeywords;

    TQPushButton *addKeywordButton;
    TQPushButton *delKeywordButton;

    TQCheckBox   *keywordsCheck;

    KLineEdit   *keywordEdit;

    TDEListBox    *keywordsBox;
};

IPTCKeywords::IPTCKeywords(TQWidget* parent)
            : TQWidget(parent)
{
    d = new IPTCKeywordsPriv;
    TQGridLayout *grid = new TQGridLayout(parent, 5, 2, 0, KDialog::spacingHint());
    grid->setAlignment( TQt::AlignTop );

    // IPTC only accept printable Ascii char.
    TQRegExp asciiRx("[\x20-\x7F]+$");
    TQValidator *asciiValidator = new TQRegExpValidator(asciiRx, TQT_TQOBJECT(this));

    // --------------------------------------------------------

    d->keywordsCheck = new TQCheckBox(i18n("Use information retrieval words:"), parent);

    d->keywordEdit   = new KLineEdit(parent);
    d->keywordEdit->setValidator(asciiValidator);
    d->keywordEdit->setMaxLength(64);
    TQWhatsThis::add(d->keywordEdit, i18n("<p>Enter here a new keyword. "
                    "This field is limited to 64 ASCII characters."));

    d->keywordsBox   = new TDEListBox(parent);
    d->keywordsBox->setVScrollBarMode(TQScrollView::AlwaysOn);
    
    d->addKeywordButton = new TQPushButton( i18n("&Add"), parent);
    d->delKeywordButton = new TQPushButton( i18n("&Delete"), parent);
    d->addKeywordButton->setIconSet(SmallIcon("add"));
    d->delKeywordButton->setIconSet(SmallIcon("remove"));
    d->delKeywordButton->setEnabled(false);

    grid->addMultiCellWidget(d->keywordsCheck, 0, 0, 0, 1);
    grid->addMultiCellWidget(d->keywordEdit, 1, 1, 0, 0);
    grid->addMultiCellWidget(d->keywordsBox, 2, 5, 0, 0);
    grid->addMultiCellWidget(d->addKeywordButton, 2, 2, 1, 1);
    grid->addMultiCellWidget(d->delKeywordButton, 3, 3, 1, 1);

    // --------------------------------------------------------

    KActiveLabel *note = new KActiveLabel(i18n("<b>Note: "
                 "<b><a href='http://en.wikipedia.org/wiki/IPTC'>IPTC</a></b> "
                 "text tags only support the printable "
                 "<b><a href='http://en.wikipedia.org/wiki/Ascii'>ASCII</a></b> "
                 "characters set and limit strings size. "
                 "Use contextual help for details.</b>"), parent);
    note->setMaximumWidth(150);

    grid->addMultiCellWidget(note, 4, 4, 1, 1);
    grid->setColStretch(0, 10);                     
    grid->setRowStretch(5, 10);      
                                         
    // --------------------------------------------------------

    connect(d->keywordsBox, TQT_SIGNAL(selectionChanged()),
            this, TQT_SLOT(slotKeywordSelectionChanged()));
    
    connect(d->addKeywordButton, TQT_SIGNAL(clicked()),
            this, TQT_SLOT(slotAddKeyword()));
    
    connect(d->delKeywordButton, TQT_SIGNAL(clicked()),
            this, TQT_SLOT(slotDelKeyword()));

    // --------------------------------------------------------

    connect(d->keywordsCheck, TQT_SIGNAL(toggled(bool)),
            d->keywordEdit, TQT_SLOT(setEnabled(bool)));

    connect(d->keywordsCheck, TQT_SIGNAL(toggled(bool)),
            d->keywordsBox, TQT_SLOT(setEnabled(bool)));

    connect(d->keywordsCheck, TQT_SIGNAL(toggled(bool)),
            d->addKeywordButton, TQT_SLOT(setEnabled(bool)));

    connect(d->keywordsCheck, TQT_SIGNAL(toggled(bool)),
            d->delKeywordButton, TQT_SLOT(setEnabled(bool)));

    // --------------------------------------------------------

    connect(d->keywordsCheck, TQT_SIGNAL(toggled(bool)),
            this, TQT_SIGNAL(signalModified()));

    connect(d->addKeywordButton, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalModified()));
    
    connect(d->delKeywordButton, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalModified()));
}

IPTCKeywords::~IPTCKeywords()
{
    delete d;
}

void IPTCKeywords::slotDelKeyword()
{
    int index = d->keywordsBox->currentItem();
    if (index == -1)
        return;

    TQListBoxItem* item = d->keywordsBox->item(index);
    if (!item) return;
    delete item;
}

void IPTCKeywords::slotKeywordSelectionChanged()
{
    if (d->keywordsBox->currentItem() != -1)
        d->delKeywordButton->setEnabled(true);
    else
        d->delKeywordButton->setEnabled(false);
}

void IPTCKeywords::slotAddKeyword()
{
    TQString newKeyword = d->keywordEdit->text();
    if (newKeyword.isEmpty()) return;

    bool found = false;
    for (TQListBoxItem *item = d->keywordsBox->firstItem();
         item; item = item->next()) 
    {
        if (newKeyword == item->text()) 
        {
            found = true;
            break;
        }
    }

    if (!found)
        d->keywordsBox->insertItem(newKeyword);
}

void IPTCKeywords::readMetadata(TQByteArray& iptcData)
{
    blockSignals(true);
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setIptc(iptcData);
    d->oldKeywords = exiv2Iface.getImageKeywords();

    d->keywordsBox->clear();
    d->keywordsCheck->setChecked(false);
    if (!d->oldKeywords.isEmpty())
    {
        d->keywordsBox->insertStringList(d->oldKeywords);
        d->keywordsCheck->setChecked(true);
    }
    d->keywordEdit->setEnabled(d->keywordsCheck->isChecked());
    d->keywordsBox->setEnabled(d->keywordsCheck->isChecked());
    d->addKeywordButton->setEnabled(d->keywordsCheck->isChecked());
    d->delKeywordButton->setEnabled(d->keywordsCheck->isChecked());

    blockSignals(false);
}

void IPTCKeywords::applyMetadata(TQByteArray& iptcData)
{
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setIptc(iptcData);
    TQStringList newKeywords;    

    for (TQListBoxItem *item = d->keywordsBox->firstItem();
         item; item = item->next()) 
        newKeywords.append(item->text());

    if (d->keywordsCheck->isChecked())
        exiv2Iface.setImageKeywords(d->oldKeywords, newKeywords);
    else
        exiv2Iface.setImageKeywords(d->oldKeywords, TQStringList());

    iptcData = exiv2Iface.getIptc();
}

}  // namespace KIPIMetadataEditPlugin

