/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2005-17-06
 * Description : a kipi plugin to export images to Flickr web service
 *
 * Copyright (C) 2005-2008 by Vardhman Jain <vardhman at gmail dot com>
 * Copyright (C) 2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqpushbutton.h>
#include <tqtimer.h>
#include <tqpixmap.h>
#include <tqcursor.h>
#include <tqprogressdialog.h>
#include <tqspinbox.h>
#include <tqcheckbox.h>
#include <tqstringlist.h>
#include <tqradiobutton.h>

// KDE includes.

#include <klineedit.h>
#include <khelpmenu.h>
#include <tdepopupmenu.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <tdeapplication.h>
#include <kiconloader.h>
#include <tdehtml_part.h>
#include <tdehtmlview.h>
#include <ktabwidget.h>
#include <krun.h>
#include <kdebug.h>
#include <tdeconfig.h>
#include <tdeversion.h>
#if KDE_IS_VERSION(3,2,0)
#include <tdewallet.h>
#endif

// Libkipi includes.

#include <libkipi/interface.h>
#include <libkipi/imagedialog.h>

// Local includes.

#include "kpaboutdata.h"
#include "pluginsversion.h"
#include "login.h"
#include "imageslist.h"
#include "flickrtalker.h"
#include "flickritem.h"
#include "flickrviewitem.h"
#include "flickrwidget.h"
#include "flickralbumdialog.h"
#include "flickrwindow.h"
#include "flickrwindow.moc"

namespace KIPIFlickrExportPlugin
{

FlickrWindow::FlickrWindow(KIPI::Interface* interface, const TQString &tmpFolder, TQWidget *parent)
            : KDialogBase(parent, 0, false, i18n("Export to Flickr Web Service"),
                          Help|User1|Close, Close, false)
{ 
    m_tmp                    = tmpFolder;
    m_interface              = interface;
    m_uploadCount            = 0;
    m_uploadTotal            = 0;
//  m_wallet                 = 0;
    m_widget                 = new FlickrWidget(this, interface);
    m_photoView              = m_widget->m_photoView;
//  m_newAlbumBtn            = m_widget->m_newAlbumBtn;
    m_resizeCheckBox         = m_widget->m_resizeCheckBox;
    m_publicCheckBox         = m_widget->m_publicCheckBox;
    m_familyCheckBox         = m_widget->m_familyCheckBox;
    m_friendsCheckBox        = m_widget->m_friendsCheckBox;
    m_dimensionSpinBox       = m_widget->m_dimensionSpinBox;
    m_imageQualitySpinBox    = m_widget->m_imageQualitySpinBox;
    m_tagsLineEdit           = m_widget->m_tagsLineEdit;
    m_exportHostTagsCheckBox = m_widget->m_exportHostTagsCheckBox;
    m_stripSpaceTagsCheckBox = m_widget->m_stripSpaceTagsCheckBox;
    m_changeUserButton       = m_widget->m_changeUserButton;
    m_userNameDisplayLabel   = m_widget->m_userNameDisplayLabel;
    m_imglst                 = m_widget->m_imglst;

    setButtonGuiItem(User1, KGuiItem(i18n("Start Uploading"), SmallIcon("network")));
    setMainWidget(m_widget);
    m_widget->setMinimumSize(600, 400);

    connect(m_imglst, TQT_SIGNAL(signalImageListChanged(bool)),
            this, TQT_SLOT(slotImageListChanged(bool)));

    //m_startUploadButton->setEnabled(false);
    //m_albumView->setRootIsDecorated(true);
    //m_newAlbumBtn->setEnabled(false);

    // --------------------------------------------------------------------------
    // About data and help button.

    m_about = new KIPIPlugins::KPAboutData(I18N_NOOP("Flickr Export"),
                                           0,
                                           TDEAboutData::License_GPL,
                                           I18N_NOOP("A Kipi plugin to export image collection to "
                                                     "Flickr web service."),
                                           "(c) 2005-2008, Vardhman Jain\n"
                                           "(c) 2008, Gilles Caulier");

    m_about->addAuthor("Vardhman Jain", I18N_NOOP("Author and maintainer"),
                       "Vardhman at gmail dot com");

    m_about->addAuthor("Gilles Caulier", I18N_NOOP("Developer"),
                       "caulier dot gilles at gmail dot com");

    KHelpMenu* helpMenu = new KHelpMenu(this, m_about, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("Plugin Handbook"), this, TQT_SLOT(slotHelp()), 0, -1, 0);
    actionButton(Help)->setPopup(helpMenu->menu());

    // --------------------------------------------------------------------------

    m_talker = new FlickrTalker(this);

    connect(m_talker, TQT_SIGNAL( signalError( const TQString& ) ),
            m_talker, TQT_SLOT( slotError( const TQString& ) ));

    connect(m_talker, TQT_SIGNAL( signalBusy( bool ) ),
            this, TQT_SLOT( slotBusy( bool ) ));

    connect(m_talker, TQT_SIGNAL( signalAddPhotoSucceeded() ),
            this, TQT_SLOT( slotAddPhotoSucceeded() ));

    connect(m_talker, TQT_SIGNAL( signalAddPhotoFailed( const TQString& ) ),
            this, TQT_SLOT( slotAddPhotoFailed( const TQString& ) ));

    connect(m_talker, TQT_SIGNAL( signalListPhotoSetsSucceeded( const TQValueList<FPhotoSet>& ) ),
            this, TQT_SLOT( slotListPhotoSetsResponse( const TQValueList<FPhotoSet>& ) ));

    //connect( m_talker, TQT_SIGNAL( signalAlbums( const TQValueList<GAlbum>& ) ),
    //         TQT_SLOT( slotAlbums( const TQValueList<GAlbum>& ) ) );

    //connect( m_talker, TQT_SIGNAL( signalPhotos( const TQValueList<GPhoto>& ) ),
    //         TQT_SLOT( slotPhotos( const TQValueList<GPhoto>& ) ) );

    // --------------------------------------------------------------------------

    m_progressDlg = new TQProgressDialog(this, 0, true);
    m_progressDlg->setAutoReset(true);
    m_progressDlg->setAutoClose(true);

    connect(m_progressDlg, TQT_SIGNAL( canceled() ),
            this, TQT_SLOT( slotAddPhotoCancel() ));

    connect(m_changeUserButton, TQT_SIGNAL( clicked() ),
            this, TQT_SLOT( slotUserChangeRequest() ));

    connect(m_talker, TQT_SIGNAL( signalTokenObtained(const TQString&) ),
            this, TQT_SLOT( slotTokenObtained(const TQString&) ));

    //connect( m_tagView, TQT_SIGNAL( selectionChanged() ),
    //         TQT_SLOT( slotTagSelected() ) );

    //connect( m_photoView->browserExtension(), TQT_SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ),
    //         TQT_SLOT( slotOpenPhoto( const KURL& ) ) );

    //connect( m_newAlbumBtn, TQT_SIGNAL( clicked() ),
    //         TQT_SLOT( slotNewAlbum() ) );

    // --------------------------------------------------------------------------

    readSettings();

    m_authProgressDlg = new TQProgressDialog(this, 0, true);
    m_authProgressDlg->setAutoReset(true);
    m_authProgressDlg->setAutoClose(true);

    connect(m_authProgressDlg, TQT_SIGNAL( canceled() ),
            this, TQT_SLOT( slotAuthCancel() ));

    m_talker->m_authProgressDlg = m_authProgressDlg; 
    m_widget->setEnabled(false);

    kdDebug() << "Calling auth methods" << endl; 

    if(m_token.length()< 1)  
        m_talker->getFrob();
    else
        m_talker->checkToken(m_token);
}

FlickrWindow::~FlickrWindow()
{
#if KDE_IS_VERSION(3,2,0)
//   if (m_wallet)
//      delete m_wallet;
#endif

    delete m_progressDlg;
    delete m_authProgressDlg;
    delete m_talker;
    delete m_widget;
    delete m_about;
}

void FlickrWindow::readSettings()
{
    TDEConfig config("kipirc");
    config.setGroup("FlickrExport Settings");
    m_token = config.readEntry("token");

    if (config.readBoolEntry("Resize", false))
    {
        m_resizeCheckBox->setChecked(true);
        m_dimensionSpinBox->setEnabled(true);
    }
    else
    {
        m_resizeCheckBox->setChecked(false);
        m_dimensionSpinBox->setEnabled(false);
    }

    m_dimensionSpinBox->setValue(config.readNumEntry("Maximum Width", 1600));
    m_imageQualitySpinBox->setValue(config.readNumEntry("Image Quality", 85));
    m_exportHostTagsCheckBox->setChecked(config.readBoolEntry("Export Host Tags", false));
    m_stripSpaceTagsCheckBox->setChecked(config.readBoolEntry("Strip Space Host Tags", false));
    m_stripSpaceTagsCheckBox->setEnabled(m_exportHostTagsCheckBox->isChecked());
    
    if(!m_interface->hasFeature(KIPI::HostSupportsTags))
    {
        m_exportHostTagsCheckBox->setEnabled(false);
        m_stripSpaceTagsCheckBox->setEnabled(false);
    }
    
    m_publicCheckBox->setChecked(config.readBoolEntry("Public Sharing", false));
    m_familyCheckBox->setChecked(config.readBoolEntry("Family Sharing", false));
    m_friendsCheckBox->setChecked(config.readBoolEntry("Friends Sharing", false));

    resize(configDialogSize(config, TQString("FlickrExport Dialog")));
}

void FlickrWindow::writeSettings()
{
    TDEConfig config("kipirc");
    config.setGroup("FlickrExport Settings");
    config.writeEntry("token", m_token);
    config.writeEntry("Resize", m_resizeCheckBox->isChecked());
    config.writeEntry("Maximum Width",  m_dimensionSpinBox->value());
    config.writeEntry("Image Quality",  m_imageQualitySpinBox->value());
    config.writeEntry("Export Host Tags", m_exportHostTagsCheckBox->isChecked());
    config.writeEntry("Strip Space Host Tags", m_stripSpaceTagsCheckBox->isChecked());
    config.writeEntry("Public Sharing", m_publicCheckBox->isChecked());
    config.writeEntry("Family Sharing", m_familyCheckBox->isChecked());
    config.writeEntry("Friends Sharing", m_friendsCheckBox->isChecked());
    saveDialogSize(config, TQString("FlickrExport Dialog"));
    config.sync();
}

void FlickrWindow::slotHelp()
{
    TDEApplication::kApplication()->invokeHelp("flickrexport", "kipi-plugins");
}

void FlickrWindow::slotDoLogin()
{
}

void FlickrWindow::slotClose()
{
    writeSettings();
    done(Close);
}

void FlickrWindow::slotTokenObtained(const TQString& token)
{
    m_token    = token;
    m_username = m_talker->getUserName();
    m_userId   = m_talker->getUserId();
    kdDebug() << "SlotTokenObtained invoked setting user Display name to " << m_username << endl;
    m_userNameDisplayLabel->setText(TQString("<qt><b>%1</b></qt>").arg(m_username));
    m_widget->setEnabled(true);
}

void FlickrWindow::slotBusy(bool val)
{
    if (val)
    {
        setCursor(TQCursor::WaitCursor);
//      m_newAlbumBtn->setEnabled( false );
//      m_addPhotoButton->setEnabled( false );
    }
    else
    {
        setCursor(TQCursor::ArrowCursor);
//      m_newAlbumBtn->setEnabled( loggedIn );
//      m_addPhotoButton->setEnabled( loggedIn && m_albumView->selectedItem() );
    }
}

void FlickrWindow::slotError(const TQString& msg)
{
    //m_talker->slotError(msg);
    KMessageBox::error( this, msg );
}

void FlickrWindow::slotUserChangeRequest()
{
    kdDebug() << "Slot Change User Request " << endl;
    m_talker->getFrob();
//  m_addPhotoButton->setEnabled(m_selectImagesButton->isChecked());
}

/*
void FlickrWindow::slotAlbums( const TQValueList<GAlbum>& albumList )
{
    m_albumDict.clear();
    m_tagView->clear();
    //m_photoView->begin();
    //m_photoView->write( "<html></html>" );
    //m_photoView->end();

    TDEIconLoader* iconLoader = TDEApplication::kApplication()->iconLoader();
    TQPixmap pix = iconLoader->loadIcon( "folder", TDEIcon::NoGroup, 32 );

    typedef TQValueList<GAlbum> GAlbumList;
    GAlbumList::const_iterator iter;
    for ( iter = albumList.begin(); iter != albumList.end(); ++iter )
    {
        const GAlbum& album = *iter;

        if ( album.parent_ref_num == 0 )
        {
            GAlbumViewItem* item = new GAlbumViewItem( m_tagView, album.name,
                                                       album );
            item->setPixmap( 0, pix );
            m_albumDict.insert( album.ref_num, item );
        }
        else
        {
            TQListViewItem* parent = m_albumDict.find( album.parent_ref_num );
            if ( parent )
            {
                GAlbumViewItem* item = new GAlbumViewItem( parent, album.name,
                                                           album);
                item->setPixmap( 0, pix );
                m_albumDict.insert( album.ref_num, item );
            }
            else
            {
                kdWarning() << "Failed to find parent for album "
                            << album.name
                            << "with id " << album.ref_num;
            }
        }
    }


    // find and select the last selected album
    int lastSelectedID = 0;
    for ( iter = albumList.begin(); iter != albumList.end(); ++iter )
    {
        if ((*iter).name == m_lastSelectedAlbum)
        {
            lastSelectedID = (*iter).ref_num;
            break;
        }
    }

    if (lastSelectedID > 0)
    {
        GAlbumViewItem* lastSelectedItem = m_albumDict.find( lastSelectedID );
        if (lastSelectedItem)
        {
            m_tagView->setSelected( lastSelectedItem, true );
            m_tagView->ensureItemVisible( lastSelectedItem );
        }
    }
}
*/

void FlickrWindow::slotAuthCancel()
{
    m_talker->cancel();
    m_authProgressDlg->hide();
}

/*
void FlickrWindow::slotPhotos( const TQValueList<GPhoto>& photoList)
{
    // TODO
}

void FlickrWindow::slotTagSelected()
{
    // TODO
}
void FlickrWindow::slotOpenPhoto( const KURL& url )
{
    new KRun(url);
}
*/

void FlickrWindow::slotListPhotoSetsResponse(const TQValueList <FPhotoSet>& /*photoSetList*/)
{
    kdDebug() << "SlotListPhotoSetsResponse invoked" << endl;
    // TODO
}

void FlickrWindow::slotNewPhotoSet()
{
    // TODO
}

/** This slot is call when 'Start Uploading' button is pressed.
*/
void FlickrWindow::slotUser1()
{
    kdDebug() << "SlotUploadImages invoked" << endl;

    m_widget->m_tab->setCurrentPage(FlickrWidget::FILELIST);
    KURL::List urls = m_imglst->imageUrls();

    if (urls.isEmpty())
        return;

    typedef TQPair<KURL, FPhotoInfo> Pair;

    m_uploadQueue.clear();

    for (KURL::List::iterator it = urls.begin(); it != urls.end(); ++it)
    {
        KIPI::ImageInfo info = m_interface->info(*it);
        kdDebug() << "Adding images to the list" << endl;
        FPhotoInfo temp;

        temp.title                 = info.title();
        temp.description           = info.description();
        temp.is_public             = m_publicCheckBox->isChecked();
        temp.is_family             = m_familyCheckBox->isChecked();
        temp.is_friend             = m_friendsCheckBox->isChecked();
        TQStringList tagsFromDialog = TQStringList::split(" ", m_tagsLineEdit->text(), false);

        TQStringList           allTags;
        TQStringList::Iterator itTags;

        // Tags from the dialog
        itTags = tagsFromDialog.begin();
        while(itTags != tagsFromDialog.end())
        {
            allTags.append(*itTags);
            ++itTags;
        }

        // Tags from the database
        TQStringVariantMap attribs = info.attributes();
        TQStringList tagsFromDatabase;

        if(m_exportHostTagsCheckBox->isChecked())
        {
            tagsFromDatabase = attribs["tags"].asStringList();
            if (m_stripSpaceTagsCheckBox->isChecked())
            {
                for (TQStringList::iterator it = tagsFromDatabase.begin(); it != tagsFromDatabase.end() ; ++it)
                    *it = (*it).stripWhiteSpace().remove(" ");
            }
        }

        itTags = tagsFromDatabase.begin();

        while(itTags != tagsFromDatabase.end()) 
        {
            allTags.append(*itTags);
            ++itTags;
        }

        itTags = allTags.begin();

        while(itTags != allTags.end())
        {
            kdDebug() << "Tags list: " << (*itTags) << endl;
            ++itTags;
        }

        temp.tags = allTags; 
        m_uploadQueue.append(Pair(*it, temp));
    }

    m_uploadTotal = m_uploadQueue.count();
    m_uploadCount = 0;
    m_progressDlg->reset();
    slotAddPhotoNext();
    kdDebug() << "SlotUploadImages done" << endl;
}

void FlickrWindow::slotAddPhotoNext()
{
    if (m_uploadQueue.isEmpty())
    {
        m_progressDlg->reset();
        m_progressDlg->hide();
        //slotAlbumSelected();
        return;
    }

    typedef TQPair<KURL, FPhotoInfo> Pair;
    Pair pathComments = m_uploadQueue.first();
    FPhotoInfo info   = pathComments.second;
    bool res          = m_talker->addPhoto(pathComments.first.path(), //the file path
                                           info,
                                           m_resizeCheckBox->isChecked(),
                                           m_dimensionSpinBox->value(), 
                                           m_imageQualitySpinBox->value());
    if (!res)
    {
        slotAddPhotoFailed("");
        return;
    }

    m_progressDlg->setLabelText(i18n("Uploading file %1").arg(pathComments.first.filename()));

    if (m_progressDlg->isHidden())
        m_progressDlg->show();
}

void FlickrWindow::slotAddPhotoSucceeded()
{
    // Remove photo uploaded from the list
    m_imglst->removeItemByUrl(m_uploadQueue.first().first);
    m_uploadQueue.pop_front();
    m_uploadCount++;
    m_progressDlg->setProgress(m_uploadCount, m_uploadTotal);
    slotAddPhotoNext();
}

void FlickrWindow::slotAddPhotoFailed(const TQString& msg)
{
    if (KMessageBox::warningContinueCancel(this,
                     i18n("Failed to upload photo into Flickr. %1\nDo you want to continue?").arg(msg))
                     != KMessageBox::Continue)
    {
        m_uploadQueue.clear();
        m_progressDlg->reset();
        m_progressDlg->hide();
        // refresh the thumbnails
        //slotTagSelected();
    }
    else
    {
        m_uploadQueue.pop_front();
        m_uploadTotal--;
        m_progressDlg->setProgress(m_uploadCount, m_uploadTotal);
        slotAddPhotoNext();
    }
}

void FlickrWindow::slotAddPhotoCancel()
{
    m_uploadQueue.clear();
    m_progressDlg->reset();
    m_progressDlg->hide();

    m_talker->cancel();

    // refresh the thumbnails
    //slotTagSelected();
}

void FlickrWindow::slotImageListChanged(bool state)
{
    enableButton(User1, !state);
}

} // namespace KIPIFlickrExportPlugin
