/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2004-10-01
 * Description : Screenshot batch dialog
 *
 * Copyright (C) 2004-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) Richard J. Moore 1997-2002 from KSnapshot
 * Copyright (C) Matthias Ettrich 2000 from KSnapshot
 * Copyright (C) Aaron J. Seigo 2002 from KSnapshot
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// C Ansi includes

extern "C"
{
#include <unistd.h>
}

// Include files for TQt

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqfileinfo.h>
#include <tqapplication.h>
#include <tqwhatsthis.h>
#include <tqcheckbox.h>
#include <tqpushbutton.h>
#include <tqframe.h>
#include <tqwidgetlist.h>

// Include files for KDE

#include <tdelocale.h>
#include <knotifyclient.h>
#include <kdebug.h>
#include <tdeconfig.h>
#include <tdemessagebox.h>
#include <knuminput.h>
#include <tdeapplication.h>
#include <tdeaboutdata.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <tdepopupmenu.h>
#include <kstandarddirs.h>

// Local includes

#include "pluginsversion.h"
#include "screenshotdialog.h"
#include "acquireimagedialog.h"
#include "screenshotdialog.moc"

namespace KIPIAcquireImagesPlugin
{

ScreenGrabDialog::ScreenGrabDialog( KIPI::Interface* interface, TQWidget *parent, const char *name)
                : KDialogBase(parent, name, false, i18n("Screenshot"),
                              Help|User1|Close, Close, false, i18n("&New Snapshot")),
                  m_interface( interface )
{
    m_inSelect = false;
    TQWidget* box = new TQWidget( this );
    setMainWidget(box);
    TQVBoxLayout *layout = new TQVBoxLayout(box);

    //---------------------------------------------
   
    TQLabel *label1 = new TQLabel(i18n("This dialog will grab either your desktop or a single\n"
                                     "application window. If you grab a single window your mouse\n"
                                     "cursor will change into crosshairs; then, simply select the\n"
                                     "window with your mouse."), box);
    layout->addWidget(label1);

    //---------------------------------------------

    m_desktopCB = new TQCheckBox(i18n("Grab the entire desktop"), box);
    TQWhatsThis::add( m_desktopCB, i18n( "<p>If you enable this option, the entire desktop will be grabbed; "
                                        "otherwise, only the active windows." ) );
    layout->addWidget(m_desktopCB);

    //---------------------------------------------

    m_hideCB = new TQCheckBox(i18n("Hide all host application windows"), box);
    TQWhatsThis::add( m_hideCB, i18n( "<p>If you enable this option, all host application windows will be hidden "
                                     "during the grab operation." ) );
    layout->addWidget(m_hideCB);

    //---------------------------------------------

    TQLabel *label2 = new TQLabel(i18n("Delay:"), box);
    layout->addWidget(label2);
    m_delay = new KIntNumInput(box);
    TQWhatsThis::add( m_delay, i18n( "<p>The delay in seconds before the grab operation is started.") );
    m_delay->setRange(0, 60);
    layout->addWidget(m_delay);
    layout->addStretch(1);

    //---------------------------------------------

    m_grabber = new TQWidget( 0, 0, WStyle_Customize | WX11BypassWM );
    m_grabber->move( -4000, -4000 );
    m_grabber->installEventFilter( this );

    //---------------------------------------------

    connect(this, TQT_SIGNAL(user1Clicked()),
            this, TQT_SLOT(slotGrab()));
    
    connect(this, TQT_SIGNAL(closeClicked()),
            this, TQT_SLOT(slotClose()));

    connect( &m_grabTimer, TQT_SIGNAL(timeout()),
             this, TQT_SLOT(slotPerformGrab()));

    //---------------------------------------------

    // Read all settings from configuration file.

    m_config = new TDEConfig("kipirc");
    m_config->setGroup("ScreenshotImages Settings");

    if (m_config->readEntry("GrabDesktop", "true") == "true")
        m_desktopCB->setChecked( true );
    else
        m_desktopCB->setChecked( false );

    if (m_config->readEntry("HideHostWin", "true") == "true")
        m_hideCB->setChecked( true );
    else
        m_hideCB->setChecked( false );

    m_delay->setValue(m_config->readNumEntry("Delay", 1));

    delete m_config;
    
    // About data and help button.
    
    TDEAboutData* about = new TDEAboutData("kipiplugins",
                                       I18N_NOOP("Acquire images"), 
                                       kipiplugins_version,
                                       I18N_NOOP("A Kipi plugin to acquire images"),
                                       TDEAboutData::License_GPL,
                                       "(c) 2003-2008, Gilles Caulier", 
                                       0,
                                       "http://extragear.kde.org/apps/kipi");
    
    about->addAuthor("Gilles Caulier", I18N_NOOP("Author and maintainer"),
                     "caulier dot gilles at gmail dot com");

    m_helpButton = actionButton( Help );
    KHelpMenu* helpMenu = new KHelpMenu(this, about, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("Plugin Handbook"), this, TQT_SLOT(slotHelp()), 0, -1, 0);
    m_helpButton->setPopup( helpMenu->menu() );
}

ScreenGrabDialog::~ScreenGrabDialog()
{
}

void ScreenGrabDialog::slotHelp()
{
    TDEApplication::kApplication()->invokeHelp("acquireimages", "kipi-plugins");
}    

void ScreenGrabDialog::slotClose( void )
{
    // Write all settings in configuration file.

    m_config = new TDEConfig("kipirc");
    m_config->setGroup("ScreenshotImages Settings");
    m_config->writeEntry("GrabDesktop", m_desktopCB->isChecked());
    m_config->writeEntry("HideHostWin", m_hideCB->isChecked());
    m_config->writeEntry("Delay", m_delay->value());
    m_config->sync();
    delete m_config;

    close();
    delete this;
}

void ScreenGrabDialog::slotGrab()
{
    hide();

    // Hiding the Host windows
    m_hiddenWindows.clear();
    if (m_hideCB->isChecked())
    {
        TQWidgetList  *list = TQApplication::topLevelWidgets();
        TQWidgetListIt it( *list );
        TQWidget * w;
        while ( (w=it.current()) != 0 )
        {
            ++it;
            if ( w->isVisible())
            {
                m_hiddenWindows.append( w );
                w->hide();
            }
        }
        delete list;
    }

    kapp->processEvents();
    TQApplication::syncX();

    if ( m_delay->value() != 0 )
        m_grabTimer.start( m_delay->value() * 1000, true );
    else
    {
        m_grabber->show();
        m_grabber->grabMouse( crossCursor );
    }
}

void ScreenGrabDialog::slotPerformGrab()
{
    m_grabber->releaseMouse();
    m_grabber->hide();
    m_grabTimer.stop();

    if ( m_desktopCB->isChecked() == false )
    {
        Window root;
        Window child;
        uint mask;
        int rootX, rootY, winX, winY;
        XQueryPointer( tqt_xdisplay(), tqt_xrootwin(), &root, &child,
                       &rootX, &rootY, &winX, &winY,
                       &mask);

        int x, y;
        unsigned int w, h;
        unsigned int border;
        unsigned int depth;
        XGetGeometry( tqt_xdisplay(), child, &root, &x, &y,
                      &w, &h, &border, &depth );

        m_snapshot = TQPixmap::grabWindow( tqt_xrootwin(), x, y, w, h );
    }
    else
        m_snapshot = TQPixmap::grabWindow( tqt_xrootwin() );

    if (m_snapshot.isNull())
    {
       KMessageBox::sorry(this, i18n("Unable to take snapshot."),
                          i18n("Screenshot Error"));

       endGrab();
       return;
    }

    TQApplication::restoreOverrideCursor();
    KNotifyClient::beep();

    m_screenshotImage = m_snapshot.convertToImage();
    m_acquireImageDialog = new AcquireImageDialog( m_interface, this, m_screenshotImage);
    m_acquireImageDialog->setMinimumWidth(400);
    m_acquireImageDialog->exec();

    endGrab();
}

void ScreenGrabDialog::endGrab(void)
{
    // Restore the Host windows

    if (m_hideCB->isChecked())
    {
       for( TQValueList< TQWidget* >::ConstIterator it = m_hiddenWindows.begin();
            it != m_hiddenWindows.end();
            ++it )
           (*it)->show();
       TQApplication::syncX();
    }

    show();
}

bool ScreenGrabDialog::eventFilter( TQObject* o, TQEvent* e)
{
    if ( o == m_grabber && e->type() == TQEvent::MouseButtonPress )
    {
        TQMouseEvent* me = (TQMouseEvent*) e;

        if ( TQWidget::mouseGrabber() != m_grabber )
           return false;

        if ( me->button() == Qt::LeftButton )
           slotPerformGrab();
    }

    return false;
}

}  // NameSpace KIPIAcquireImagesPlugin
