/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2008-08-21
 * Description : Editor tool settings template box
 *
 * Copyright (C) 2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqhbox.h>
#include <tqvbox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqstring.h>
#include <tqtooltip.h>
#include <tqwhatsthis.h>

// KDE includes.

#include <tdeapplication.h>
#include <kcolorbutton.h>
#include <kdialog.h>
#include <tdeglobalsettings.h>
#include <kiconloader.h>
#include <tdelocale.h>
#include <kpushbutton.h>
#include <kstandarddirs.h>
#include <kstdguiitem.h>

// LibKDcraw includes.

#include <libkdcraw/rnuminput.h>

// Local includes.

#include "ddebug.h"
#include "imagepaniconwidget.h"
#include "editortoolsettings.h"
#include "editortoolsettings.moc"

using namespace KDcrawIface;

namespace Digikam
{

class EditorToolSettingsPriv
{

public:

    EditorToolSettingsPriv()
    {
        okBtn        = 0;
        cancelBtn    = 0;
        tryBtn       = 0;
        defaultBtn   = 0;
        mainVBox     = 0;
        plainPage    = 0;
        btnBox1      = 0;
        btnBox2      = 0;
        btnBox3      = 0;
        saveAsBtn    = 0;
        loadBtn      = 0;
        guideBox     = 0;
        guideColorBt = 0;
        guideSize    = 0;
        panIconView  = 0;
    }

    TQHBox              *btnBox1;
    TQHBox              *btnBox2;
    TQHBox              *btnBox3;
    TQHBox              *guideBox;

    TQVBox              *mainVBox;
    TQWidget            *plainPage;

    KPushButton        *okBtn;
    KPushButton        *cancelBtn;
    KPushButton        *tryBtn;
    KPushButton        *defaultBtn;
    KPushButton        *saveAsBtn;
    KPushButton        *loadBtn;

    KColorButton       *guideColorBt;

    ImagePanIconWidget *panIconView;

    RIntNumInput       *guideSize;
};

EditorToolSettings::EditorToolSettings(int buttonMask, int toolMask, TQWidget *parent)
                  : TQScrollView(parent)
{
    d = new EditorToolSettingsPriv;

    viewport()->setBackgroundMode(TQt::PaletteBackground);
    setResizePolicy(TQScrollView::AutoOneFit);
    setFrameStyle(TQFrame::NoFrame);

    d->mainVBox = new TQVBox(viewport());
    addChild(d->mainVBox);

    // ---------------------------------------------------------------

    TQFrame *frame     = new TQFrame(d->mainVBox);
    frame->setFrameStyle(TQFrame::Panel|TQFrame::Sunken);
    TQVBoxLayout* vlay = new TQVBoxLayout(frame, 5, 0);
    d->panIconView    = new ImagePanIconWidget(360, 240, frame);
    TQWhatsThis::add(d->panIconView, i18n("<p>Here you can see the original image panel "
                                         "which can help you to select the clip preview."
                                         "<p>Click and drag the mouse cursor in the "
                                         "red rectangle to change the clip focus."));
    vlay->addWidget(d->panIconView, 0, TQt::AlignCenter);

    if (!(toolMask & PanIcon))
        frame->hide();

    // ---------------------------------------------------------------

    d->plainPage = new TQWidget(d->mainVBox);
    d->guideBox  = new TQHBox(d->mainVBox);
    d->btnBox1   = new TQHBox(d->mainVBox);
    d->btnBox2   = new TQHBox(d->mainVBox);

    // ---------------------------------------------------------------

    new TQLabel(i18n("Guide:"), d->guideBox);
    TQLabel *space4  = new TQLabel(d->guideBox);
    d->guideColorBt = new KColorButton(TQColor(TQt::red), d->guideBox);
    TQWhatsThis::add(d->guideColorBt, i18n("<p>Set here the color used to draw guides dashed-lines."));
    d->guideSize    = new RIntNumInput(d->guideBox);
    d->guideSize->setRange(1, 5, 1);
    d->guideSize->setDefaultValue(1);
    TQWhatsThis::add(d->guideSize, i18n("<p>Set here the width in pixels used to draw guides dashed-lines."));

    d->guideBox->setStretchFactor(space4, 10);
    d->guideBox->setSpacing(spacingHint());
    d->guideBox->setMargin(0);

    if (!(toolMask & ColorGuide))
        d->guideBox->hide();

    // ---------------------------------------------------------------

    d->defaultBtn = new KPushButton(d->btnBox1);
    d->defaultBtn->setGuiItem(KStdGuiItem::defaults());
    d->defaultBtn->setIconSet(SmallIconSet("reload_page"));
    TQToolTip::add(d->defaultBtn, i18n("<p>Reset all settings to their default values."));
    if (!(buttonMask & Default))
        d->defaultBtn->hide();

    TQLabel *space = new TQLabel(d->btnBox1);

    d->okBtn = new KPushButton(d->btnBox1);
    d->okBtn->setGuiItem(KStdGuiItem::ok());
    if (!(buttonMask & Ok))
        d->okBtn->hide();

    d->cancelBtn = new KPushButton(d->btnBox1);
    d->cancelBtn->setGuiItem(KStdGuiItem::cancel());
    if (!(buttonMask & Cancel))
        d->cancelBtn->hide();

    d->btnBox1->setStretchFactor(space, 10);
    d->btnBox1->setSpacing(spacingHint());
    d->btnBox1->setMargin(0);

    if (!(buttonMask & Default) && !(buttonMask & Ok) && !(buttonMask & Cancel))
        d->btnBox1->hide();

    // ---------------------------------------------------------------

    d->loadBtn = new KPushButton(d->btnBox2);
    d->loadBtn->setGuiItem(KStdGuiItem::open());
    d->loadBtn->setText(i18n("Load..."));
    TQToolTip::add(d->loadBtn, i18n("<p>Load all parameters from settings text file."));
    if (!(buttonMask & Load))
        d->loadBtn->hide();

    d->saveAsBtn = new KPushButton(d->btnBox2);
    d->saveAsBtn->setGuiItem(KStdGuiItem::saveAs());
    TQToolTip::add(d->saveAsBtn, i18n("<p>Save all parameters to settings text file."));
    if (!(buttonMask & SaveAs))
        d->saveAsBtn->hide();

    TQLabel *space2 = new TQLabel(d->btnBox2);

    d->tryBtn = new KPushButton(d->btnBox2);
    d->tryBtn->setGuiItem(KStdGuiItem::apply());
    d->tryBtn->setText(i18n("Try"));
    TQToolTip::add(d->tryBtn, i18n("<p>Try all settings."));
    if (!(buttonMask & Try))
        d->tryBtn->hide();

    d->btnBox2->setStretchFactor(space2, 10);
    d->btnBox2->setSpacing(spacingHint());
    d->btnBox2->setMargin(0);

    if (!(buttonMask & Load) && !(buttonMask & SaveAs) && !(buttonMask & Try))
        d->btnBox2->hide();

    // ---------------------------------------------------------------

    connect(d->okBtn, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalOkClicked()));

    connect(d->cancelBtn, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalCancelClicked()));

    connect(d->tryBtn, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalTryClicked()));

    connect(d->defaultBtn, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalDefaultClicked()));

    connect(d->saveAsBtn, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalSaveAsClicked()));

    connect(d->loadBtn, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalLoadClicked()));

    connect(d->guideColorBt, TQT_SIGNAL(changed(const TQColor&)),
            this, TQT_SIGNAL(signalColorGuideChanged()));

    connect(d->guideSize, TQT_SIGNAL(valueChanged(int)),
            this, TQT_SIGNAL(signalColorGuideChanged()));
}

EditorToolSettings::~EditorToolSettings()
{
    delete d;
}

TQSize EditorToolSettings::minimumSizeHint() const
{
    // Editor Tools usually require a larger horizontal space than other widgets in right side bar
    // Set scroll area to a horizontal minimum size sufficient for the settings.
    // Do not touch vertical size hint.
    // Limit to 40% of the desktop width.
    TQSize hint = TQScrollView::minimumSizeHint();
    TQRect desktopRect = TDEGlobalSettings::desktopGeometry(d->mainVBox);
    hint.setWidth(TQMIN(d->mainVBox->minimumSizeHint().width(), desktopRect.width() * 2 / 5));
    return hint;
}

int EditorToolSettings::marginHint()
{
    return KDialog::marginHint();
}

int EditorToolSettings::spacingHint()
{
    return KDialog::spacingHint();
}

TQWidget *EditorToolSettings::plainPage() const
{
    return d->plainPage;
}

ImagePanIconWidget* EditorToolSettings::panIconView() const
{
    return d->panIconView;
}

KPushButton* EditorToolSettings::button(int buttonCode) const
{
    if (buttonCode & Default)
        return d->defaultBtn;

    if (buttonCode & Try)
        return d->tryBtn;

    if (buttonCode & Ok)
        return d->okBtn;

    if (buttonCode & Cancel)
        return d->cancelBtn;

    if (buttonCode & Load)
        return d->loadBtn;

    if (buttonCode & SaveAs)
        return d->saveAsBtn;

    return 0;
}

void EditorToolSettings::enableButton(int buttonCode, bool state)
{
    KPushButton *btn = button(buttonCode);
    if (btn) btn->setEnabled(state);
}

TQColor EditorToolSettings::guideColor() const
{
    return d->guideColorBt->color();
}

void EditorToolSettings::setGuideColor(const TQColor& color)
{
    d->guideColorBt->setColor(color);
}

int EditorToolSettings::guideSize() const
{
    return d->guideSize->value();
}

void EditorToolSettings::setGuideSize(int size)
{
    d->guideSize->setValue(size);
}

} // NameSpace Digikam
