/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2006-07-24
 * Description : a dialog to select a camera folders.
 * 
 * Copyright (C) 2006-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.

#include <tqlabel.h>
#include <tqlayout.h>
#include <tqframe.h>

// KDE includes.

#include <tdelocale.h>
#include <kiconloader.h>
#include <tdeapplication.h>

// Local includes.

#include "ddebug.h"
#include "cameraiconview.h"
#include "camerafolderitem.h"
#include "camerafolderview.h"
#include "camerafolderdialog.h"
#include "camerafolderdialog.moc"

namespace Digikam
{

CameraFolderDialog::CameraFolderDialog(TQWidget *parent, CameraIconView *cameraView, 
                                       const TQStringList& cameraFolderList,
                                       const TQString& cameraName, const TQString& rootPath)
                  : KDialogBase(parent, 0, true,
                                i18n("%1 - Select Camera Folder").arg(cameraName), 
                                Help|Ok|Cancel, Ok, true)
{
    setHelp("camerainterface.anchor", "digikam");
    enableButtonOK(false);

    m_rootPath = rootPath;

    TQFrame *page      = makeMainWidget();
    TQGridLayout* grid = new TQGridLayout(page, 2, 1, 0, spacingHint());
    
    m_folderView    = new CameraFolderView(page);
    TQLabel *logo    = new TQLabel(page);
    TQLabel *message = new TQLabel(page);

    TDEIconLoader* iconLoader = TDEApplication::kApplication()->iconLoader();
    logo->setPixmap(iconLoader->loadIcon("digikam", TDEIcon::NoGroup, 128, TDEIcon::DefaultState, 0, true));    
    message->setText(i18n("<p>Please select the camera folder "
                          "where you want to upload the images.</p>"));

    grid->addMultiCellWidget(logo, 0, 0, 0, 0);
    grid->addMultiCellWidget(message, 1, 1, 0, 0);
    grid->addMultiCellWidget(m_folderView, 0, 2, 1, 1);
    grid->setRowStretch(2, 10);

    m_folderView->addVirtualFolder(cameraName);
    m_folderView->addRootFolder("/", cameraView->countItemsByFolder(rootPath));

    for (TQStringList::const_iterator it = cameraFolderList.begin();
         it != cameraFolderList.end(); ++it)
    {
        TQString folder(*it);
        if (folder.startsWith(rootPath) && rootPath != TQString("/"))
            folder.remove(0, rootPath.length());

        if (folder != TQString("/") && !folder.isEmpty())
        {
            TQString root = folder.section( '/', 0, -2 );
            if (root.isEmpty()) root = TQString("/");

            TQString sub = folder.section( '/', -1 );
            m_folderView->addFolder(root, sub, cameraView->countItemsByFolder(*it));
            DDebug() << "Camera folder: '" << folder << "' (root='" << root << "', sub='" <<sub <<"')" << endl;
        }
    }

    connect(m_folderView, TQT_SIGNAL(signalFolderChanged(CameraFolderItem*)),
            this, TQT_SLOT(slotFolderPathSelectionChanged(CameraFolderItem*)));

    resize(500, 500);
}

CameraFolderDialog::~CameraFolderDialog()
{
}

TQString CameraFolderDialog::selectedFolderPath()
{
    TQListViewItem *item = m_folderView->currentItem();
    if (!item) return TQString();

    CameraFolderItem *folderItem = static_cast<CameraFolderItem *>(item);
    if (folderItem->isVirtualFolder())
        return TQString(m_rootPath);

    // Case of Gphoto2 cameras. No need to duplicate root '/'.
    if (m_rootPath == TQString("/"))
        return(folderItem->folderPath());

    return(m_rootPath + folderItem->folderPath());
}

void CameraFolderDialog::slotFolderPathSelectionChanged(CameraFolderItem* item)
{
    if (item) 
    {
        enableButtonOK(true);
        DDebug() << "Camera folder path: " << selectedFolderPath() << endl;
    }
    else
    {
        enableButtonOK(false);
    }
}

}  // namespace Digikam

