/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2005-02-14
 * Description : a plugin to insert a text over an image.
 * 
 * Copyright (C) 2005-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) 2006-2008 by Marcel Wiesweg <marcel dot wiesweg at gmx dot de>
 * 
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */
 
// TQt includes. 
 
#include <tqvgroupbox.h>
#include <tqlabel.h>
#include <tqpushbutton.h>
#include <tqwhatsthis.h>
#include <tqlayout.h>
#include <tqframe.h>
#include <tqcombobox.h>
#include <tqcheckbox.h>
#include <tqpixmap.h>
#include <tqpainter.h>
#include <tqbrush.h>
#include <tqpen.h>
#include <tqfont.h>
#include <tqtimer.h> 
#include <tqhbuttongroup.h> 
#include <tqtooltip.h>

// KDE includes.

#include <tdelocale.h>
#include <kcursor.h>
#include <tdeaboutdata.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <tdeapplication.h>
#include <tdepopupmenu.h>
#include <kstandarddirs.h>
#include <tdeconfig.h>
#include <kcolorbutton.h>
#include <ktextedit.h> 

// Local includes.

#include "version.h"
#include "ddebug.h"
#include "dimg.h"
#include "imageiface.h"
#include "inserttextwidget.h"
#include "fontchooserwidget.h"
#include "imageeffect_inserttext.h"
#include "imageeffect_inserttext.moc"

namespace DigikamInsertTextImagesPlugin
{

ImageEffect_InsertText::ImageEffect_InsertText(TQWidget* parent)
                      : Digikam::ImageDlgBase(parent, i18n("Insert Text on Photograph"),
                                              "inserttext", false, false)
{
    TQString whatsThis;
    
    // About data and help button.
    
    TDEAboutData* about = new TDEAboutData("digikam",
                                       I18N_NOOP("Insert Text"), 
                                       digikam_version,
                                       I18N_NOOP("A digiKam image plugin for inserting text on a photograph."),
                                       TDEAboutData::License_GPL,
                                       "(c) 2005-2006, Gilles Caulier\n"
                                       "(c) 2006-2007, Gilles Caulier and Marcel Wiesweg",
                                       0,
                                       "http://www.digikam.org");
                                       
    about->addAuthor("Gilles Caulier", I18N_NOOP("Author and maintainer"),
                     "caulier dot gilles at gmail dot com");

    about->addAuthor("Marcel Wiesweg", I18N_NOOP("Developer"),
                     "marcel dot wiesweg at gmx dot de");

    setAboutData(about);
    
    // -------------------------------------------------------------
    
    TQFrame *frame = new TQFrame(plainPage());
    frame->setFrameStyle(TQFrame::Panel|TQFrame::Sunken);
    TQVBoxLayout* l  = new TQVBoxLayout(frame, 5, 0);
    m_previewWidget = new InsertTextWidget(480, 320, frame);
    l->addWidget(m_previewWidget);
    TQWhatsThis::add( m_previewWidget, i18n("<p>This previews the text inserted in the image. "
                                           "You can use the mouse to move the text to the right location."));
    setPreviewAreaWidget(frame);                                           
    
    // -------------------------------------------------------------

    TQWidget *gbox2        = new TQWidget(plainPage());
    TQGridLayout *gridBox2 = new TQGridLayout( gbox2, 9, 1, spacingHint());
    
    m_textEdit = new KTextEdit(gbox2);
    m_textEdit->setCheckSpellingEnabled(true);
    m_textEdit->setWordWrap(TQTextEdit::NoWrap);
    TQWhatsThis::add( m_textEdit, i18n("<p>Here, enter the text you want to insert in your image."));
    gridBox2->addMultiCellWidget(m_textEdit, 0, 2, 0, 1);
    
    // -------------------------------------------------------------
    
    m_fontChooserWidget = new FontChooserWidget(gbox2);
    TQWhatsThis::add( m_textEdit, i18n("<p>Here you can choose the font to be used."));
    gridBox2->addMultiCellWidget(m_fontChooserWidget, 3, 3, 0, 1);
    
    // -------------------------------------------------------------
    
    TDEIconLoader icon;
    m_alignButtonGroup = new TQHButtonGroup(gbox2);
    
    TQPushButton *alignLeft = new TQPushButton( m_alignButtonGroup );
    m_alignButtonGroup->insert(alignLeft, ALIGN_LEFT);
    alignLeft->setPixmap( icon.loadIcon( "format-text-direction-ltr", (TDEIcon::Group)TDEIcon::Small ) );
    alignLeft->setToggleButton(true);
    TQToolTip::add( alignLeft, i18n( "Align text to the left" ) );
    
    TQPushButton *alignRight = new TQPushButton( m_alignButtonGroup );
    m_alignButtonGroup->insert(alignRight, ALIGN_RIGHT);
    alignRight->setPixmap( icon.loadIcon( "format-text-direction-rtl", (TDEIcon::Group)TDEIcon::Small ) );
    alignRight->setToggleButton(true);
    TQToolTip::add( alignRight, i18n( "Align text to the right" ) );
    
    TQPushButton *alignCenter = new TQPushButton( m_alignButtonGroup );
    m_alignButtonGroup->insert(alignCenter, ALIGN_CENTER);
    alignCenter->setPixmap( icon.loadIcon( "text_center", (TDEIcon::Group)TDEIcon::Small ) );
    alignCenter->setToggleButton(true);
    TQToolTip::add( alignCenter, i18n( "Align text to center" ) );
    
    TQPushButton *alignBlock = new TQPushButton( m_alignButtonGroup );
    m_alignButtonGroup->insert(alignBlock, ALIGN_BLOCK);
    alignBlock->setPixmap( icon.loadIcon( "text_block", (TDEIcon::Group)TDEIcon::Small ) );
    alignBlock->setToggleButton(true);
    TQToolTip::add( alignBlock, i18n( "Align text to a block" ) );
    
    m_alignButtonGroup->setExclusive(true);
    m_alignButtonGroup->setFrameShape(TQFrame::NoFrame);
    gridBox2->addMultiCellWidget(m_alignButtonGroup, 4, 4, 0, 1);
    
    // -------------------------------------------------------------
    
    TQLabel *label1 = new TQLabel(i18n("Rotation:"), gbox2);
    m_textRotation = new TQComboBox( false, gbox2 );
    m_textRotation->insertItem( i18n("None") );
    m_textRotation->insertItem( i18n("90 Degrees") );
    m_textRotation->insertItem( i18n("180 Degrees") );
    m_textRotation->insertItem( i18n("270 Degrees") );
    TQWhatsThis::add( m_textRotation, i18n("<p>Select the text rotation to use."));
    gridBox2->addMultiCellWidget(label1, 5, 5, 0, 0);
    gridBox2->addMultiCellWidget(m_textRotation, 5, 5, 1, 1);

    // -------------------------------------------------------------
        
    TQLabel *label2    = new TQLabel(i18n("Color:"), gbox2);
    m_fontColorButton = new KColorButton( TQt::black, gbox2 );
    TQWhatsThis::add( m_fontColorButton, i18n("<p>Select the font color to use."));
    gridBox2->addMultiCellWidget(label2, 6, 6, 0, 0);
    gridBox2->addMultiCellWidget(m_fontColorButton, 6, 6, 1, 1);

    // -------------------------------------------------------------
        
    m_borderText = new TQCheckBox( i18n( "Add border"), gbox2 );
    TQToolTip::add( m_borderText, i18n( "Add a solid border around text using current text color" ) );

    m_transparentText = new TQCheckBox( i18n( "Semi-transparent"), gbox2 );
    TQToolTip::add( m_transparentText, i18n( "Use semi-transparent text background under image" ) );

    gridBox2->addMultiCellWidget(m_borderText, 7, 7, 0, 1);                            
    gridBox2->addMultiCellWidget(m_transparentText, 8, 8, 0, 1);                            
    gridBox2->setRowStretch(9, 10);    
    
    setUserAreaWidget(gbox2);

    // -------------------------------------------------------------
    
    connect(m_fontChooserWidget, TQT_SIGNAL(fontSelected (const TQFont &)),
            this, TQT_SLOT(slotFontPropertiesChanged(const TQFont &)));       
            
    connect(m_fontColorButton, TQT_SIGNAL(changed(const TQColor &)),
            this, TQT_SLOT(slotUpdatePreview()));         
    
    connect(m_textEdit, TQT_SIGNAL(textChanged()),
            this, TQT_SLOT(slotUpdatePreview()));           
            
    connect(m_alignButtonGroup, TQT_SIGNAL(released(int)),
            this, TQT_SLOT(slotAlignModeChanged(int)));                        
    
    connect(m_borderText, TQT_SIGNAL(toggled(bool)),
            this, TQT_SLOT(slotUpdatePreview()));                        

    connect(m_transparentText, TQT_SIGNAL(toggled(bool)),
            this, TQT_SLOT(slotUpdatePreview()));                        
                
    connect(m_textRotation, TQT_SIGNAL(activated(int)),
            this, TQT_SLOT(slotUpdatePreview()));

    connect(this, TQT_SIGNAL(signalUpdatePreview()), TQT_SLOT(slotUpdatePreview()));
    // -------------------------------------------------------------

    slotUpdatePreview();
}

ImageEffect_InsertText::~ImageEffect_InsertText()
{
}

void ImageEffect_InsertText::readUserSettings()
{
    TDEConfig *config = kapp->config();
    config->setGroup("inserttext Tool Dialog");
    TQColor black(0, 0, 0);
    TQFont  defaultFont;

    int orgW = m_previewWidget->imageIface()->originalWidth();
    int orgH = m_previewWidget->imageIface()->originalHeight();

    if ( orgW > orgH ) m_defaultSizeFont = (int)(orgH / 8.0);
    else m_defaultSizeFont = (int)(orgW / 8.0);

    defaultFont.setPointSize(m_defaultSizeFont);
    m_textRotation->setCurrentItem( config->readNumEntry("Text Rotation", 0) );
    m_fontColorButton->setColor(config->readColorEntry("Font Color", &black));
    m_textEdit->setText(config->readEntry("Text String", i18n("Enter your text here!")));
    m_textFont = config->readFontEntry("Font Properties", &defaultFont);
    m_fontChooserWidget->setFont(m_textFont);
    m_alignTextMode = config->readNumEntry("Text Alignment", ALIGN_LEFT);
    m_borderText->setChecked( config->readBoolEntry("Border Text", false) );
    m_transparentText->setChecked( config->readBoolEntry("Transparent Text", false) );
    m_previewWidget->setPositionHint( config->readRectEntry("Position Hint") );

    static_cast<TQPushButton*>(m_alignButtonGroup->find(m_alignTextMode))->setOn(true);
    slotAlignModeChanged(m_alignTextMode);
}

void ImageEffect_InsertText::writeUserSettings()
{
    TDEConfig *config = kapp->config();
    config->setGroup("inserttext Tool Dialog");

    config->writeEntry( "Text Rotation", m_textRotation->currentItem() );
    config->writeEntry( "Font Color", m_fontColorButton->color() );
    config->writeEntry( "Text String", m_textEdit->text() );
    config->writeEntry( "Font Properties", m_textFont );
    config->writeEntry( "Text Alignment", m_alignTextMode );
    config->writeEntry( "Border Text", m_borderText->isChecked() );
    config->writeEntry( "Transparent Text", m_transparentText->isChecked() );
    config->writeEntry( "Position Hint", m_previewWidget->getPositionHint() );

    config->sync();
}

void ImageEffect_InsertText::resetValues()
{
    m_fontColorButton->blockSignals(true);
    m_alignButtonGroup->blockSignals(true);
    m_fontChooserWidget->blockSignals(true);
    
    m_textRotation->setCurrentItem(0);    // No rotation.
    m_fontColorButton->setColor(TQt::black);      
    TQFont defaultFont;  
    m_textFont = defaultFont; // Reset to default TDE font.
    m_textFont.setPointSize( m_defaultSizeFont );
    m_fontChooserWidget->setFont(m_textFont);
    m_borderText->setChecked( false ); 
    m_transparentText->setChecked( false ); 
    m_previewWidget->resetEdit();
    static_cast<TQPushButton*>(m_alignButtonGroup->find(ALIGN_LEFT))->setOn(true);
    
    m_fontChooserWidget->blockSignals(false);
    m_fontColorButton->blockSignals(false);
    m_alignButtonGroup->blockSignals(false);    
    slotAlignModeChanged(ALIGN_LEFT);
} 

void ImageEffect_InsertText::slotAlignModeChanged(int mode)
{
    m_alignTextMode = mode;
    m_textEdit->selectAll(true);     
    
    switch (m_alignTextMode)
        {
        case ALIGN_LEFT:
           m_textEdit->setAlignment( TQt::AlignLeft );
           break;
        
        case ALIGN_RIGHT:
           m_textEdit->setAlignment( TQt::AlignRight );
           break;

        case ALIGN_CENTER:
           m_textEdit->setAlignment( TQt::AlignHCenter );
           break;

        case ALIGN_BLOCK:
           m_textEdit->setAlignment( TQt::AlignJustify );
           break;
        }
        
    m_textEdit->selectAll(false);        
    emit signalUpdatePreview();
}

void ImageEffect_InsertText::slotFontPropertiesChanged(const TQFont &font)
{
    m_textFont = font;
    emit signalUpdatePreview();
}

void ImageEffect_InsertText::slotUpdatePreview()
{
    m_previewWidget->setText(m_textEdit->text(), m_textFont, m_fontColorButton->color(), m_alignTextMode, 
                             m_borderText->isChecked(), m_transparentText->isChecked(),
                             m_textRotation->currentItem());
}

void ImageEffect_InsertText::finalRendering()
{
    accept();
    kapp->setOverrideCursor( KCursor::waitCursor() );

    Digikam::ImageIface iface(0, 0);
    Digikam::DImg dest = m_previewWidget->makeInsertText();
    iface.putOriginalImage(i18n("Insert Text"), dest.bits(), dest.width(), dest.height());

    kapp->restoreOverrideCursor();
}

}  // NameSpace DigikamInsertTextImagesPlugin

