/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2005-03-27
 * Description : a digiKam image plugin for fixing dots produced by
 *               hot/stuck/dead pixels from a CCD.
 * 
 * Copyright (C) 2005-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) 2005-2006 by Unai Garro <ugarro at users dot sourceforge dot net>
 * 
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes. 

#include <tqcombobox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqwhatsthis.h>
#include <tqpushbutton.h>
#include <tqpointarray.h>

// KDE includes.

#include <tdelocale.h>
#include <tdeconfig.h>
#include <kimageio.h>
#include <tdeaboutdata.h>
#include <tdeapplication.h>
#include <kstandarddirs.h>
#include <tdefiledialog.h>
#include <kprogress.h>

// Local includes.

#include "version.h"
#include "ddebug.h"
#include "dimg.h"
#include "imageiface.h"
#include "imagewidget.h"
#include "imagedialog.h"
#include "blackframelistview.h"
#include "imageeffect_hotpixels.h"
#include "imageeffect_hotpixels.moc"

namespace DigikamHotPixelsImagesPlugin
{

ImageEffect_HotPixels::ImageEffect_HotPixels(TQWidget* parent)
                     : CtrlPanelDlg(parent, i18n("Hot Pixels Correction"), 
                                    "hotpixels", false, false, false, 
                                    Digikam::ImagePannelWidget::SeparateViewDuplicate)
{
    // No need Abort button action.
    showButton(User1, false); 

    TQString whatsThis;

    TDEAboutData* about = new TDEAboutData("digikam",
                                       I18N_NOOP("Hot Pixels Correction"), 
                                       digikam_version,
                                       I18N_NOOP("A digiKam image plugin for fixing dots produced by "
                                                 "hot/stuck/dead pixels from a CCD."),
                                       TDEAboutData::License_GPL,
                                       "(c) 2005-2006, Unai Garro\n(c) 2005-2008, Gilles Caulier", 
                                       0,
                                       "http://www.digikam.org");

    about->addAuthor("Unai Garro", I18N_NOOP("Author and maintainer"),
                     "ugarro at sourceforge dot net");

    about->addAuthor("Gilles Caulier", I18N_NOOP("Developer"),
                     "caulier dot gilles at gmail dot com");

    setAboutData(about);

    // -------------------------------------------------------------

    TQWidget *gboxSettings     = new TQWidget(m_imagePreviewWidget);
    TQGridLayout* gridSettings = new TQGridLayout(gboxSettings, 3, 2, 0, spacingHint());

    TQLabel *filterMethodLabel = new TQLabel(i18n("Filter:"), gboxSettings);
    m_filterMethodCombo       = new TQComboBox(gboxSettings);
    m_filterMethodCombo->insertItem(i18n("Average"));
    m_filterMethodCombo->insertItem(i18n("Linear"));
    m_filterMethodCombo->insertItem(i18n("Quadratic"));
    m_filterMethodCombo->insertItem(i18n("Cubic"));

    m_blackFrameButton = new TQPushButton(i18n("Black Frame..."), gboxSettings);    
    setButtonWhatsThis( Apply, i18n("<p>Use this button to add a new black frame file which will "
                                    "be used by the hot pixels removal filter.") );  

    m_blackFrameListView = new BlackFrameListView(gboxSettings);
    m_progressBar        = new KProgress(100, gboxSettings);
    m_progressBar->setValue(0);
    m_progressBar->hide();

    gridSettings->addMultiCellWidget(filterMethodLabel,    0, 0, 0, 0);
    gridSettings->addMultiCellWidget(m_filterMethodCombo,  0, 0, 1, 1);
    gridSettings->addMultiCellWidget(m_blackFrameButton,   0, 0, 2, 2);
    gridSettings->addMultiCellWidget(m_blackFrameListView, 1, 2, 0, 2);
    gridSettings->addMultiCellWidget(m_progressBar,        3, 3, 0, 2);

    m_imagePreviewWidget->setUserAreaWidget(gboxSettings);

    // -------------------------------------------------------------

    connect(m_filterMethodCombo, TQT_SIGNAL(activated(int)),
            this, TQT_SLOT(slotEffect()));

    connect(m_blackFrameButton, TQT_SIGNAL(clicked()),
            this, TQT_SLOT(slotAddBlackFrame()));

    connect(m_blackFrameListView, TQT_SIGNAL(blackFrameSelected(TQValueList<HotPixel>, const KURL&)),
            this, TQT_SLOT(slotBlackFrame(TQValueList<HotPixel>, const KURL&))); 
}

ImageEffect_HotPixels::~ImageEffect_HotPixels()
{
}

void ImageEffect_HotPixels::readUserSettings()
{
    TDEConfig *config = kapp->config();
    config->setGroup("hotpixels Tool Dialog");
    m_blackFrameURL = KURL(config->readEntry("Last Black Frame File", TQString()));
    m_filterMethodCombo->setCurrentItem(config->readNumEntry("Filter Method",
                                        HotPixelFixer::QUADRATIC_INTERPOLATION));

    if (m_blackFrameURL.isValid())
    {
        BlackFrameListViewItem *item = new BlackFrameListViewItem(m_blackFrameListView, m_blackFrameURL);

        connect(item, TQT_SIGNAL(signalLoadingProgress(float)),
                this, TQT_SLOT(slotLoadingProgress(float)));

        connect(item, TQT_SIGNAL(signalLoadingComplete()),
                this, TQT_SLOT(slotLoadingComplete()));
    }
}

void ImageEffect_HotPixels::writeUserSettings()
{
    TDEConfig *config = kapp->config();
    config->setGroup("hotpixels Tool Dialog");
    config->writeEntry("Last Black Frame File", m_blackFrameURL.url());
    config->writeEntry("Filter Method", m_filterMethodCombo->currentItem());
    config->sync();
}

void ImageEffect_HotPixels::resetValues()
{
    m_filterMethodCombo->blockSignals(true);
    m_filterMethodCombo->setCurrentItem(HotPixelFixer::QUADRATIC_INTERPOLATION);
    m_filterMethodCombo->blockSignals(false);
} 

void ImageEffect_HotPixels::slotAddBlackFrame()
{
    KURL url = Digikam::ImageDialog::getImageURL(this, m_blackFrameURL,
                                                 i18n("Select Black Frame Image"));

    if (!url.isEmpty())
    {
        // Load the selected file and insert into the list.

        m_blackFrameURL = url;
        m_blackFrameListView->clear();
        BlackFrameListViewItem *item = new BlackFrameListViewItem(m_blackFrameListView, m_blackFrameURL);

        connect(item, TQT_SIGNAL(signalLoadingProgress(float)),
                this, TQT_SLOT(slotLoadingProgress(float)));

        connect(item, TQT_SIGNAL(signalLoadingComplete()),
                this, TQT_SLOT(slotLoadingComplete()));
    }
}

void ImageEffect_HotPixels::slotLoadingProgress(float v)
{
    m_progressBar->show();
    m_progressBar->setValue((int)(v*100));
}

void ImageEffect_HotPixels::slotLoadingComplete()
{
    m_progressBar->hide();
}

void ImageEffect_HotPixels::renderingFinished()
{
    m_filterMethodCombo->setEnabled(true);
    m_blackFrameListView->setEnabled(true);
    enableButton(Apply, true);
}

void ImageEffect_HotPixels::prepareEffect()
{
    m_filterMethodCombo->setEnabled(false);
    m_blackFrameListView->setEnabled(false);
    enableButton(Apply, false);     

    Digikam::DImg image     = m_imagePreviewWidget->getOriginalRegionImage();
    int interpolationMethod = m_filterMethodCombo->currentItem();

    TQValueList<HotPixel> hotPixelsRegion;
    TQRect area = m_imagePreviewWidget->getOriginalImageRegionToRender();
    TQValueList<HotPixel>::Iterator end(m_hotPixelsList.end()); 
    
    for (TQValueList<HotPixel>::Iterator it = m_hotPixelsList.begin() ; it != end ; ++it )
    {
        HotPixel hp = (*it);
        
        if ( area.contains( hp.rect ) )
        {
           hp.rect.moveTopLeft(TQPoint( hp.rect.x()-area.x(), hp.rect.y()-area.y() ));
           hotPixelsRegion.append(hp);
        }
    }

    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>(
                       new HotPixelFixer(&image, this, hotPixelsRegion, interpolationMethod));
}

void ImageEffect_HotPixels::prepareFinal()
{
    m_filterMethodCombo->setEnabled(false);
    m_blackFrameListView->setEnabled(false);
    enableButton(Apply, false);     
        
    int interpolationMethod = m_filterMethodCombo->currentItem();

    Digikam::ImageIface iface(0, 0);
    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>(
                       new HotPixelFixer(iface.getOriginalImg(), this,m_hotPixelsList,interpolationMethod));
}

void ImageEffect_HotPixels::putPreviewData()
{
    m_imagePreviewWidget->setPreviewImage(m_threadedFilter->getTargetImage());
}

void ImageEffect_HotPixels::putFinalData()
{
    Digikam::ImageIface iface(0, 0);
    iface.putOriginalImage(i18n("Hot Pixels Correction"), m_threadedFilter->getTargetImage().bits());
}

void ImageEffect_HotPixels::slotBlackFrame(TQValueList<HotPixel> hpList, const KURL& blackFrameURL)
{
    m_blackFrameURL = blackFrameURL;
    m_hotPixelsList = hpList;
    
    TQPointArray pointList(m_hotPixelsList.size());
    TQValueList <HotPixel>::Iterator it;
    int i = 0;
    TQValueList <HotPixel>::Iterator end(m_hotPixelsList.end());
    
    for (it = m_hotPixelsList.begin() ; it != end ; ++it, i++)
       pointList.setPoint(i, (*it).rect.center());
        
    m_imagePreviewWidget->setPanIconHighLightPoints(pointList);
    
    slotEffect();
}

}  // NameSpace DigikamHotPixelsImagesPlugin
