/*
 *
 *  TrayIcon for tdebluez
 *
 *  Copyright (C) 2018  Emanoil Kotsev <deloptes@gmail.com>
 *
 *
 *  This file is part of tdebluez.
 *
 *  tdebluez is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  tdebluez is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with kbluetooth; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <objectmanagerImpl.h>
#include <adapterImpl.h>

#include "application.h"

#include "trayicon.h"

#define PIXMAP_LOGO "tdebluez"
#define HELP_DOCUMENT "tdebluez/components.html#components.TDEBluetooth"
#define PROCESS_KINPUTWIZARD "kinputwizard"
#define PROCESS_KBLUEMON "kbluemon"
#define PROCESS_KBLUELOCK "kbluelock"


using namespace TDEBluetooth;

TrayIcon::TrayIcon(TDEBluetoothApp* _app)
{
    app = _app;

    iconConnectingBlinkState = false;
    helpMenu = new KHelpMenu(this, TDEApplication::kApplication()->aboutData());

    adapterConfigDialog = new AdapterConfigDialog(app);
    deviceWizard = new DeviceWizard(app);

// alwaysVisible was part of the old code and is to be decided if we need it in the future
// the idea is that the tray icon may be invisible if desired
//	config->setGroup("UI");
//	alwaysVisible = config->readBoolEntry("alwaysVisible", true);

    bool agentstate = app->getStartAuthAgent();
    bool obexserver = app->getStartObex();
    bool autostart = app->getAutoStart();

//    blinkTimer = new TQTimer(this);
//    connect(blinkTimer, SIGNAL(timeout()), this, TQT_SLOT(slotSearchIcon()));

// This following commented out lines are also part of the old code
// They represent features that were not migrated to Bluez5
// KBlueMon
////	kbluemonAction = new TDEAction(i18n("KBlue&Mon"),
////		TDEGlobal::iconLoader()->loadIcon("info", TDEIcon::Small, 16),
////		TDEShortcut::null(), this, "kbluemon");
////	connect(kbluemonAction, SIGNAL(activated()), this, SLOT(slotkbluemonitor()));
////
////	// KBlueLock
////	kbluelockAction = new TDEAction(i18n("KBlue&Lock"),
////		TDEGlobal::iconLoader()->loadIcon("kbluelock", TDEIcon::Small, 16),
////		TDEShortcut::null(), this, "kbluelock");
////	connect(kbluelockAction, SIGNAL(activated()), this, SLOT(slotkbluelock()));
////
////	// KBtobexclient
////	kbtobexclientAction = new TDEAction(i18n("&Send File..."),
////		TDEGlobal::iconLoader()->loadIcon("attach", TDEIcon::Small, 16),
////		TDEShortcut::null(), this, "kbtobexclient");
////	connect(kbtobexclientAction, SIGNAL(activated()), this, SLOT(slotkbtobexclient()));
////
//    // BT switch
    bool anyPowered = false;
    TDEBluetoothApp::AdaptersMap::iterator it = app->adapters.begin();
    for (it; it != app->adapters.end(); ++it)
    {
        //      slotAddAdapter(it.key());
        TQT_DBusError dbuserr;
        bool powered = app->adapters[it.key()]->getPowered(dbuserr);
        if (dbuserr.isValid())
            tqDebug(i18n("Adapter getPowered failed: %1").arg(dbuserr.message()));
        TQString name = app->adapters[it.key()]->getAlias(dbuserr);
        if (dbuserr.isValid())
            tqDebug(i18n("Adapter getAlias failed: %1").arg(dbuserr.message()));

        TQString msg;
        if (powered) {
            anyPowered = powered;
            msg = i18n("Power off [%1]").arg(name);
        }
        else
        {
            msg = i18n("Power on [%1]").arg(name);
        }

        TDEToggleAction *ad_a = new TDEToggleAction(this, name.utf8());
        ad_a->setText(msg);
        ad_a->setChecked(powered);

        showPowerOnActionMap.insert(it.key(), ad_a);
        connect(ad_a, SIGNAL(toggled(bool)), this, TQT_SLOT(slotPowerOn(bool)));

    }

    // Configuration menu
    configActionMenu = new TDEActionMenu(i18n("&Configuration"), TDEGlobal::iconLoader()->loadIcon("configure", TDEIcon::Small, 16), this, "config_menu");

    // Help menu
    helpActionMenu = new TDEActionMenu(i18n("&Help"), TDEGlobal::iconLoader()->loadIcon("help", TDEIcon::Small, 16), this, "help_menu");

    // show documentation
    showHelpAction = KStdAction::help(this, TQT_SLOT(slotShowHelp()), actionCollection());

    // "About" menu item
    aboutAction = KStdAction::aboutApp(this, TQT_SLOT(slotAbout()), actionCollection());

//	// Inputdevice configuration
//	inputConfigAction = new TDEAction(i18n("&Input Devices..."),
//		TDEGlobal::iconLoader()->loadIcon("configure", TDEIcon::Small, 16),
//		TDEShortcut::null(), this, "input_config");
//	connect(inputConfigAction, SIGNAL(activated()), this, SLOT(slotInputConfig()));

    // Bluetooth Device configuration
    // from 06_rev748007.patch
    deviceConfigAction = new TDEAction(i18n("&Devices..."), TDEGlobal::iconLoader()->loadIcon("configure", TDEIcon::Small, 16), TDEShortcut::null(), this, "device_config");
    connect(deviceConfigAction, SIGNAL(activated()), this, TQT_SLOT(slotDeviceConfig()));

    // Bluetooth Adapter configuration
    adapterConfigAction = new TDEAction(i18n("&Adapters..."), TDEGlobal::iconLoader()->loadIcon("configure", TDEIcon::Small, 16), TDEShortcut::null(), this, "adapter_config");
    connect(adapterConfigAction, SIGNAL(activated()), this, TQT_SLOT(slotAdapterConfig()));

//	// Paired device configuration
//	pairedConfigAction = new TDEAction(i18n("&Paired/Trusted Devices..."),
//		TDEGlobal::iconLoader()->loadIcon("configure", TDEIcon::Small, 16),
//		TDEShortcut::null(), this, "paired_config");
//	connect(pairedConfigAction, SIGNAL(activated()), this, SLOT(slotPairedConfig()));

    // Autostart switch
    autostartIconAction = new TDEToggleAction(this, "autostart");
    connect(autostartIconAction, SIGNAL(toggled(bool)), this, TQT_SLOT(slotIconAutostartToggled(bool)));
    autostartIconAction->setChecked(autostart);
    autostartIconAction->setText(i18n("&Auto Start"));

    // Agent switch
    agentIconAction = new TDEToggleAction(this, "tdeauthagent");
    connect(agentIconAction, SIGNAL(toggled(bool)), this, TQT_SLOT(slotIconAgentToggled(bool)));
    agentIconAction->setChecked(agentstate);
    agentIconAction->setText(i18n("Auth A&gent"));

    // Obexsrv switch
    obexIconAction = new TDEToggleAction(this, "tdebtobexsrv");
    connect(obexIconAction, SIGNAL(toggled(bool)), this, TQT_SLOT(slotIconObexToggled(bool)));
    obexIconAction->setChecked(obexserver);
    obexIconAction->setText(i18n("&Obex Server"));

    slotIconAgentToggled(agentstate);
    slotIconObexToggled(obexserver);

//    // Always-visible switch
//    showIconAction = new TDEToggleAction(this, "always_visible");
//    connect(showIconAction, SIGNAL(toggled(bool)), this, SLOT(slotShowIconToggled(bool)));
//    showIconAction->setChecked(alwaysVisible);
//    showIconAction->setText(i18n("Always &Visible"));
//    configActionMenu->insert(showIconAction);

    KPixmap logoPixmap = TDEGlobal::iconLoader()->loadIcon(PIXMAP_LOGO, TDEIcon::Small, 22);
    iconIdle = logoPixmap;
    iconOff = logoPixmap;
    iconConnected = logoPixmap;
    iconNoAdapter = logoPixmap;
    KPixmapEffect::toGray(iconOff);
    KPixmapEffect::fade(iconOff, 0.3, TQColor(255, 255, 255));
    KPixmapEffect::fade(iconConnected, 0.5, TQColor(255, 0, 0));
    KPixmapEffect::toGray(iconNoAdapter);
    KPixmapEffect::fade(iconNoAdapter, 0.5, TQColor(128, 128, 128));

    setupTray(anyPowered);

    connect(app, SIGNAL(signalAdapterAdded(const TQString&)), this, TQT_SLOT(slotAddAdapter(const TQString&)));
    connect(app, SIGNAL(signalAdapterRemoved(const TQString&)), this, TQT_SLOT(slotRemoveAdapter(const TQString&)));

    connect(app->manager, SIGNAL(adapterPowerOnChanged(const TQString&, bool)), this, TQT_SLOT(slotPowerOnChanged(const TQString&, bool)));
    connect(app->manager, SIGNAL(adapterNameChanged(const TQString&, const TQString&)), this, TQT_SLOT(slotUpdateToolTip(const TQString&, const TQString&)));
    connect(app->manager, SIGNAL(adapterAliasChanged(const TQString&, const TQString&)), this, TQT_SLOT(slotUpdateToolTip(const TQString&, const TQString&)));
    connect(app->manager, SIGNAL(adapterNameChanged(const TQString&, const TQString&)), this, TQT_SLOT(slotAdapterAliasChanged(const TQString&, const TQString&)));
    connect(app->manager, SIGNAL(adapterAliasChanged(const TQString&, const TQString&)), this, TQT_SLOT(slotAdapterAliasChanged(const TQString&, const TQString&)));

//        connect(app->manager, SIGNAL(adapterModeChanged(const TQString&, const TQString&)), this, TQT_SLOT(slotUpdateToolTip(const TQString&, const TQString&)));
//        connect(app->manager, SIGNAL(adapterPowerOnChanged(const TQString&, bool)), this, TQT_SLOT(slotPowerOnChanged(const TQString&, bool)));
    connect(app->manager, SIGNAL(adapterDiscoverableTimeoutChanged(const TQString&, TQ_UINT32)),
            this, TQT_SLOT(slotDiscoverableTimeoutChanged(const TQString&, TQ_UINT32)));
    connect(app->manager, SIGNAL(adapterDiscoverableChanged(const TQString&, bool)),
            this, TQT_SLOT(slotDiscoverableChanged(const TQString&, bool)));
//        connect(app->manager, SIGNAL(adapterDiscoveringChanged(const TQString&, bool)), this, TQT_SLOT(slotDiscoveringChanged(const TQString&, bool)));
    connect(app->manager, SIGNAL(deviceConnectedChanged(const TQString&, bool)), this, TQT_SLOT(slotDeviceConnectedChanged(const TQString&, bool)));

    connect(adapterConfigDialog, SIGNAL(signalAdapterSelected(const TQString&, const TQString&)),
            deviceWizard, TQT_SLOT(slotSetAdapter(const TQString&, const TQString&)));
    connect(adapterConfigDialog, SIGNAL(signalAdapterSelected(const TQString&, const TQString&)),
            this, TQT_SLOT(slotUpdateToolTip(const TQString&, const TQString&)));

    connect(this, SIGNAL(quitSelected()), this, SLOT(slotQuitSelected()));

}

TrayIcon::~TrayIcon()
{

//    disconnect(app->manager, SIGNAL(deviceConnectedChanged(const TQString&, bool)), this, TQT_SLOT(slotDeviceConnectedChanged(const TQString&, bool)));
//    disconnect(app->manager, SIGNAL(adapterNameChanged(const TQString&, const TQString&)), this, TQT_SLOT(slotUpdateToolTip(const TQString&, const TQString&)));
//    disconnect(app->manager, SIGNAL(adapterAliasChanged(const TQString&, const TQString&)), this, TQT_SLOT(slotUpdateToolTip(const TQString&, const TQString&)));
////        disconnect(app->manager, SIGNAL(adapterModeChanged(const TQString&, const TQString&)), this, SLOT(slotUpdateToolTip(const TQString&, const TQString&)));
////        disconnect(app->manager, SIGNAL(adapterPowerOnChanged(const TQString&, bool)), this, TQT_SLOT(slotPowerOnChanged(const TQString&, bool)));
//    disconnect(app->manager, SIGNAL(adapterDiscoverableTimeoutChanged(const TQString&, TQ_UINT32)),
//            this, SLOT(slotDiscoverableTimeoutChanged(const TQString&, TQ_UINT32)));
//    disconnect(app->manager, SIGNAL(adapterDiscoverableChanged(const TQString&, bool)),
//            this, SLOT(slotDiscoverableChanged(const TQString&, bool)));
////        disconnect(app->manager, SIGNAL(adapterDiscoveringChanged(const TQString&, bool)), this, SLOT(slotDiscoveringChanged(const TQString&, bool)));

    if (deviceWizard)
        delete deviceWizard;
    if (adapterConfigDialog)
        delete adapterConfigDialog;

    delete obexIconAction;
    delete agentIconAction;
    delete autostartIconAction;
    delete adapterConfigAction;
    delete deviceConfigAction;
    delete helpActionMenu;
    delete configActionMenu;
    delete helpMenu;
}

void TrayIcon::setupTray(bool enable)
{

    // CONTEXT MENU

    //	if (!kbluemonAction->isPlugged())
    //	     kbluemonAction->plug(contextMenu());
    //	kbluemonAction->setEnabled(enable);
    //
    //	if (!kbluelockAction->isPlugged())
    //		kbluelockAction->plug(contextMenu());
    //	kbluelockAction->setEnabled(enable);
    //
    //	if (!kbtobexclientAction->isPlugged())
    //		kbtobexclientAction->plug(contextMenu());
    //	kbtobexclientAction->setEnabled(enable);

    TQMap<TQString, TDEToggleAction *>::iterator it = showPowerOnActionMap.begin();
    for (it; it != showPowerOnActionMap.end(); ++it)
    {
        if (!it.data()->isPlugged())
            it.data()->plug(contextMenu(), 1);
        // The toggle should always be active in order to power on and off
        it.data()->setEnabled(true);
    }

    // device wizard
    if (!deviceConfigAction->isPlugged())
        deviceConfigAction->plug(contextMenu());
    deviceConfigAction->setEnabled(enable);

    // adapter configuration
    if (!adapterConfigAction->isPlugged())
        adapterConfigAction->plug(contextMenu());
    adapterConfigAction->setEnabled(enable);

    // the config menu
    if (!configActionMenu->isPlugged())
        configActionMenu->plug(contextMenu());
    if (!showPowerOnActionMap.isEmpty())
        configActionMenu->setEnabled(true);
    else
        configActionMenu->setEnabled(false);

    // the help menu
    if (!helpActionMenu->isPlugged())
        helpActionMenu->plug(contextMenu());

    // the help sub menu
    if (!showHelpAction->isPlugged())
        showHelpAction->plug(helpActionMenu->popupMenu());

    // Report bug menu item
    /*
     TDEAction* reportBugAction = KStdAction::reportBug(this,
     SLOT(slotReportBug()), actionCollection());
     helpActionMenu->insert(reportBugAction);
     */

    // ABOUT MENU
    if (!aboutAction->isPlugged())
        aboutAction->plug(helpActionMenu->popupMenu());

    // Menu entries:
    // -------------

    //	// CONFIG MENU
    //	if (!inputConfigAction->isPlugged())
    //		inputConfigAction->plug(configActionMenu->popupMenu());

    // Service configuration
    /* TODO: create BlueZ service UI
     serviceConfigAction = new TDEAction(i18n("Configure &Services..."),
     TDEGlobal::iconLoader()->loadIcon("configure", TDEIcon::Small, 16),
     TDEShortcut::null(), this, "service_config");
     connect(serviceConfigAction, SIGNAL(activated()), this, SLOT(slotServiceConfig()));
     configActionMenu->insert(serviceConfigAction);
     */

    //	if (!pairedConfigAction->isPlugged())
    //		pairedConfigAction->plug(configActionMenu->popupMenu());
    // configure notifications
    /*
     TDEAction* notificationAction = KStdAction::configureNotifications(this,
     SLOT(slotConfigureNotifications()), actionCollection());
     configActionMenu->insert(notificationAction);
     */

    // Autostart option
    if (!autostartIconAction->isPlugged())
        autostartIconAction->plug(configActionMenu->popupMenu());

    // Agent option
    if (!agentIconAction->isPlugged())
        agentIconAction->plug(configActionMenu->popupMenu());

    //Obex option
    if (!obexIconAction->isPlugged())
        obexIconAction->plug(configActionMenu->popupMenu());

    //	if (!showIconAction->isPlugged())
    //		showIconAction->plug(configActionMenu->popupMenu());

    updateIcon();
    show();
}

void TrayIcon::updateIcon()
{
    TQString oldText = TQToolTip::textFor(this);
    if (showPowerOnActionMap.isEmpty())
    {
        setPixmap(iconNoAdapter);
        updateTooltip(i18n("No BT adapter"));
        if (!this->isHidden())
            KNotifyClient::event(TDEApplication::kApplication()->mainWidget()->winId(),
                "AdapterDetached", i18n("<b>No BT adapter available.</b><br/>(BT adapter removed)"));

        adapterConfigAction->setEnabled(false);
        deviceConfigAction->setEnabled(false);
        return;
    }

    TQT_DBusError dbuserr;
    bool anyPowered = false;
    TQString name;
    TQString path;
    TQMap<TQString, TDEToggleAction *>::iterator it = showPowerOnActionMap.begin();
    for (it; it != showPowerOnActionMap.end(); ++it)
    {
        anyPowered = app->adapters[it.key()]->getPowered(dbuserr);
        if (dbuserr.isValid())
            tqDebug(i18n("Adapter getPowered failed: %1").arg(dbuserr.message()));
        // know if any adapter is powered
        if (anyPowered)
        {
            path = app->adapters[it.key()]->getPath();
            name = app->adapters[it.key()]->getAlias(dbuserr);
            if (dbuserr.isValid())
                tqDebug(i18n("Adapter getAlias failed: %1").arg(dbuserr.message()));
            break;
        }
    }

    if (anyPowered)
    {
        bool anyConnected = false;
        TDEBluetoothApp::DevicesMap::Iterator dit = app->devices.begin();
        for (dit; dit != app->devices.end(); ++dit)
        {
            anyConnected = dit.data()->getConnected(dbuserr);
            if (dbuserr.isValid())
                tqDebug(i18n("Check for connected device failed: %1").arg(dbuserr.message()));
            if (anyConnected)
                break;
        }

        if (anyConnected)
            setPixmap(iconConnected);
        else
            setPixmap(iconIdle);
   }
    else
    {
        if (deviceWizard)
            deviceWizard->hide();

        if (adapterConfigDialog)
            adapterConfigDialog->hide();

        //if no other adapter was powered, use the first one for the tooltip
        path = app->adapters.begin().data()->getPath();
        name = app->adapters.begin().data()->getAlias(dbuserr);
        if (dbuserr.isValid())
            tqDebug(i18n("Adapter getAlias failed: %1").arg(dbuserr.message()));

        setPixmap(iconOff);
    }

    slotUpdateToolTip(path, name);

    adapterConfigAction->setEnabled(anyPowered);
    deviceConfigAction->setEnabled(anyPowered);
}

void TrayIcon::slotIconAutostartToggled(bool state)
{
    app->setAutoStart(state);
}

void TrayIcon::slotServiceConfig()
{
    kdDebug() << k_funcinfo << endl;
}

void TrayIcon::slotAdapterConfig()
{
    adapterConfigDialog->show();
    adapterConfigDialog->raise();
}

void TrayIcon::slotDeviceConfig()
{
    deviceWizard->show();
    deviceWizard->raise();
    deviceWizard->setFocus();
}

void TrayIcon::slotInputConfig()
{
    TDEProcess process(0);

    process << PROCESS_KINPUTWIZARD;
    if (!process.start(TDEProcess::DontCare))
    {
        KMessageBox::information(this, i18n("Could not execute kinputwizard."), i18n("TDEBluetooth"));
    }
}

void TrayIcon::slotkbluemonitor()
{
    TDEProcess process(0);
    process << PROCESS_KBLUEMON;

    if (!process.start(TDEProcess::DontCare))
    {
        KMessageBox::information(this, i18n("Could not execute KBlueMon."), i18n("TDEBluetooth"));
    }
}

void TrayIcon::slotkbluelock()
{
    TDEProcess process(0);
    process << PROCESS_KBLUELOCK;

    if (!process.start(TDEProcess::DontCare))
    {
        KMessageBox::information(this, i18n("Could not execute KBlueLock."), i18n("TDEBluetooth"));
    }
}

void TrayIcon::slotIconObexToggled(bool state)
{
    app->setStartObex(state);

    if (state)
    {
        if (!app->startObexSrv())
        {
            KMessageBox::information(this, i18n("Could not start OBEX server."), i18n("TDEBluetooth"));
        }
    }
    else
    {
        if (!app->stopObexSrv())
        {
            KMessageBox::information(this, i18n("Could not stop OBEX server."), i18n("TDEBluetooth"));
        }
    }
}

void TrayIcon::slotIconAgentToggled(bool state)
{
    app->setStartAuthAgent(state);

    if (state)
    {
        if (!app->startAuthAgent())
        {
            KMessageBox::information(this, i18n("Could not start TDEBluez Authentication Agent."), i18n("TDEBluetooth"));
        }
    }
    else
    {
        if (!app->stopAuthAgent())
        {
            KMessageBox::information(this, i18n("Could not stop TDEBluez Authentication Agent."), i18n("TDEBluetooth"));
        }
    }
}

void TrayIcon::slotPowerOn(bool state)
{
    // who send the signal ?
    const TQObject * o = TQObject::sender();
    const TDEToggleAction* obj = const_cast<TDEToggleAction*>(reinterpret_cast<const TDEToggleAction*>(o));
    TQMap<TQString, TDEToggleAction *>::Iterator it = showPowerOnActionMap.begin();

    TQString path;
    for (it; it != showPowerOnActionMap.end(); ++it)
    {
        if (obj == it.data())
        {
            path = it.key();
            break;
        }
    }

    app->adapters[path]->powerOn(state);
}

void TrayIcon::slotPowerOnChanged(const TQString &adapter, bool state)
{
    TQT_DBusError dbuserr;
    TQString name = app->adapters[adapter]->getAlias(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Adapter getAlias failed: %1").arg(dbuserr.message()));

    TQString addr = app->adapters[adapter]->getAddress(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Adapter getAddress failed: %1").arg(dbuserr.message()));

    if (name.isEmpty() && addr.isEmpty()) // adapter was removed
        return;

    TQString msg;
    if (state)
    {
        KNotifyClient::event(TDEApplication::kApplication()->mainWidget()->winId(),
                "AdapterAvailable", i18n("BT adapter <b>%1</b> is on<br/>(<b>%2</b>)").arg(name).arg(addr));
        msg = i18n("Power off [%1]").arg(name);
    }
    else
    {
        KNotifyClient::event(TDEApplication::kApplication()->mainWidget()->winId(),
                "AdapterAvailable", i18n("BT adapter <b>%1</b> is off<br/>(<b>%2</b>)").arg(name).arg(addr));
        msg = i18n("Power on [%1]").arg(name);
    }

    showPowerOnActionMap[adapter]->setText(msg);
    showPowerOnActionMap[adapter]->setChecked(state);
    slotUpdateToolTip(adapter, name);
    updateIcon();
}

void TrayIcon::slotDeviceConnectedChanged(const TQString& path, bool state)
{
    TQT_DBusError dbuserr;
    TQString name = app->devices[path]->getAlias(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Device getAlias failed: %1").arg(dbuserr.message()));
    TQString address = app->devices[path]->getAddress(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Adapter getAddress failed: %1").arg(dbuserr.message()));
    if (state)
    {
        KNotifyClient::event(TDEApplication::kApplication()->mainWidget()->winId(),
                "IncomingConnection", i18n("Connected to <b>%1</b> (<b>%2</b>)").arg(name).arg(address));
    }
    else
    {
        KNotifyClient::event(TDEApplication::kApplication()->mainWidget()->winId(),
                "IncomingConnection", i18n("Disconnected from <b>%1</b> (<b>%2</b>)").arg(name).arg(address));
    }
    updateIcon();
}

TQString TrayIcon::localAndEnglish(const TQCString& s)
{
    if (TQString(s) != i18n(s))
        return TQString("%1 (\"%2\")").arg(i18n(s)).arg(s);
    else
        return s;
}

void TrayIcon::mousePressEvent(TQMouseEvent *e)
{
    if (e->button() == TQMouseEvent::LeftButton)
    {
//        TQMessageBox::information(this,
//                i18n("Use right click to open the menu"),
//                i18n("Left click is reserved for ObexFTP browsing.\n"
//                "ObexFTP is subject of development."));
        		e->accept();
        		TDEProcess proc;
        		proc << "kfmclient" << "openURL" << "bluetooth:/";
        		proc.start(TDEProcess::DontCare);
    }
    else
    {
        KSystemTray::mousePressEvent(e);
    }
}

void TrayIcon::slotQuitSelected()
{
    // Ask if the user want to simply quit or disable
    // automatic start of TDEBluetooth
    if (!autostartIconAction->isChecked())
    {
        int autoStart = KMessageBox::questionYesNoCancel(0,
                i18n("Should TDEBluetooth still be restarted when you login?"),
                i18n("Automatically Start TDEBluetooth?"),
                i18n("Start"),
                i18n("Do Not Start"));

        if (autoStart == KMessageBox::Yes)
            app->setAutoStart(true);
        else if (autoStart == KMessageBox::No)
            app->setAutoStart(false);
    }
}

void TrayIcon::slotShowHelp()
{
    // TODO: This is surely not the correct way to jump to the
    // right chapter. Do I really have to mention the html-file,
    // or is the id enough?
    TDEApplication::kApplication()->invokeHelp("", HELP_DOCUMENT);
}

void TrayIcon::slotAddAdapter(const TQString& path)
{
    kdDebug() << k_funcinfo << endl; // for debugging MicheleC report
    TQT_DBusError dbuserr;

    bool powered = app->adapters[path]->getPowered(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Adapter getPowered failed: %1").arg(dbuserr.message()));

    TQString name = app->adapters[path]->getAlias(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Adapter getAlias failed: %1").arg(dbuserr.message()));

    TQString addr = app->adapters[path]->getAddress(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Adapter getAddress failed: %1").arg(dbuserr.message()));

    TQString msg;
    if (powered)
        msg = i18n("Power off [%1]").arg(name);
    else
        msg = i18n("Power on [%1]").arg(name);

    TDEToggleAction *ad_a = new TDEToggleAction(this, name.utf8());
    ad_a->setText(msg);
    ad_a->setChecked(powered);

    showPowerOnActionMap.insert(path, ad_a);
    connect(ad_a, SIGNAL(toggled(bool)), this, SLOT(slotPowerOn(bool)));

    KNotifyClient::event(TDEApplication::kApplication()->mainWidget()->winId(),
            "AdapterAttached", i18n("BT adapter <b>%1</b> attached<br/>(<b>%2</b>)").arg(name).arg(addr));

    if(!powered) // find out if we have some powered adapter
    {
        TQMap<TQString, TDEToggleAction *>::iterator it = showPowerOnActionMap.begin();
        for (it; it != showPowerOnActionMap.end(); ++it)
        {
            powered = app->adapters[it.key()]->getPowered(dbuserr);
            if (dbuserr.isValid())
                tqDebug(i18n("Adapter getPowered failed: %1").arg(dbuserr.message()));
            // know if any adapter is powered
            if (powered)
                break;
        }
    }

    if (obexIconAction->isChecked())
    {
        if (!app->obexServer->isRunning())
            app->startObexSrv();
    }

    if (agentIconAction->isChecked())
    {
        if (!app->authAgent->isRunning())
            app->startAuthAgent();
    }

    setupTray(powered);
}

void TrayIcon::slotRemoveAdapter(const TQString& path)
{
    kdDebug() << k_funcinfo << endl; // for debugging MicheleC report
    disconnect(showPowerOnActionMap[path], SIGNAL(toggled(bool)), this, SLOT(slotPowerOn(bool)));

    if (showPowerOnActionMap[path]->isPlugged())
        showPowerOnActionMap[path]->unplug(contextMenu());
    showPowerOnActionMap.remove(path);

    KNotifyClient::event(TDEApplication::kApplication()->mainWidget()->winId(),
            "AdapterDetached", i18n("BT adapter detached"));

    if (showPowerOnActionMap.isEmpty())
    {
        if (deviceWizard)
            deviceWizard->hide();
        if (adapterConfigDialog)
            adapterConfigDialog->hide();
        if (app->obexServer->isRunning())
            app->stopObexSrv();
        if (app->authAgent->isRunning())
            app->stopAuthAgent();
        setupTray(false);
    }
    else
    {
        setupTray(true);
    }
}

void TrayIcon::slotAdapterAliasChanged(const TQString &path, const TQString &alias)
{
    //      slotAddAdapter(it.key());
    TQT_DBusError dbuserr;
    bool powered = showPowerOnActionMap[path]->isChecked();
    TQString msg;
    if (powered)
        msg = i18n("Power off [%1]").arg(alias);
    else
        msg = i18n("Power on [%1]").arg(alias);
    showPowerOnActionMap[path]->setText(msg);
}

void TrayIcon::slotDiscoverableTimeoutChanged(const TQString &path, TQ_UINT32 timeout)
{
    TQT_DBusError dbuserr;
    TQString name = app->adapters[path]->getAlias(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Error: %1").arg(dbuserr.message()));
    slotUpdateToolTip(path, name);
}

void TrayIcon::slotDiscoverableChanged(const TQString &path, bool state)
{
    TQT_DBusError dbuserr;
    TQString name = app->adapters[path]->getAlias(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Error: %1").arg(dbuserr.message()));
    slotUpdateToolTip(path, name);
}

void TrayIcon::slotUpdateToolTip(const TQString& adapter, const TQString& name)
{
    TQT_DBusError dbuserr;
    TQString addr = app->adapters[adapter]->getAddress(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Error: %1").arg(dbuserr.message()));
    TQString disc =
            (app->adapters[adapter]->getDiscoverable(dbuserr)) ? "discoverable" : "hidden";
    if (dbuserr.isValid())
        tqDebug(i18n("Error: %1").arg(dbuserr.message()));

    TQString newText = i18n("Adapter: %1\nAddress: %2\nMode: %3\n").arg(name, addr, disc);
    updateTooltip(newText);
}

void TrayIcon::updateTooltip(const TQString &text)
{
    TQString oldText = TQToolTip::textFor(this);
    if (oldText != text)
    {
        TQToolTip::remove(this);
        TQToolTip::add(this, text);
    }
}

#include "trayicon.moc"
