/*
 *
 *  Adapter Manager Gui for tdebluez
 *
 *  Copyright (C) 2018  Emanoil Kotsev <deloptes@gmail.com>
 *
 *
 *  This file is part of tdebluez.
 *
 *  tdebluez is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  tdebluez is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with kbluetooth; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "adapterconfig.h"
#include "adapterconfigdialog.h"

AdapterConfigDialog::AdapterConfigDialog(TDEBluetoothApp *_app) :
        KDialogBase(NULL, "AdapterConfigDialog", true, "Adapter Configuration", (Ok)), app(_app), manager(_app->manager)
{
    nodevice = NULL;
    tabWidget = new TQTabWidget(this);

    ObjectManagerImpl::AdapterList list = manager->getAdapters();
    ObjectManagerImpl::AdapterList::iterator it;
    for (it = list.begin(); it != list.end(); ++it)
        addAdapter((*it));

    if (list.count() == 0)
    {
        nodevice = new TQLabel(i18n("No Bluetooth adapter found!"), this);
        tabWidget->addTab(nodevice, i18n("no adapter"));
        tabWidget->setMinimumSize(250, 300);
    }

    tabWidget->show();
    setMainWidget(tabWidget);
    setModal(false);

    connect(this, SIGNAL(okClicked()), this, TQT_SLOT(hide()));
    connect(manager, SIGNAL(adapterAdded(const TQString&)), TQT_SLOT(addAdapter(const TQString&)));
    connect(manager, SIGNAL(adapterRemoved(const TQString&)), TQT_SLOT(removeAdapter(const TQString&)));
    connect(tabWidget, SIGNAL(currentChanged(TQWidget *)), this, TQT_SLOT(slotCurrentChanged(TQWidget *)));
}

AdapterConfigDialog::~AdapterConfigDialog()
{
    close();
    if (nodevice)
        delete nodevice;
    delete tabWidget;
//    tabWidget = 0;
}

void AdapterConfigDialog::addAdapter(const TQString &path)
{
    AdapterConfig *aconfig = new AdapterConfig(app->manager, app->adapters[path]);
    tabWidget->addTab(aconfig->dialog(), aconfig->getName());

    TQT_DBusError dbuserr;
    bool powered = app->adapters[path]->getPowered(dbuserr);
    if (dbuserr.isValid())
        tqDebug(i18n("Adapter getPowered failed: %1").arg(dbuserr.message()));

    aconfig->dialog()->setEnabled(powered);
//
//    if (tabWidget->isTabEnabled(aconfig->dialog()))
//    {
        tabWidget->showPage(aconfig->dialog());
//    }
    adapterList.insert(path, aconfig);

    connect(aconfig->dialog()->adapterName, SIGNAL(textChanged(const TQString &)),
            this, TQT_SLOT(slotChangeName(const TQString &)));
    connect(aconfig->dialog()->adapterName, SIGNAL(textChanged(const TQString &)),
            app->adapters[path], TQT_SLOT(slotSetAlias(const TQString &)));

    if (nodevice)
    {
        tabWidget->removePage(nodevice);
        nodevice = 0;
    }
}

void AdapterConfigDialog::removeAdapter(const TQString &path)
{
    AdapterConfig *aconfig = adapterList[path];
    if (!aconfig)
        return;

    tabWidget->removePage(aconfig->dialog());
    delete adapterList[path];
    adapterList.remove(path);

    if (adapterList.count() == 0)
    {
        nodevice = new TQLabel(i18n("No Bluetooth device found!"), tabWidget);
        tabWidget->addTab(nodevice, i18n("no device"));
        if (tabWidget->isTabEnabled(nodevice))
        {
            tabWidget->showPage(nodevice);
        }
    }
}

void AdapterConfigDialog::slotChangeName(const TQString &name)
{
    tabWidget->changeTab(tabWidget->currentPage(), name);
}

void AdapterConfigDialog::slotCurrentChanged(TQWidget *widget)
{
    TQMap<TQString,AdapterConfig *>::iterator it;
    for (it = adapterList.begin(); it != adapterList.end(); ++it)
    {
        if (it.data()->dialog() == widget)
        {
            TQString path = it.key();
            TQT_DBusError dbuserr;
            TQString name = app->adapters[path]->getAlias(dbuserr);
            if (dbuserr.isValid())
                tqDebug(i18n("Adapter getAlias failed: %1").arg(dbuserr.message()));
//            kdDebug() << "Adapter changed: " << it.data()->dialog()->macLabel->text() << endl;
            emit signalAdapterSelected(path,name);
            break;
        }
    }
}

#include "adapterconfigdialog.moc"
