/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  shows all URLs found by the client
  begin:     Die Mai 27 2003
  copyright: (C) 2003 by Dario Abatianni
  email:     eisfuchs@tigress.com
*/

#include "urlcatcher.h"
#include "channel.h"
#include "server.h"
#include "konversationapplication.h"
#include "viewcontainer.h"

#include <tqhbox.h>
#include <tqpushbutton.h>
#include <tqregexp.h>
#include <tqclipboard.h>
#include <tqwhatsthis.h>
#include <tqlayout.h>

#include <tdeapplication.h>
#include <tdeactionclasses.h>
#include <tdelocale.h>
#include <kdebug.h>
#include <tdelistview.h>
#include <krun.h>
#include <tdefiledialog.h>
#include <kprocess.h>
#include <tdeversion.h>
#include <kshell.h>
#include <tdelistviewsearchline.h>


UrlCatcher::UrlCatcher(TQWidget* parent) : ChatWindow(parent)
{
    layout()->setAutoAdd(false);
    setName(i18n("URL Catcher"));
    setType(ChatWindow::UrlCatcher);

    urlListView=new TDEListView(this,"url_list_view");
    urlListView->addColumn(i18n("Nick"));
    urlListView->addColumn(i18n("URL"));
    urlListView->setFullWidth(true);
    urlListView->setAllColumnsShowFocus(true);
    TQString urlListViewWT = i18n(
        "List of Uniform Resource Locators mentioned in any of the Konversation windows "
        "during this session.");
    TQWhatsThis::add(urlListView, urlListViewWT);

    searchWidget = new TDEListViewSearchLineWidget(urlListView, this, "search_line");
    searchWidget->setEnabled(false);

    TQHBox* buttonBox=new TQHBox(this);
    buttonBox->setSpacing(spacing());

    openUrlButton=new TQPushButton(i18n("&Open URL"),buttonBox,"open_url_button");
    TQString openUrlButtonWT = i18n(
        "<p>Select a <b>URL</b> above, then click this button to launch the "
        "application associated with the mimetype of the URL.</p>"
        "<p>In the <b>Settings</b>, under <b>Behavior</b> | <b>General</b>, "
        "you can specify a custom web browser for web URLs.</p>");
    TQWhatsThis::add(openUrlButton, openUrlButtonWT);
    copyUrlButton=new TQPushButton(i18n("&Copy URL"),buttonBox,"copy_url_button");
    TQString copyUrlButtonWT = i18n(
        "Select a <b>URL</b> above, then click this button to copy the URL to the clipboard.");
    TQWhatsThis::add(copyUrlButton, copyUrlButtonWT);
    deleteUrlButton=new TQPushButton(i18n("&Delete URL"),buttonBox,"delete_url_button");
    TQString deleteUrlButtonWT = i18n(
        "Select a <b>URL</b> above, then click this button to delete the URL from the list.");
    TQWhatsThis::add(deleteUrlButton, deleteUrlButtonWT);
    saveListButton=new TQPushButton(i18n("Sa&ve List..."),buttonBox,"save_list_button");
    TQString saveListButtonWT = i18n(
        "Click to save the entire list to a file.");
    TQWhatsThis::add(saveListButton, saveListButtonWT);
    clearListButton=new TQPushButton(i18n("C&lear List"),buttonBox,"clear_list_button");
    TQString clearListButtonWT = i18n(
        "Click to erase the entire list.");
    TQWhatsThis::add(clearListButton, clearListButtonWT);

    connect(urlListView,TQT_SIGNAL (executed(TQListViewItem*)),this,TQT_SLOT (openUrl(TQListViewItem*)) );
    connect(urlListView,TQT_SIGNAL (selectionChanged()),this,TQT_SLOT (urlSelected()) );

    connect(openUrlButton,TQT_SIGNAL (clicked()),this,TQT_SLOT (openUrlClicked()) );
    connect(copyUrlButton,TQT_SIGNAL (clicked()),this,TQT_SLOT (copyUrlClicked()) );
    connect(deleteUrlButton,TQT_SIGNAL (clicked()),this,TQT_SLOT (deleteUrlClicked()) );
    connect(saveListButton,TQT_SIGNAL (clicked()),this,TQT_SLOT (saveListClicked()) );
    connect(clearListButton,TQT_SIGNAL (clicked()),this,TQT_SLOT (clearListClicked()) );

    saveListButton->setEnabled(false);
    clearListButton->setEnabled(false);

    layout()->add(searchWidget);
    layout()->add(urlListView);
    layout()->add(buttonBox);

    urlSelected();
}


UrlCatcher::~UrlCatcher()
{
}

void UrlCatcher::urlSelected()
{
    TQListViewItem* item=urlListView->selectedItem();
    if(item)
    {
        openUrlButton->setEnabled(true);
        copyUrlButton->setEnabled(true);
        deleteUrlButton->setEnabled(true);
    }
    else
    {
        openUrlButton->setEnabled(false);
        copyUrlButton->setEnabled(false);
        deleteUrlButton->setEnabled(false);
    }
}

void UrlCatcher::addUrl(const TQString& who,const TQString& url)
{
    new TDEListViewItem(urlListView,who,url);
    clearListButton->setEnabled(true);
    saveListButton->setEnabled(true);
    searchWidget->setEnabled(true);
}

void UrlCatcher::openUrl(TQListViewItem* item)
{
    TQString url = item->text(1);
    if (!Preferences::useCustomBrowser() || url.lower().startsWith("mailto:") )
    {
        new KRun(KURL(url));
    }
    else
    {
        TQString cmd = Preferences::webBrowserCmd();
        cmd.replace("%u", url);
        TDEProcess *proc = new TDEProcess;
        TQStringList cmdAndArgs = KShell::splitArgs(cmd);
        *proc << cmdAndArgs;
        //    This code will also work, but starts an extra shell process.
        //    kdDebug() << "UrlCatcher::openUrl(): cmd = " << cmd << endl;
        //    *proc << cmd;
        //    proc->setUseShell(true);
        proc->start(TDEProcess::DontCare);
        delete proc;
    }
}

void UrlCatcher::openUrlClicked()
{
    TQListViewItem* item=urlListView->selectedItem();
    if(item) openUrl(item);
}

void UrlCatcher::copyUrlClicked()
{
    TQListViewItem* item=urlListView->selectedItem();
    if(item)
    {
        TQClipboard *cb=TDEApplication::kApplication()->clipboard();
        cb->setText(item->text(1),TQClipboard::Selection);
        cb->setText(item->text(1),TQClipboard::Clipboard);
    }
}

void UrlCatcher::deleteUrlClicked()
{
    TQListViewItem* item=urlListView->selectedItem();
    if(item)
    {
        emit deleteUrl(item->text(0),item->text(1));
        delete item;
        // select next item
        item=urlListView->currentItem();
        if(item) urlListView->setSelected(item,true);
        else
        {
            saveListButton->setEnabled(false);
            clearListButton->setEnabled(false);
            searchWidget->setEnabled(false);
        }
    }
}

void UrlCatcher::clearListClicked()
{
    urlListView->clear();
    saveListButton->setEnabled(false);
    clearListButton->setEnabled(false);
    urlSelected();
    emit clearUrlList();
}

void UrlCatcher::saveListClicked()
{
    // Ask user for file name
    TQString fileName=KFileDialog::getSaveFileName(
        TQString(),
        TQString(),
        this,
        i18n("Save URL List"));

    if(!fileName.isEmpty())
    {
        // now save the list to disk
        TQFile listFile(fileName);
        listFile.open(IO_WriteOnly);
        // wrap the file into a stream
        TQTextStream stream(&listFile);
        TQListViewItem* item=urlListView->itemAtIndex(0);
        while(item)
        {
            stream << item->text(0) << ": " << item->text(1) << endl;
            item=item->itemBelow();
        }                                         // while
    }
}

void UrlCatcher::childAdjustFocus()
{
}

#include "urlcatcher.moc"
