/* This file is part of the KDE project
   Copyright (C) 2005 Thomas Zander <zander@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "KoCompletionDia.h"
#include "KoAutoFormat.h"

#include <tdelocale.h>
#include <tdemessagebox.h>
#include <kinputdialog.h>
#include <kcompletion.h>
#include <tdeconfig.h>
#include <kdebug.h>
#include <tqlayout.h>
#include <tqvbox.h>
#include <tqcheckbox.h>
#include <tqpushbutton.h>
#include <tqspinbox.h>
#include <tqcombobox.h>
#include <tqgroupbox.h>
#include <tqwhatsthis.h>

KoCompletionDia::KoCompletionDia( TQWidget *parent, const char *name, KoAutoFormat * autoFormat )
    : KDialogBase( parent, name , true, i18n( "Completion" ), Ok|Cancel|User1,
      Ok, true, KGuiItem( i18n( "&Reset" ), "edit-undo" ) )
{
    TQVBox *page = makeVBoxMainWidget();
    m_widget = new KoCompletion(page, autoFormat);
    m_widget->layout()->setMargin(0);
    connect( this, TQT_SIGNAL( user1Clicked() ), m_widget, TQT_SLOT(slotResetConf()));
    setButtonWhatsThis(Ok,i18n("This will save your options."));
    setButtonWhatsThis(Cancel,i18n("This will abort all changes."));
    setButtonWhatsThis(User1,i18n("This will reset to the state after you clicked on the Make Default button."));
}

void KoCompletionDia::slotOk()
{
    m_widget->saveSettings();
    KDialogBase::slotOk();
}

KoCompletion::KoCompletion(TQWidget *parent, KoAutoFormat *autoFormat) : KoCompletionBase(parent),
      m_autoFormat( *autoFormat ),
      m_docAutoFormat( autoFormat )
{
    connect(cbAllowCompletion, TQT_SIGNAL(toggled ( bool )), this, TQT_SLOT( changeButtonStatus()));
    TQStringList lst;
    lst << i18n( "Enter" );
    lst << i18n( "Tab" );
    lst << i18n( "Space" );
    lst << i18n( "End" );
    lst << i18n( "Right" );
    m_completionKeyAction->insertStringList( lst );

    connect( m_lbListCompletion, TQT_SIGNAL( selected ( const TQString & ) ), this, TQT_SLOT( slotCompletionWordSelected( const TQString & )));
    connect( m_lbListCompletion, TQT_SIGNAL( highlighted ( const TQString & ) ), this, TQT_SLOT( slotCompletionWordSelected( const TQString & )));

    connect( pbAddCompletionEntry, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotAddCompletionEntry()));
    connect( pbRemoveCompletionEntry, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotRemoveCompletionEntry()));
    connect( pbSaveCompletionEntry, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotSaveCompletionEntry()));

    slotResetConf(); // aka load config
    changeButtonStatus();
}

void KoCompletion::changeButtonStatus() {
    bool state = cbAllowCompletion->isChecked();

    completionBox->setEnabled( state);
    cbAddCompletionWord->setEnabled( state );
    pbAddCompletionEntry->setEnabled( state );
    m_lbListCompletion->setEnabled( state );
    state = state && (m_lbListCompletion->count()!=0 && !m_lbListCompletion->currentText().isEmpty());
    pbRemoveCompletionEntry->setEnabled( state );
}

void KoCompletion::slotResetConf() {
    cbAllowCompletion->setChecked( m_autoFormat.getConfigCompletion());
    cbShowToolTip->setChecked( m_autoFormat.getConfigToolTipCompletion());
    cbAddCompletionWord->setChecked( m_autoFormat.getConfigAddCompletionWord());
    m_lbListCompletion->clear();
    m_listCompletion = m_docAutoFormat->listCompletion();
    m_lbListCompletion->insertStringList( m_listCompletion );
    m_lbListCompletion->sort();
    if( m_listCompletion.isEmpty() || m_lbListCompletion->currentText().isEmpty())
        pbRemoveCompletionEntry->setEnabled( false );
    m_minWordLength->setValue ( m_docAutoFormat->getConfigMinWordLength() );
    m_maxNbWordCompletion->setValue ( m_docAutoFormat->getConfigNbMaxCompletionWord() );
    cbAppendSpace->setChecked( m_autoFormat.getConfigAppendSpace() );

    switch( m_docAutoFormat->getConfigKeyAction() )
    {
    case KoAutoFormat::Enter:
        m_completionKeyAction->setCurrentItem( 0 );
        break;
    case KoAutoFormat::Tab:
        m_completionKeyAction->setCurrentItem( 1 );
        break;
    case KoAutoFormat::Space:
        m_completionKeyAction->setCurrentItem( 2 );
        break;
    case KoAutoFormat::End:
        m_completionKeyAction->setCurrentItem( 3 );
        break;
    case KoAutoFormat::Right:
        m_completionKeyAction->setCurrentItem( 4 );
        break;
    default:
        m_completionKeyAction->setCurrentItem( 0 );
    }
    changeButtonStatus();
}

void KoCompletion::slotAddCompletionEntry() {
    bool ok;
    TQString const newWord = KInputDialog::getText( i18n("Add Completion Entry"), i18n("Enter entry:"), TQString(), &ok, this ).lower();
    if ( ok )
    {
        if ( !m_listCompletion.contains( newWord ))
        {
            m_listCompletion.append( newWord );
            m_lbListCompletion->insertItem( newWord );
            pbRemoveCompletionEntry->setEnabled( !m_lbListCompletion->currentText().isEmpty() );
            m_lbListCompletion->sort();
        }

    }
}

void KoCompletion::slotRemoveCompletionEntry() {
    TQString text = m_lbListCompletion->currentText();
    if( !text.isEmpty() )
    {
        m_listCompletion.remove( text );
        m_lbListCompletion->removeItem( m_lbListCompletion->currentItem () );
        if( m_lbListCompletion->count()==0 )
            pbRemoveCompletionEntry->setEnabled( false );
    }
}

void KoCompletion::slotCompletionWordSelected( const TQString & word) {
    pbRemoveCompletionEntry->setEnabled( !word.isEmpty() );
}

void KoCompletion::saveSettings() {
    m_docAutoFormat->configCompletion( cbAllowCompletion->isChecked());
    m_docAutoFormat->configToolTipCompletion( cbShowToolTip->isChecked());
    m_docAutoFormat->configAppendSpace( cbAppendSpace->isChecked() );
    m_docAutoFormat->configMinWordLength( m_minWordLength->value() );
    m_docAutoFormat->configNbMaxCompletionWord( m_maxNbWordCompletion->value () );
    m_docAutoFormat->configAddCompletionWord( cbAddCompletionWord->isChecked());

    m_docAutoFormat->getCompletion()->setItems( m_listCompletion );
    m_docAutoFormat->updateMaxWords();
    switch( m_completionKeyAction->currentItem() ) {
        case 1:
            m_docAutoFormat->configKeyCompletionAction( KoAutoFormat::Tab );
            break;
        case 2:
            m_docAutoFormat->configKeyCompletionAction( KoAutoFormat::Space );
            break;
        case 3:
            m_docAutoFormat->configKeyCompletionAction( KoAutoFormat::End );
            break;
        case 4:
            m_docAutoFormat->configKeyCompletionAction( KoAutoFormat::Right );
            break;
        case 0:
        default:
            m_docAutoFormat->configKeyCompletionAction( KoAutoFormat::Enter );
    }

    // Save to config file
    m_docAutoFormat->saveConfig();
}

void KoCompletion::slotSaveCompletionEntry() {
    TDEConfig config("kofficerc");
    TDEConfigGroupSaver cgs( &config, "Completion Word" );
    config.writeEntry( "list", m_listCompletion );
    config.sync();
    KMessageBox::information( this, i18n(
            "Completion list saved.\nIt will be used for all documents "
            "from now on."), i18n("Completion List Saved") );
}

#include "KoCompletionDia.moc"
