/*
 * Kivio - Visual Modelling and Flowcharting
 * Copyright (C) 2000 theKompany.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */


#include "tkcoloractions.h"
#include "ttdetoolbarbutton.h"

#include <tqlayout.h>
#include <kcolordialog.h>
#include <tdetoolbar.h>
#include <tqpainter.h>
#include <tqtooltip.h>
#include <tqwhatsthis.h>
#include <tdelocale.h>
#include <kdebug.h>
#include <tqapplication.h>

TKColorPopupMenu::TKColorPopupMenu( TQWidget* parent, const char* name )
: TDEPopupMenu(parent,name)
{
}

TKColorPopupMenu::~TKColorPopupMenu()
{
}

void TKColorPopupMenu::updateItemSize()
{
  styleChange(style());
}
/****************************************************************************************/
class TTDESelectColorActionPrivate
{
public:
    TTDESelectColorActionPrivate()
    {
    }
    bool defaultColorMenu;
    TQColor defaultColor;
};


TTDESelectColorAction::TTDESelectColorAction( const TQString& text, Type type, TQObject* parent, const char* name, bool menuDefaultColor )
: TTDEAction(parent,name)
{
    d=new TTDESelectColorActionPrivate();
    d->defaultColorMenu=menuDefaultColor;
    d->defaultColor=TQColor();
    setText(text);
    m_type = type;
    init();
}

TTDESelectColorAction::TTDESelectColorAction( const TQString& text, Type type,
                                          TQObject* receiver, const char* slot,
                                          TQObject* parent, const char* name, bool menuDefaultColor)
: TTDEAction(parent,name)
{
    d=new TTDESelectColorActionPrivate();
    d->defaultColorMenu=menuDefaultColor;
    d->defaultColor=TQColor();
    setText(text);
    m_type = type;
    connect( this, TQT_SIGNAL( activated() ), receiver, slot );
    init();
}

void TTDESelectColorAction::init()
{
  m_pStandardColor = new TKColorPanel();
  m_pRecentColor = new TKColorPanel();

  connect(m_pStandardColor,TQT_SIGNAL(colorSelected(const TQColor&)),TQT_SLOT(panelColorSelected(const TQColor&)));
  connect(m_pStandardColor,TQT_SIGNAL(reject()),TQT_SLOT(panelReject()));
  connect(m_pRecentColor,TQT_SIGNAL(colorSelected(const TQColor&)),TQT_SLOT(panelColorSelected(const TQColor&)));
  connect(m_pRecentColor,TQT_SIGNAL(reject()),TQT_SLOT(panelReject()));

  m_pRecentColor->clear();

  m_pMenu = new TKColorPopupMenu();
  m_pMenu->insertItem(m_pStandardColor);
  m_pMenu->insertSeparator();
  m_pMenu->insertItem(m_pRecentColor);
  m_pMenu->insertSeparator();

  switch (m_type) {
    case TextColor:
      m_pMenu->insertItem(i18n("More Text Colors..."),this,TQT_SLOT(selectColorDialog()));
      setCurrentColor(black);
      setIcon("textcolor");
      break;
    case LineColor:
      m_pMenu->insertItem(i18n("More Line Colors..."),this,TQT_SLOT(selectColorDialog()));
      setCurrentColor(black);
      setIcon("color_line");
      break;
    case FillColor:
      m_pMenu->insertItem(i18n("More Fill Colors..."),this,TQT_SLOT(selectColorDialog()));
      setCurrentColor(white);
      setIcon("color_fill");
      break;
    case Color:
      break;
  }
  if(d->defaultColorMenu)
  {
      m_pMenu->insertSeparator();
      m_pMenu->insertItem(i18n("Default Color"),this,TQT_SLOT(defaultColor()));
  }

  connect(m_pStandardColor,TQT_SIGNAL(sizeChanged()),m_pMenu,TQT_SLOT(updateItemSize()));
  connect(m_pRecentColor,TQT_SIGNAL(sizeChanged()),m_pMenu,TQT_SLOT(updateItemSize()));
}

TTDESelectColorAction::~TTDESelectColorAction()
{
  delete m_pMenu;
  delete d;
}

void TTDESelectColorAction::initToolBarButton(TTDEToolBarButton* b)
{
  TQWhatsThis::add( b, whatsThis() );
  TTDEAction::initToolBarButton(b);
  b->setDelayedPopup( popupMenu() );
  updatePixmap(b);
  updatePixmap();
}

void TTDESelectColorAction::defaultColor()
{
   m_pCurrentColor = d->defaultColor;
   emit activated();
}

void TTDESelectColorAction::setDefaultColor(const TQColor &_col)
{
    d->defaultColor=_col;
}

void TTDESelectColorAction::updatePixmap()
{
  for( int id = 0; id < containerCount(); ++id ) {
    TQWidget* w = container(id);
    if ( w->inherits("TDEToolBar") ) {
      TQWidget* r = static_cast<TDEToolBar*>(w)->getWidget(itemId(id));
      if ( r->inherits("TTDEToolBarButton") ) {
        updatePixmap(static_cast<TTDEToolBarButton*>(r));
      }
    }
    else if(w->inherits("TQPopupMenu") ) {
        TQPixmap pix =iconSet(TDEIcon::Small).pixmap(TQIconSet::Automatic,TQIconSet::Active);
  if ( pix.isNull() )
      return;
        TQPainter p(&pix);
        switch (m_type) {
            case TextColor:
                p.fillRect(TQRect(0,12,16,5), m_pCurrentColor);
                break;
            case LineColor:
                p.fillRect(TQRect(0,13,16,5), m_pCurrentColor);
                p.fillRect(TQRect(3,12,1,1), m_pCurrentColor);
                break;
            case FillColor:
                p.fillRect(TQRect(0,13,16,5), m_pCurrentColor);
                p.fillRect(TQRect(1,10,5,3), m_pCurrentColor);
                break;
            case Color:
                break;
        }
        p.end();
        setIconSet( pix );
    }
  }
}

void TTDESelectColorAction::updatePixmap(TTDEToolBarButton* b)
{
  if (!b)
    return;
  // Not much point in painting with an invalid color
  if (!m_pCurrentColor.isValid())
    return;
  TQPixmap pix =b->getActivePixmap();
  TQPainter p(&pix);
  switch (m_type) {
    case TextColor:
      p.fillRect(TQRect(0,12,16,5), m_pCurrentColor);
      break;
    case LineColor:
      p.fillRect(TQRect(0,13,16,5), m_pCurrentColor);
      p.fillRect(TQRect(3,12,1,1), m_pCurrentColor);
      break;
    case FillColor:
      p.fillRect(TQRect(0,13,16,5), m_pCurrentColor);
      p.fillRect(TQRect(1,10,5,3), m_pCurrentColor);
      break;
    case Color:
      break;
  }
  p.end();
  b->setPixmap(pix);
}

void TTDESelectColorAction::setCurrentColor( const TQColor& color )
{
    if ( color == m_pCurrentColor )
        return;
    m_pCurrentColor = color;
    setActiveColor( color );
    m_pRecentColor->setActiveColor(color );
  updatePixmap();
}

void TTDESelectColorAction::setActiveColor( const TQColor& color )
{
  m_pStandardColor->setActiveColor(color);
}

void TTDESelectColorAction::selectColorDialog()
{
    TQColor c = color();

    if ( d->defaultColorMenu )
    {
        if ( KColorDialog::getColor(c,d->defaultColor, TQT_TQWIDGET(tqApp->activeWindow())) == TQDialog::Accepted )
        {
            setCurrentColor(c);
            m_pRecentColor->insertColor(m_pCurrentColor);
            activate();
        }

    }
    else
    {
        if ( KColorDialog::getColor(c, TQT_TQWIDGET(tqApp->activeWindow())) == TQDialog::Accepted )
        {
            setCurrentColor(c);
            m_pRecentColor->insertColor(m_pCurrentColor);
            activate();
        }
    }
}

// Called when activating the menu item
void TTDESelectColorAction::slotActivated()
{
  //kdDebug() << "TTDESelectColorAction::slotActivated" << endl;
  selectColorDialog();
}

void TTDESelectColorAction::activate()
{
  emit colorSelected(m_pCurrentColor);
  emit activated();
}

void TTDESelectColorAction::panelColorSelected( const TQColor& color )
{
  m_pMenu->hide();
  setCurrentColor(color);

  activate();
}

void TTDESelectColorAction::panelReject()
{
  m_pMenu->hide();
}

class TKColorPanel::TKColorPanelPrivate
{
public:
  TKColorPanelPrivate()
  {
    panelCreated = false;
  }

  bool panelCreated;
};

/****************************************************************************************/
TKColorPanel::TKColorPanel( TQWidget* parent, const char* name )
: TQWidget(parent,name)
{
  d = new TKColorPanel::TKColorPanelPrivate();
  m_activeColor = black;

  //m_iX = 0;  // happens in setNumCols() -> resetGrid()
  //m_iY = 0;  // anyway, so...

  m_pLayout = 0L;
  setNumCols(15);
}

void TKColorPanel::setNumCols( int col )
{
  m_iWidth = col;
  resetGrid();

  TQDictIterator<TKColorPanelButton> it(m_pColorDict);
  while ( it.current() ) {
    addToGrid(it.current());
    ++it;
  }
}

TKColorPanel::~TKColorPanel()
{
  delete d;
}

void TKColorPanel::resetGrid()
{
  m_iX = 0;
  m_iY = 0;

  delete m_pLayout;
  m_pLayout = new TQGridLayout(this,0,m_iWidth+1,0,0);

  emit sizeChanged();
}

void TKColorPanel::clear()
{
  m_pColorDict.setAutoDelete(true);
  m_pColorDict.clear();
  m_pColorDict.setAutoDelete(false);
  d->panelCreated = true;  // we don't want to create the default
                           // panel anymore now (b/c of recent colors)
  resetGrid();
}

void TKColorPanel::insertColor( const TQColor& color, const TQString& text )
{
  if (m_pColorDict[color.name()])
    return;

  insertColor(color);
  TQToolTip::add(m_pColorDict[color.name()],text);
}

void TKColorPanel::insertColor( const TQColor& color )
{
  if (m_pColorDict[color.name()])
    return;

  m_pLayout->setMargin(3);
  TKColorPanelButton* f = new TKColorPanelButton(color,this);
  m_pColorDict.insert(color.name(),f);
  if ( m_activeColor == color )
      f->setActive(true);

  connect(f,TQT_SIGNAL(selected(const TQColor&)),TQT_SLOT(selected(const TQColor&)));

  addToGrid(f);
}

void TKColorPanel::addToGrid( TKColorPanelButton* f )
{
  m_pLayout->addWidget(f,m_iY,m_iX);
  f->show();  // yeehaaaw! ugly hack (Werner)
  m_iX++;
  if ( m_iX == m_iWidth ) {
    m_iX = 0;
    m_iY++;
  }
  emit sizeChanged();
}

void TKColorPanel::setActiveColor( const TQColor& color )
{
    TKColorPanelButton* b = m_pColorDict[m_activeColor.name()];
  if (b)
    b->setActive(false);

  m_activeColor = color;

  b = m_pColorDict[m_activeColor.name()];
  if (b)
    b->setActive(true);
}

void TKColorPanel::mouseReleaseEvent( TQMouseEvent* )
{
  reject();
}

void TKColorPanel::showEvent( TQShowEvent *e )
{
  if ( !d->panelCreated )
    fillPanel();
  TQWidget::showEvent(e);
}

void TKColorPanel::selected( const TQColor& color )
{
  emit colorSelected(color);
}

void TKColorPanel::fillPanel()
{
  d->panelCreated = true;
  blockSignals(true); // don't emit sizeChanged() all the time

  // ### TODO: names without space (e.g. red) are lower case in rgb.txt
  insertColor(TQColor( 255,   0,   0 ), i18n( "color", "Red"));
  insertColor(TQColor( 255, 165,   0 ), i18n( "color", "Orange"));
  insertColor(TQColor( 255,   0, 255 ), i18n( "color", "Magenta"));
  insertColor(TQColor(   0,   0, 255 ), i18n( "color", "Blue"));
  insertColor(TQColor(   0, 255, 255 ), i18n( "color", "Cyan"));
  insertColor(TQColor(   0, 255,   0 ), i18n( "color", "Green"));
  insertColor(TQColor( 255, 255,   0 ), i18n( "color", "Yellow"));
  insertColor(TQColor( 165,  42,  42 ), i18n( "color", "Brown"));
  insertColor(TQColor( 139,   0,   0 ), i18n( "color", "DarkRed"));
  insertColor(TQColor( 255, 140,   0 ), i18n( "color", "DarkOrange"));
  insertColor(TQColor( 139,   0, 139 ), i18n( "color", "DarkMagenta"));
  insertColor(TQColor(   0,   0, 139 ), i18n( "color", "DarkBlue"));
  insertColor(TQColor(   0, 139, 139 ), i18n( "color", "DarkCyan"));
  insertColor(TQColor(   0, 100,   0 ), i18n( "color", "DarkGreen"));
  insertColor(TQColor( 130, 127,   0 ), i18n( "color", "DarkYellow")); // ### not in rgb.txt

  insertColor(TQColor( 255, 255, 255 ), i18n( "color", "White"));
  insertColor(TQColor( 229, 229, 229 ), i18n( "color", "Gray 90%")); // ### not in rgb.txt
  insertColor(TQColor( 204, 204, 204 ), i18n( "color", "Gray 80%")); // ### not in rgb.txt
  insertColor(TQColor( 178, 178, 178 ), i18n( "color", "Gray 70%")); // ### not in rgb.txt
  insertColor(TQColor( 153, 153, 153 ), i18n( "color", "Gray 60%")); // ### not in rgb.txt
  insertColor(TQColor( 127, 127, 127 ), i18n( "color", "Gray 50%")); // ### not in rgb.txt
  insertColor(TQColor( 102, 102, 102 ), i18n( "color", "Gray 40%")); // ### not in rgb.txt
  insertColor(TQColor(  76,  76,  76 ), i18n( "color", "Gray 30%")); // ### not in rgb.txt
  insertColor(TQColor(  51,  51,  51 ), i18n( "color", "Gray 20%")); // ### not in rgb.txt
  insertColor(TQColor(  25,  25,  25 ), i18n( "color", "Gray 10%")); // ### not in rgb.txt
  insertColor(TQColor(   0,   0,   0 ), i18n( "color", "Black"));

  insertColor(TQColor( 255, 255, 240 ), i18n( "color", "Ivory"));
  insertColor(TQColor( 255, 250, 250 ), i18n( "color", "Snow"));
  insertColor(TQColor( 245, 255, 250 ), i18n( "color", "MintCream"));
  insertColor(TQColor( 255, 250, 240 ), i18n( "color", "FloralWhite"));
  insertColor(TQColor( 255, 255, 224 ), i18n( "color", "LightYellow"));
  insertColor(TQColor( 240, 255, 255 ), i18n( "color", "Azure"));
  insertColor(TQColor( 248, 248, 255 ), i18n( "color", "GhostWhite"));
  insertColor(TQColor( 240, 255, 240 ), i18n( "color", "Honeydew"));
  insertColor(TQColor( 255, 245, 238 ), i18n( "color", "Seashell"));
  insertColor(TQColor( 240, 248, 255 ), i18n( "color", "AliceBlue"));
  insertColor(TQColor( 255, 248, 220 ), i18n( "color", "Cornsilk"));
  insertColor(TQColor( 255, 240, 245 ), i18n( "color", "LavenderBlush"));
  insertColor(TQColor( 253, 245, 230 ), i18n( "color", "OldLace"));
  insertColor(TQColor( 245, 245, 245 ), i18n( "color", "WhiteSmoke"));
  insertColor(TQColor( 255, 250, 205 ), i18n( "color", "LemonChiffon"));
  insertColor(TQColor( 224, 255, 255 ), i18n( "color", "LightCyan"));
  insertColor(TQColor( 250, 250, 210 ), i18n( "color", "LightGoldenrodYellow"));
  insertColor(TQColor( 250, 240, 230 ), i18n( "color", "Linen"));
  insertColor(TQColor( 245, 245, 220 ), i18n( "color", "Beige"));
  insertColor(TQColor( 255, 239, 213 ), i18n( "color", "PapayaWhip"));
  insertColor(TQColor( 255, 235, 205 ), i18n( "color", "BlanchedAlmond"));
  insertColor(TQColor( 250, 235, 215 ), i18n( "color", "AntiqueWhite"));
  insertColor(TQColor( 255, 228, 225 ), i18n( "color", "MistyRose"));
  insertColor(TQColor( 230, 230, 250 ), i18n( "color", "Lavender"));
  insertColor(TQColor( 255, 228, 196 ), i18n( "color", "Bisque"));
  insertColor(TQColor( 255, 228, 181 ), i18n( "color", "Moccasin"));
  insertColor(TQColor( 255, 222, 173 ), i18n( "color", "NavajoWhite"));
  insertColor(TQColor( 255, 218, 185 ), i18n( "color", "PeachPuff"));
  insertColor(TQColor( 238, 232, 170 ), i18n( "color", "PaleGoldenrod"));
  insertColor(TQColor( 245, 222, 179 ), i18n( "color", "Wheat"));
  insertColor(TQColor( 220, 220, 220 ), i18n( "color", "Gainsboro"));
  insertColor(TQColor( 240, 230, 140 ), i18n( "color", "Khaki"));
  insertColor(TQColor( 175, 238, 238 ), i18n( "color", "PaleTurquoise"));
  insertColor(TQColor( 255, 192, 203 ), i18n( "color", "Pink"));
  insertColor(TQColor( 238, 221, 130 ), i18n( "color", "LightGoldenrod"));
  insertColor(TQColor( 211, 211, 211 ), i18n( "color", "LightGray"));
  insertColor(TQColor( 255, 182, 193 ), i18n( "color", "LightPink"));
  insertColor(TQColor( 176, 224, 230 ), i18n( "color", "PowderBlue"));
  insertColor(TQColor( 127, 255, 212 ), i18n( "color", "Aquamarine"));
  insertColor(TQColor( 216, 191, 216 ), i18n( "color", "Thistle"));
  insertColor(TQColor( 173, 216, 230 ), i18n( "color", "LightBlue"));
  insertColor(TQColor( 152, 251, 152 ), i18n( "color", "PaleGreen"));
  insertColor(TQColor( 255, 215,   0 ), i18n( "color", "Gold"));
  insertColor(TQColor( 173, 255,  47 ), i18n( "color", "GreenYellow"));
  insertColor(TQColor( 176, 196, 222 ), i18n( "color", "LightSteelBlue"));
  insertColor(TQColor( 144, 238, 144 ), i18n( "color", "LightGreen"));
  insertColor(TQColor( 221, 160, 221 ), i18n( "color", "Plum"));
  insertColor(TQColor( 190, 190, 190 ), i18n( "color", "Gray"));
  insertColor(TQColor( 222, 184, 135 ), i18n( "color", "BurlyWood"));
  insertColor(TQColor( 135, 206, 250 ), i18n( "color", "LightSkyblue"));
  insertColor(TQColor( 255, 160, 122 ), i18n( "color", "LightSalmon"));
  insertColor(TQColor( 135, 206, 235 ), i18n( "color", "SkyBlue"));
  insertColor(TQColor( 210, 180, 140 ), i18n( "color", "Tan"));
  insertColor(TQColor( 238, 130, 238 ), i18n( "color", "Violet"));
  insertColor(TQColor( 244, 164,  96 ), i18n( "color", "SandyBrown"));
  insertColor(TQColor( 233, 150, 122 ), i18n( "color", "DarkSalmon"));
  insertColor(TQColor( 189, 183, 107 ), i18n( "color", "DarkKhaki"));
  insertColor(TQColor( 127, 255,   0 ), i18n( "color", "Chartreuse"));
  insertColor(TQColor( 169, 169, 169 ), i18n( "color", "DarkGray"));
  insertColor(TQColor( 124, 252,   0 ), i18n( "color", "LawnGreen"));
  insertColor(TQColor( 255, 105, 180 ), i18n( "color", "HotPink"));
  insertColor(TQColor( 250, 128, 114 ), i18n( "color", "Salmon"));
  insertColor(TQColor( 240, 128, 128 ), i18n( "color", "LightCoral"));
  insertColor(TQColor(  64, 224, 208 ), i18n( "color", "Turquoise"));
  insertColor(TQColor( 143, 188, 143 ), i18n( "color", "DarkSeagreen"));
  insertColor(TQColor( 218, 112, 214 ), i18n( "color", "Orchid"));
  insertColor(TQColor( 102, 205, 170 ), i18n( "color", "MediumAquamarine"));
  insertColor(TQColor( 255, 127,  80 ), i18n( "color", "Coral"));
  insertColor(TQColor( 154, 205,  50 ), i18n( "color", "YellowGreen"));
  insertColor(TQColor( 218, 165,  32 ), i18n( "color", "Goldenrod"));
  insertColor(TQColor(  72, 209, 204 ), i18n( "color", "MediumTurquoise"));
  insertColor(TQColor( 188, 143, 143 ), i18n( "color", "RosyBrown"));
  insertColor(TQColor( 219, 112, 147 ), i18n( "color", "PaleVioletRed"));
  insertColor(TQColor(   0, 250, 154 ), i18n( "color", "MediumSpringGreen"));
  insertColor(TQColor( 255,  99,  71 ), i18n( "color", "Tomato"));
  insertColor(TQColor(   0, 255, 127 ), i18n( "color", "SpringGreen"));
  insertColor(TQColor( 205, 133,  63 ), i18n( "color", "Peru"));
  insertColor(TQColor( 100, 149, 237 ), i18n( "color", "CornflowerBlue"));
  insertColor(TQColor( 132, 112, 255 ), i18n( "color", "LightSlateBlue"));
  insertColor(TQColor( 147, 112, 219 ), i18n( "color", "MediumPurple"));
  insertColor(TQColor( 186,  85, 211 ), i18n( "color", "MediumOrchid"));
  insertColor(TQColor(  95, 158, 160 ), i18n( "color", "CadetBlue"));
  insertColor(TQColor(   0, 206, 209 ), i18n( "color", "DarkTurquoise"));
  insertColor(TQColor(   0, 191, 255 ), i18n( "color", "DeepSkyblue"));
  insertColor(TQColor( 119, 136, 153 ), i18n( "color", "LightSlateGray"));
  insertColor(TQColor( 184, 134,  11 ), i18n( "color", "DarkGoldenrod"));
  insertColor(TQColor( 123, 104, 238 ), i18n( "color", "MediumSlateBlue"));
  insertColor(TQColor( 205,  92,  92 ), i18n( "color", "IndianRed"));
  insertColor(TQColor( 210, 105,  30 ), i18n( "color", "Chocolate"));
  insertColor(TQColor(  60, 179, 113 ), i18n( "color", "MediumSeaGreen"));
  insertColor(TQColor(  50, 205,  50 ), i18n( "color", "LimeGreen"));
  insertColor(TQColor(  32, 178, 170 ), i18n( "color", "LightSeaGreen"));
  insertColor(TQColor( 112, 128, 144 ), i18n( "color", "SlateGray"));
  insertColor(TQColor(  30, 144, 255 ), i18n( "color", "DodgerBlue"));
  insertColor(TQColor( 255,  69,   0 ), i18n( "color", "OrangeRed"));
  insertColor(TQColor( 255,  20, 147 ), i18n( "color", "DeepPink"));
  insertColor(TQColor(  70, 130, 180 ), i18n( "color", "SteelBlue"));
  insertColor(TQColor( 106,  90, 205 ), i18n( "color", "SlateBlue"));
  insertColor(TQColor( 107, 142,  35 ), i18n( "color", "OliveDrab"));
  insertColor(TQColor(  65, 105, 225 ), i18n( "color", "RoyalBlue"));
  insertColor(TQColor( 208,  32, 144 ), i18n( "color", "VioletRed"));
  insertColor(TQColor( 153,  50, 204 ), i18n( "color", "DarkOrchid"));
  insertColor(TQColor( 160,  32, 240 ), i18n( "color", "Purple"));
  insertColor(TQColor( 105, 105, 105 ), i18n( "color", "DimGray"));
  insertColor(TQColor( 138,  43, 226 ), i18n( "color", "BlueViolet"));
  insertColor(TQColor( 160,  82,  45 ), i18n( "color", "Sienna"));
  insertColor(TQColor( 199,  21, 133 ), i18n( "color", "MediumVioletRed"));
  insertColor(TQColor( 176,  48,  96 ), i18n( "color", "Maroon"));
  insertColor(TQColor(  46, 139,  87 ), i18n( "color", "SeaGreen"));
  insertColor(TQColor(  85, 107,  47 ), i18n( "color", "DarkOliveGreen"));
  insertColor(TQColor(  34, 139,  34 ), i18n( "color", "ForestGreen"));
  insertColor(TQColor( 139,  69,  19 ), i18n( "color", "SaddleBrown"));
  insertColor(TQColor( 148,   0, 211 ), i18n( "color", "DarkViolet"));
  insertColor(TQColor( 178,  34,  34 ), i18n( "color", "FireBrick"));
  insertColor(TQColor(  72,  61, 139 ), i18n( "color", "DarkSlateBlue"));
  insertColor(TQColor(  47,  79,  79 ), i18n( "color", "DarkSlateGray"));
  insertColor(TQColor(  25,  25, 112 ), i18n( "color", "MidnightBlue"));
  insertColor(TQColor(   0,   0, 205 ), i18n( "color", "MediumBlue"));
  insertColor(TQColor(   0,   0, 128 ), i18n( "color", "Navy"));

  blockSignals(false);  // Signals allowed again
  emit sizeChanged();   // one call should be enough ;)
}

/****************************************************************************************/
TKColorPanelButton::TKColorPanelButton( const TQColor& color, TQWidget* parent, const char* name )
: TQFrame(parent,name), m_Color(color), m_bActive(false)
{
  setFixedSize(16,16);
  setFrameStyle( NoFrame );
}

TKColorPanelButton::~TKColorPanelButton()
{
}

void TKColorPanelButton::enterEvent( TQEvent* )
{
  if (!m_bActive)
    setFrameStyle( Panel | Sunken );
}

void TKColorPanelButton::leaveEvent( TQEvent* )
{
  if (!m_bActive)
    setFrameStyle( NoFrame );
}

void TKColorPanelButton::paintEvent( TQPaintEvent* ev )
{
  TQFrame::paintEvent(ev);

  TQPainter p(this,this);
  p.fillRect(2,2,12,12,m_Color);
  p.setPen(gray);
  p.drawRect(2,2,12,12);
  p.end();
}

void TKColorPanelButton::setActive( bool f )
{
  m_bActive = f;
  setFrameStyle( m_bActive ? Panel | Sunken : NoFrame );
}

void TKColorPanelButton::mouseReleaseEvent( TQMouseEvent* )
{
  emit selected(m_Color);
}
#include "tkcoloractions.moc"
