/* This file is part of the KDE project
   Copyright (C) 2003-2004 Peter Simonsson <psn@linux.se>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "kivioglobal.h"

#include <tqdom.h>
#include <tqpixmap.h>
#include <tqprinter.h>

#include <kdebug.h>
#include <tdeglobal.h>
#include <tdelocale.h>

#include <KoGenStyles.h>
#include <KoZoomHandler.h>
#include <KoGlobal.h>

#include "kivio_common.h"
#include "kivio_settings.h"
#include "kivio_stencil.h"
#include "kivio_intra_stencil_data.h"
#include "kivio_screen_painter.h"

struct PaperSizeDef {
  const char* title;
  float width;
  float height;
  int unit;
};

void Kivio::savePageLayout(TQDomElement& e, const KoPageLayout& layout)
{
  XmlWriteFloat(e, "width", layout.ptWidth);
  XmlWriteFloat(e, "height", layout.ptHeight);
  XmlWriteFloat(e, "marginLeft", layout.ptLeft);
  XmlWriteFloat(e, "marginRight", layout.ptRight);
  XmlWriteFloat(e, "marginTop", layout.ptTop);
  XmlWriteFloat(e, "marginBottom", layout.ptBottom);
  XmlWriteString(e, "format", KoPageFormat::formatString(layout.format));
  XmlWriteString(e, "orientation", Kivio::orientationString(layout.orientation));
}

// ## TODO: remove this method, now that layout.saveOasis() does it all
KoGenStyle Kivio::savePageLayout(const KoPageLayout& layout)
{
  return layout.saveOasis();
}

KoPageLayout Kivio::loadPageLayout(const TQDomElement& e)
{
  KoPageLayout layout;

  if(e.hasAttribute("unit")) {
    // Compatibility with Kivio <= 1.2.x
    kdDebug(43000) << "Compatibility mode..." << endl;
    KoUnit::Unit unit = Kivio::convToKoUnit(XmlReadInt(e, "unit", 0));
    layout.ptWidth = KoUnit::fromUserValue(XmlReadFloat(e, "width", 0.0), unit);
    layout.ptHeight = KoUnit::fromUserValue(XmlReadFloat(e, "height", 0.0), unit);
    layout.ptLeft = KoUnit::fromUserValue(XmlReadFloat(e, "marginLeft", 0.0), unit);
    layout.ptRight = KoUnit::fromUserValue(XmlReadFloat(e, "marginRight", 0.0), unit);
    layout.ptTop = KoUnit::fromUserValue(XmlReadFloat(e, "marginTop", 0.0), unit);
    layout.ptBottom = KoUnit::fromUserValue(XmlReadFloat(e, "marginBottom", 0.0), unit);
    Kivio::setFormatOrientation(layout);
    kdDebug(43000) << "Ready." << endl;
  } else {
    layout.ptWidth = XmlReadFloat(e, "width", 0.0);
    layout.ptHeight = XmlReadFloat(e, "height", 0.0);
    layout.ptLeft = XmlReadFloat(e, "marginLeft", 0.0);
    layout.ptRight = XmlReadFloat(e, "marginRight", 0.0);
    layout.ptTop = XmlReadFloat(e, "marginTop", 0.0);
    layout.ptBottom = XmlReadFloat(e, "marginBottom", 0.0);
    layout.format = KoPageFormat::formatFromString(XmlReadString(e, "format", pageSizeString(TDEGlobal::locale()->pageSize())));
    layout.orientation = Kivio::orientationFromString(XmlReadString(e, "orientation", "Portrait"));
  }

  return layout;
}

KoUnit::Unit Kivio::convToKoUnit(int tkUnit)
{
  switch(tkUnit) {
    case 0:
      return KoUnit::U_PT;
    case 1:
      return KoUnit::U_MM;
    case 2:
      return KoUnit::U_INCH;
    case 3:
      return KoUnit::U_PI;
    case 4:
      return KoUnit::U_CM;
    case 5:
      return KoUnit::U_DD;
    case 6:
      return KoUnit::U_CC;
  }

  return KoUnit::U_PT;
}

TQString Kivio::orientationString(KoOrientation o)
{
  if(o == PG_LANDSCAPE) {
    return "Landscape";
  }

  return "Portrait";
}

KoOrientation Kivio::orientationFromString(const TQString& s)
{
  if(s == "Landscape") {
    return PG_LANDSCAPE;
  }

  return PG_PORTRAIT;
}

void Kivio::setFormatOrientation(KoPageLayout& layout)
{
  const PaperSizeDef PapersTable[] = {
    {"A0",841,1189,KoUnit::U_MM},
    {"A1",594,841,KoUnit::U_MM},
    {"A2",420,594,KoUnit::U_MM},
    {"A3",297,420,KoUnit::U_MM},
    {"A4",210,297,KoUnit::U_MM},
    {"A5",148,210,KoUnit::U_MM},

    {"B4",250,353,KoUnit::U_MM},
    {"B6",125,176,KoUnit::U_MM},

    {"C4",229,324,KoUnit::U_MM},
    {"C5",162,229,KoUnit::U_MM},
    {"C5",114,162,KoUnit::U_MM},

    {"DL",110,220,KoUnit::U_MM},

    {"Letter",8.5,11,KoUnit::U_INCH},
    {"Legal",8.5,14,KoUnit::U_INCH},

    {"A",8.5,11,KoUnit::U_INCH},
    {"B",11,17,KoUnit::U_INCH},
    {"C",17,22,KoUnit::U_INCH},
    {"D",22,34,KoUnit::U_INCH},
    {"E",34,44,KoUnit::U_INCH},

    {"null",0,0,-2}     // -2 for end of list
  };
  int j = 0;
  PaperSizeDef sizeDef = PapersTable[0];
  bool found = false;
  float width, height;
  layout.format = PG_CUSTOM;
  layout.orientation = PG_PORTRAIT;

  while((sizeDef.unit != -2) && !found) {
    width = KoUnit::fromUserValue(sizeDef.width, static_cast<KoUnit::Unit>(sizeDef.unit));
    height = KoUnit::fromUserValue(sizeDef.height, static_cast<KoUnit::Unit>(sizeDef.unit));

    if((layout.ptWidth == width) && (layout.ptHeight == height)) {
      layout.format = KoPageFormat::formatFromString(sizeDef.title);
      layout.orientation = PG_PORTRAIT;
      found = true;
    } else if((layout.ptWidth == height) && (layout.ptHeight == width)) {
      layout.format = KoPageFormat::formatFromString(sizeDef.title);
      layout.orientation = PG_LANDSCAPE;
      found = true;
    }

    sizeDef = PapersTable[++j];
  }
}

KoSize Kivio::loadSize(const TQDomElement& e, const TQString& name, const KoSize& def)
{
  KoSize size;
  if(e.hasAttribute(name + "Unit")) {
    // Compatibility with Kivio <= 1.2.x
    KoUnit::Unit unit = Kivio::convToKoUnit(XmlReadInt(e, name + "Unit", 0));
    size.setWidth(KoUnit::fromUserValue(XmlReadFloat(e, name + "Width", def.width()), unit));
    size.setHeight(KoUnit::fromUserValue(XmlReadFloat(e, name + "Height", def.height()), unit));
  } else {
    size.setWidth(XmlReadFloat(e, name + "Width", def.width()));
    size.setHeight(XmlReadFloat(e, name + "Height", def.height()));
  }

  return size;
}

void Kivio::saveSize(TQDomElement& e, const TQString& name, const KoSize& size)
{
  XmlWriteFloat(e, name + "Width", size.width());
  XmlWriteFloat(e, name + "Height", size.height());
}

TQString Kivio::pageSizeString(int pageSize)
{
  TQString psStr = "A4";

  switch(pageSize) {
    case TQPrinter::A0:
      psStr = "A0";
      break;
    case TQPrinter::A1:
      psStr = "A1";
      break;
    case TQPrinter::A2:
      psStr = "A2";
      break;
    case TQPrinter::A3:
      psStr = "A3";
      break;
    case TQPrinter::A5:
      psStr = "A5";
      break;
    case TQPrinter::A6:
      psStr = "A6";
      break;
    case TQPrinter::A7:
      psStr = "A7";
      break;
    case TQPrinter::A8:
      psStr = "A8";
      break;
    case TQPrinter::A9:
      psStr = "A9";
      break;
    case TQPrinter::B0:
      psStr = "B0";
      break;
    case TQPrinter::B1:
      psStr = "B1";
      break;
    case TQPrinter::B10:
      psStr = "B10";
      break;
    case TQPrinter::B2:
      psStr = "B2";
      break;
    case TQPrinter::B3:
      psStr = "B3";
      break;
    case TQPrinter::B4:
      psStr = "B4";
      break;
    case TQPrinter::B5:
      psStr = "B5";
      break;
    case TQPrinter::B6:
      psStr = "B6";
      break;
    case TQPrinter::B7:
      psStr = "B7";
      break;
    case TQPrinter::B8:
      psStr = "B8";
      break;
    case TQPrinter::B9:
      psStr = "B9";
      break;
    case TQPrinter::C5E:
      psStr = "C5";
      break;
    case TQPrinter::Comm10E:
      psStr = "Comm10";
      break;
    case TQPrinter::DLE:
      psStr = "DL";
      break;
    case TQPrinter::Executive:
      psStr = "Executive";
      break;
    case TQPrinter::Folio:
      psStr = "Folio";
      break;
    case TQPrinter::Ledger:
      psStr = "Ledger";
      break;
    case TQPrinter::Legal:
      psStr = "Legal";
      break;
    case TQPrinter::Letter:
      psStr = "Letter";
      break;
    case TQPrinter::Tabloid:
      psStr = "Tabloid";
      break;
    case TQPrinter::Custom:
      psStr = "Custom";
      break;
    case TQPrinter::A4:
    default:
      psStr = "A4";
      break;
  };

  return psStr;
}

TQPixmap Kivio::arrowHeadPixmap()
{
  const char* lineends[] = {
    "50 272 2 1",
    "  c #000000",
    "X c #ffffff",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX     XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX        XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX           XXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX           XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX        XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX     XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X   XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX   XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX   XXXXXXX X",
    "X X                            XXXXXXXXXX  XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX   XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X   XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX    XXXX    XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX      XX      XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX                XXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX       X       XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX     XXX     XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX   XXXXX   XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX X  XXXX X  XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXX  XX XXX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXXXX   XXXXX  XXXXXX X",
    "X X                        XXXXXX  XXXXXX  XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXXX  X XXXX  XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XX  XXX XX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX   XXXXX   XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXX           XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXX               XXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXX              XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXX          XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX      XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXX  XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXX  XXXXXX X",
    "X X                        XXXXXXXXXXXXXX  XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXX  XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX  XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX  XXXXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX  XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"//,
    /*"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX    XXXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX    XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX    XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX      XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX        XXXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX     XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX    XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX      XXXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX      XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX     XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX      XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX        XXXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX        XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX      XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX     XXXXXXXX X",
    "X X                                       XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX     XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX    XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX    XXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX      XXX X",
    "X X                                          XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX      XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX    XXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXX X",
    "X X                                          XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",*/
    /*"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX XXXXXX X",
    "X X                                    XXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X  XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX  XXXXXX X",
    "X X                                XXXXXX  XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX  XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX  XXXXXX X",
    "X X                                  XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX  XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX  XXXXXX X",
    "X X                                XXXXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX  XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX  XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX  XXXXXXX X",
    "X X                                  XXXX  XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX  XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX XXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX XXX X",
    "X X                                     XXXX XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX XXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XXX X",
    "X X                                    XXXXX XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXXXXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXXXXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXX  XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX XXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX   XXXXXXXXX X",
    "X X                            XXXXX       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX   XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX XXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXX  XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX XXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XX XXXXXXXX X",
    "X X                            XXXXX       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX XXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXX XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX XXXXX XXXXX X",
    "X X                                  XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXX XXXXX XXXXX X",
    "X X                                  XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXX XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXX XXXXX XXXXX X",
    "X X                                  XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXX XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX  XXX  XXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXX  XXXXXXX  XXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXX  XXXXXXXXXXX  X XXXXX XXXXX X",
    "X X                 XXXXXXXXXXXXXXX  XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXX  XXXXXXXXXXX  X XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXX  XXXXXXX  XXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX  XXX  XXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX       XXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXX       XXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXX       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXX       XXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXX       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX  XXX  XXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXX  XXXXXXX  XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXX  XXXXXXXXXXX  X       XXXXX X",
    "X X                 XXXXXXXXXXXXXXX        XXXXX X",
    "X XXXXXXXXXXXXXXXXXX  XXXXXXXXXXX  X       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXX  XXXXXXX  XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX  XXX  XXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX    XXXX    XXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX      XX      XXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX                XXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX       X       XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX     XXX     XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX   XXXXX   XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXX       XXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXX       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXX X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXX       XXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXX       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXX X X XXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX  XXX  XXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXX  XXXXXXX  XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXX  XXXXXXXXXXX  X       XXXXX X",
    "X X                 XXXXXXXXXXXXXXX        XXXXX X",
    "X XXXXXXXXXXXXXXXXXX  XXXXXXXXXXX  X       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXX  XXXXXXX  XXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX  XXX  XXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XXXXX X",
    "X X                                  XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXX XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXX XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX       XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX     XXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX   XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXX  XXXXXX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX  XXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXX  XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
    "X                                                X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX XX  XXXXXX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX XXXX  XXXXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX XXXXXX  XXXXXX  XXXXXXX X",
    "X X                                        XXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX XXXXXX  XXXXXX  XXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX XXXX  XXXXXX  XXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXX XX  XXXXXX  XXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXX XXXXXXX XXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX X",
    "X                                                X",
    "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"*/
  };

  return TQPixmap(lineends);
}

TQPixmap Kivio::connectorTargetPixmap()
{
  const char * connectorTarget_xpm[] = {
  "7 7 3 1",
  "       c None",
  ".      c #FFFFFF",
  "+      c #051EFF",
  " .   . ",
  ".+. .+.",
  " .+.+. ",
  "  .+.  ",
  " .+.+. ",
  ".+. .+.",
  " .   . "};

  return TQPixmap(connectorTarget_xpm);
}

TQPixmap Kivio::lockPixmap()
{
  const char * lock_xpm[] = {
  "10 10 3 1",
  " 	c None",
  ".	c #FFFFFF",
  "+	c #000000",
  "    ..    ",
  "   .++.   ",
  "  .+..+.  ",
  "  .+..+.  ",
  " .++++++. ",
  " .+....+. ",
  " .+.  .+. ",
  " .+....+. ",
  " .++++++. ",
  "  ......  "};

  return TQPixmap(lock_xpm);
}

TQString Kivio::systemDefaultUnit()
{
  TQString defMS = "cm";

  if(TDEGlobal::locale()->measureSystem() == TDELocale::Imperial) {
    defMS = "in";
  }

  return defMS;
}

TQPixmap Kivio::generatePixmapFromStencil(int width, int height, KivioStencil* stencil)
{
  KoZoomHandler zoomHandler;
  TQRect rect = zoomHandler.zoomRect(stencil->rect());

  double zw = (double)(width - 2) / (double)rect.width();
  double zh = (double)(height - 2) / (double)rect.height();
  zoomHandler.setZoomAndResolution(tqRound(kMin(zw, zh) * 100.0), KoGlobal::dpiX(), KoGlobal::dpiY());
  rect = zoomHandler.zoomRect(stencil->rect());

  TQPixmap pix(width, height);
  pix.fill(TQt::white);
  KivioScreenPainter kpainter;
  kpainter.start(TQT_TQPAINTDEVICE(&pix));
  kpainter.translateBy(-rect.x() + ((width - rect.width()) / 2), -rect.y() + ((height - rect.height()) / 2));
  KivioIntraStencilData data;
  data.painter = &kpainter;
  data.zoomHandler = &zoomHandler;
  stencil->paint(&data);
  kpainter.stop();

  return pix;
}
