/*
    Copyright (C) 2000, S.R.Haque <shaheedhaque@hotmail.com>.
    This file is part of the KDE project

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.

DESCRIPTION

    This is a generic parser for Microsoft Office Drawings (MSODs). The
    specification for this is the Microsoft Office 97 Drawing File Format
    published in MSDN. The output is a series of callbacks (a.k.a. virtual
    functions) which the caller can override as required.
*/

#ifndef MSOD_H
#define MSOD_H

class TQString;
class TQPointArray;
#include <kwmf.h>
#include <tqptrvector.h>

class Msod :
    private KWmf
{
public:

    // Construction.

    Msod(
        unsigned dpi);
    virtual ~Msod();

    // Called to parse the given file. We extract a drawing by shapeId.
    // If the drawing is not found, the return value will be false.

    bool parse(
        unsigned shapeId,
        const TQString &file,
        const char *delayStream = 0L);
    bool parse(
        unsigned shapeId,
        TQDataStream &stream,
        unsigned size,
        const char *delayStream = 0L);

    typedef KWmf::DrawContext DrawContext;

    // Should be protected...

    void brushSet(
        unsigned colour,
        unsigned style);
    void penSet(
        unsigned colour,
        unsigned style,
        unsigned width);

protected:
    // Override to get results of parsing.

    virtual void gotEllipse(
        const DrawContext &dc,
        TQString type,
        TQPoint topLeft,
        TQSize halfAxes,
        unsigned startAngle,
        unsigned stopAngle) = 0;
    virtual void gotPicture(
        unsigned id,
        TQString extension,
        unsigned length,
        const char *data) = 0;
    virtual void gotPolygon(
        const DrawContext &dc,
        const TQPointArray &points) = 0;
    virtual void gotPolyline(
        const DrawContext &dc,
        const TQPointArray &points) = 0;
    virtual void gotRectangle(
        const DrawContext &dc,
        const TQPointArray &points) = 0;

private:
    Msod(const Msod &);
    const Msod &operator=(const Msod &);

    // Debug support.

    static const int s_area;

private:
    int m_dpi;
    DrawContext m_dc;
    unsigned m_dggError;
    unsigned m_requestedShapeId;
    bool m_isRequiredDrawing;
    const char *m_delayStream;
    struct
    {
        unsigned type;
        char *data;
        unsigned length;
    } m_shape;

    TQPoint normalisePoint(
        TQDataStream &operands);
    TQSize normaliseSize(
        TQDataStream &operands);
    void drawShape(
        unsigned shapeType,
        TQ_UINT32 bytes,
        TQDataStream &operands);

public:

    // Common Header (MSOBFH)
    typedef struct
    {
        union
        {
            TQ_UINT32 info;
            struct
            {
                TQ_UINT32 ver: 4;
                TQ_UINT32 inst: 12;
                TQ_UINT32 fbt: 16;
            } fields;
        } opcode;
        TQ_UINT32 cbLength;
    } Header;

private:
    typedef enum
    {
        msoblipERROR,               // An error occurred during loading.
        msoblipUNKNOWN,             // An unknown blip type.
        msoblipEMF,                 // Windows Enhanced Metafile.
        msoblipWMF,                 // Windows Metafile.
        msoblipPICT,                // Macintosh PICT.
        msoblipJPEG,                // JFIF.
        msoblipPNG,                 // PNG.
        msoblipDIB,                 // Windows DIB.
        msoblipFirstClient = 32,    // First client defined blip type.
        msoblipLastClient  = 255    // Last client defined blip type.
    } MSOBLIPTYPE;

    MSOBLIPTYPE m_blipType;
    unsigned m_imageId;
    class Image
    {
    public:
        TQString extension;
        unsigned length;
        const char *data;
        Image() { data = 0L; }
        ~Image() { delete [] data; }
    };
    TQPtrVector<Image> m_images;

    // Opcode handling and painter methods.

    void walk(
        TQ_UINT32 bytes,
        TQDataStream &operands);
    void skip(
        TQ_UINT32 bytes,
        TQDataStream &operands);
    void invokeHandler(
        Header &op,
        TQ_UINT32 bytes,
        TQDataStream &operands);

    void opAlignrule(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opAnchor(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opArcrule(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opBlip(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opBse(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opBstorecontainer(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opCalloutrule(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opChildanchor(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opClientanchor(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opClientdata(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opClientrule(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opClienttextbox(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opClsid(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opColormru(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opConnectorrule(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opDeletedpspl(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opDg(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opDgcontainer(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opDgg(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opDggcontainer(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opOleobject(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opOpt(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opRegroupitems(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opSelection(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opSolvercontainer(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opSp(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opSpcontainer(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opSpgr(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opSpgrcontainer(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opSplitmenucolors(Header &op, TQ_UINT32 bytes, TQDataStream &operands);
    void opTextbox(Header &op, TQ_UINT32 bytes, TQDataStream &operands);

    // Option handling.

    class Options
    {
    public:
        Options(Msod &parent);
        ~Options();
        void walk(
            TQ_UINT32 bytes,
            TQDataStream &operands);

        double m_rotation;

        TQ_UINT32 m_lTxid;

        TQ_UINT32 m_pib;
        TQString m_pibName;
        TQ_UINT32 m_pibFlags;
        TQ_UINT32 m_pictureId;
        bool m_fNoHitTestPicture;
        bool m_pictureGray;
        bool m_pictureBiLevel;
        bool m_pictureActive;

        TQ_UINT32 m_geoLeft;
        TQ_UINT32 m_geoTop;
        TQ_UINT32 m_geoRight;
        TQ_UINT32 m_geoBottom;
        TQ_UINT32 m_shapePath;
        TQPointArray *m_pVertices;
        bool m_fShadowOK;
        bool m_f3DOK;
        bool m_fLineOK;
        bool m_fGTextOK;
        bool m_fFillShadeShapeOK;
        bool m_fFillOK;

        bool m_fFilled;
        bool m_fHitTestFill;
        bool m_fillShape;
        bool m_fillUseRect;
        bool m_fNoFillHitTest;

        TQ_UINT32 m_lineColor;
        TQ_UINT32 m_lineBackColor;
        TQ_UINT32 m_lineType;
        TQ_UINT32 m_lineWidth;

        bool m_fArrowheadsOK;
        bool m_fLine;
        bool m_fHitTestLine;
        bool m_lineFillShape;
        bool m_fNoLineDrawDash;

        TQ_UINT32 m_bWMode;

        bool m_fOleIcon;
        bool m_fPreferRelativeResize;
        bool m_fLockShapeType;
        bool m_fDeleteAttachedObject;
        bool m_fBackground;

    private:
        Msod &m_parent;

        typedef struct
        {
            union
            {
                TQ_UINT16 info;
                struct
                {
                    TQ_UINT16 pid: 14;
                    TQ_UINT16 fBid: 1;
                    TQ_UINT16 fComplex: 1;
                } fields;
            } opcode;
            TQ_UINT32 value;
        } Header;

        void initialise();
        double from1616ToDouble(TQ_UINT32 value);
    };
    friend class Msod::Options;

    Options *m_opt;
};

#endif
