/*
 *  Copyright (c) 1999 Matthias Elter  <me@kde.org>
 *  Copyright (c) 2002, 2003 Patrick Julien <freak@codepimps.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef KIS_TOOL_H_
#define KIS_TOOL_H_

#include <tqobject.h>
#include <tqstring.h>

#include <ksharedptr.h>
#include <tdeaction.h>

#include "kis_shared_ptr_vector.h"
#include "kis_canvas_observer.h"

class TQCursor;
class TQEvent;
class TQKeyEvent;
class TQRect;
class TQWidget;
class TDEActionCollection;
class TDERadioAction;
class KDialog;
class KisBrush;
class KisGradient;
class KisPattern;
class KisButtonPressEvent;
class KisButtonReleaseEvent;
class KisDoubleClickEvent;
class KisMoveEvent;
class KisCanvasPainter;

enum enumToolType {
    TOOL_SHAPE = 0,   // Geometric shapes like ellipses and lines
    TOOL_FREEHAND = 1, // Freehand drawing tools
    TOOL_TRANSFORM = 2, // Tools that transform the layer
    TOOL_FILL = 3, // Tools that fill parts of the canvas
    TOOL_VIEW = 4,   // Tools that affect the canvas: pan, zoom, etc.
    TOOL_SELECT = 5

};

const TQ_UINT8 NUMBER_OF_TOOLTYPES = 6;

class KisTool : public TQObject, public KisCanvasObserver, public TDEShared {
    TQ_OBJECT
  

public:
    KisTool(const TQString & name);
    virtual ~KisTool();

public:

    virtual void paint(KisCanvasPainter& gc) = 0;
    virtual void paint(KisCanvasPainter& gc, const TQRect& rc) = 0;

    /**
     * This function is called after the creation of a tool to create the TDEAction corresponding
     * to the tool.
     *
     * The code should look like :
     * @code
     * 
     * @endcode
     */
    virtual void setup(TDEActionCollection *collection) = 0;

    virtual void buttonPress(KisButtonPressEvent *e) = 0;
    virtual void move(KisMoveEvent *e) = 0;
    virtual void buttonRelease(KisButtonReleaseEvent *e) = 0;
    virtual void doubleClick(KisDoubleClickEvent *e) = 0;
    virtual void keyPress(TQKeyEvent *e) = 0;
    virtual void keyRelease(TQKeyEvent *e) = 0;

    virtual TQCursor cursor() = 0;
    virtual void setCursor(const TQCursor& cursor) = 0;
    /**
     * This function is called to create the configuration widget of the tool.
     * @param parent the parent of the widget
     */
    virtual TQWidget* createOptionWidget(TQWidget* parent);
    /**
     * @return the current configuration widget.
     */
    virtual TQWidget* optionWidget();
    TDERadioAction *action() const { return m_action; }

    /**
     * Return true if this tool wants auto canvas-scrolling to 
     * work when this tool is active.
     */
    virtual bool wantsAutoScroll() const { return true; }

    // Methods for integration with karbon-style toolbox
    virtual TQ_UINT32 priority() { return 0; }
    virtual enumToolType toolType() { return TOOL_FREEHAND; }
    virtual TQString icon() { return m_action->icon(); }
    virtual TQString quickHelp() const { return ""; }

public slots:
    /**
     * This slot is called when the tool is selected in the toolbox
     */
    virtual void activate() = 0;
    
    /**
     * deactivate is called when the tool gets deactivated because another
     * tool is selected. Tools can then clean up after themselves.
     */
    virtual void deactivate() = 0;

private:
    KisTool(const KisTool&);
    KisTool& operator=(const KisTool&);

protected:
    TDERadioAction *m_action;
    bool m_ownAction;

private:
    class KisToolPrivate;
    KisToolPrivate * d;
    
};

#endif // KIS_TOOL_H_

