/***************************************************************************
    begin                : Wed Feb 6 2002
    copyright            : (C) 2002 - 2004 by Scott Wheeler
    email                : wheeler@kde.org
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <tdetoolbar.h>
#include <tdelocale.h>
#include <kiconloader.h>
#include <kdebug.h>

#include <tqtooltip.h>
#include <tqlayout.h>
#include <tqlabel.h>
#include <tqtimer.h>

#include "slideraction.h"

////////////////////////////////////////////////////////////////////////////////
// convenience class
////////////////////////////////////////////////////////////////////////////////

/**
 * This "custom" slider reverses the left and middle buttons.  Typically the
 * middle button "instantly" seeks rather than moving the slider towards the
 * click position in fixed intervals.  This behavior has now been mapped on
 * to the left mouse button.
 */

class TrackPositionSlider : public TQSlider
{
public:
    TrackPositionSlider(TQWidget *parent, const char *name) : TQSlider(parent, name)
    {
        setFocusPolicy(TQWidget::NoFocus);
    }

protected:
    virtual void mousePressEvent(TQMouseEvent *e)
    {
        if(e->button() == Qt::LeftButton) {
            TQMouseEvent reverse(TQEvent::MouseButtonPress, e->pos(), Qt::MidButton, e->state());
            TQSlider::mousePressEvent(&reverse);
            emit sliderPressed();
        }
        else if(e->button() == Qt::MidButton) {
            TQMouseEvent reverse(TQEvent::MouseButtonPress, e->pos(), Qt::LeftButton, e->state());
            TQSlider::mousePressEvent(&reverse);
        }
    }
};

////////////////////////////////////////////////////////////////////////////////
// VolumeSlider implementation
////////////////////////////////////////////////////////////////////////////////

VolumeSlider::VolumeSlider(Qt::Orientation o, TQWidget *parent, const char *name) :
    TQSlider(o, parent, name)
{
    connect(this, TQT_SIGNAL(valueChanged(int)), this, TQT_SLOT(slotValueChanged(int)));
}

void VolumeSlider::wheelEvent(TQWheelEvent *e)
{
    if(orientation() ==Qt::Horizontal) {
        TQWheelEvent transposed(e->pos(), -(e->delta()), e->state(), e->orientation());
        TQSlider::wheelEvent(&transposed);
    }
    else
        TQSlider::wheelEvent(e);
}

void VolumeSlider::focusInEvent(TQFocusEvent *)
{
    clearFocus();
}

int VolumeSlider::volume() const
{
    if(orientation() ==Qt::Horizontal)
        return value();
    else
        return maxValue() - value();    
}

void VolumeSlider::setVolume(int value)
{
    if(orientation() ==Qt::Horizontal)
        setValue(value);
    else
        setValue(maxValue() - value); 
}

void VolumeSlider::setOrientation(Qt::Orientation o)
{
    if(o == orientation())
        return;

    blockSignals(true);
    setValue(maxValue() - value());
    blockSignals(false);
    TQSlider::setOrientation(o);
}

void VolumeSlider::slotValueChanged(int value)
{
    if(orientation() ==Qt::Horizontal)
        emit signalVolumeChanged(value);
    else
        emit signalVolumeChanged(maxValue() - value);
}

////////////////////////////////////////////////////////////////////////////////
// public members
////////////////////////////////////////////////////////////////////////////////

const int SliderAction::minPosition = 0;
const int SliderAction::maxPosition = 1000;

SliderAction::SliderAction(const TQString &text, TQObject *parent, const char *name)
    : TDEAction(text, 0, parent, name),
      m_toolBar(0),
      m_layout(0),
      m_trackPositionSlider(0),
      m_volumeSlider(0),
      m_dragging(false),
      m_volumeDragging(false)
{

}

SliderAction::~SliderAction()
{

}

int SliderAction::plug(TQWidget *parent, int index)
{
    TQWidget *w = createWidget(parent);

    if(!w)
        return -1;

    // the check for null makes sure that there is only one toolbar that this is
    // "plugged" in to

    if(parent->inherits("TDEToolBar") && !m_toolBar) {
        m_toolBar = static_cast<TDEToolBar *>(parent);

        int id = TDEAction::getToolButtonID();

        m_toolBar->insertWidget(id, w->width(), w, index);

        addContainer(m_toolBar, id);

        connect(m_toolBar, TQT_SIGNAL(destroyed()), this, TQT_SLOT(slotToolbarDestroyed()));
        connect(m_toolBar, TQT_SIGNAL(orientationChanged(Qt::Orientation)),
                this, TQT_SLOT(slotUpdateOrientation()));
        connect(m_toolBar, TQT_SIGNAL(placeChanged(TQDockWindow::Place)),
                this, TQT_SLOT(slotUpdateOrientation()));

        slotUpdateOrientation();
        return (containerCount() - 1);
    }
    else
        slotUpdateOrientation();

    return -1;
}


void SliderAction::unplug(TQWidget *parent)
{
    if (parent->inherits("TDEToolBar")) {
        m_toolBar = static_cast<TDEToolBar *>(parent);

        int index = findContainer(m_toolBar);
        if (index != -1) {
            m_toolBar->removeItem(itemId(index));
            removeContainer(index);

            m_toolBar = 0;
        }
    }
}

////////////////////////////////////////////////////////////////////////////////
// public slots
////////////////////////////////////////////////////////////////////////////////

void SliderAction::slotUpdateOrientation()
{
    // if the toolbar is not null and either the dockWindow not defined or is the toolbar

    if(!m_toolBar)
        return;

    if(m_toolBar->barPos() == TDEToolBar::Right || m_toolBar->barPos() == TDEToolBar::Left) {
        m_trackPositionSlider->setOrientation(Qt::Vertical);
        m_volumeSlider->setOrientation(Qt::Vertical);
        m_layout->setDirection(TQBoxLayout::TopToBottom);
    }
    else {
        m_trackPositionSlider->setOrientation(Qt::Horizontal);
        m_volumeSlider->setOrientation(Qt::Horizontal);
        m_layout->setDirection(TQBoxLayout::LeftToRight);
    }
    slotUpdateSize();
}

////////////////////////////////////////////////////////////////////////////////
// private members
////////////////////////////////////////////////////////////////////////////////

TQWidget *SliderAction::createWidget(TQWidget *parent) // virtual -- used by base class
{
    if(parent) {
        TQWidget *base = new TQWidget(parent);
        base->setBackgroundMode(parent->backgroundMode());
        base->setName("tde toolbar widget");

        TDEToolBar *toolBar = dynamic_cast<TDEToolBar *>(parent);

        if(toolBar)
            toolBar->setStretchableWidget(base);

        Qt::Orientation orientation;

        if(toolBar && toolBar->barPos() == TDEToolBar::Right || toolBar->barPos() == TDEToolBar::Left)
            orientation =Qt::Vertical;
        else
            orientation =Qt::Horizontal;

        m_layout = new TQBoxLayout(base, TQBoxLayout::TopToBottom, 5, 5);

        m_layout->addItem(new TQSpacerItem(20, 1));

        TQLabel *trackPositionLabel = new TQLabel(base);
        trackPositionLabel->setName("tde toolbar widget");
        trackPositionLabel->setPixmap(SmallIcon("player_time"));
        TQToolTip::add(trackPositionLabel, i18n("Track position"));
        m_layout->addWidget(trackPositionLabel);

        m_trackPositionSlider = new TrackPositionSlider(base, "trackPositionSlider");
        m_trackPositionSlider->setMaxValue(maxPosition);
        TQToolTip::add(m_trackPositionSlider, i18n("Track position"));
        m_layout->addWidget(m_trackPositionSlider);
        connect(m_trackPositionSlider, TQT_SIGNAL(sliderPressed()), this, TQT_SLOT(slotSliderPressed()));
        connect(m_trackPositionSlider, TQT_SIGNAL(sliderReleased()), this, TQT_SLOT(slotSliderReleased()));

        m_layout->addItem(new TQSpacerItem(10, 1));

        TQLabel *volumeLabel = new TQLabel(base);
        volumeLabel->setName("tde toolbar widget");
        volumeLabel->setPixmap(SmallIcon("player_volume"));
        TQToolTip::add(volumeLabel, i18n("Volume"));
        m_layout->addWidget(volumeLabel);

        m_volumeSlider = new VolumeSlider(orientation, base, "volumeSlider");
        m_volumeSlider->setMaxValue(100);
        TQToolTip::add(m_volumeSlider, i18n("Volume"));
        m_layout->addWidget(m_volumeSlider);
        connect(m_volumeSlider, TQT_SIGNAL(signalVolumeChanged(int)), TQT_SIGNAL(signalVolumeChanged(int)));
        connect(m_volumeSlider, TQT_SIGNAL(sliderPressed()), this, TQT_SLOT(slotVolumeSliderPressed()));
        connect(m_volumeSlider, TQT_SIGNAL(sliderReleased()), this, TQT_SLOT(slotVolumeSliderReleased()));

        m_volumeSlider->setName("tde toolbar widget");
        m_trackPositionSlider->setName("tde toolbar widget");

        m_layout->setStretchFactor(m_trackPositionSlider, 4);
        m_layout->setStretchFactor(m_volumeSlider, 1);

        connect(parent, TQT_SIGNAL(modechange()), this, TQT_SLOT(slotUpdateSize()));

        return base;
    }
    else
        return 0;
}

////////////////////////////////////////////////////////////////////////////////
// private slots
////////////////////////////////////////////////////////////////////////////////

void SliderAction::slotUpdateSize()
{
    static const int offset = 3;
    static const int absoluteMax = 10000;

    if(!m_toolBar)
        return;

    if(m_toolBar->barPos() == TDEToolBar::Right || m_toolBar->barPos() == TDEToolBar::Left) {
        m_volumeSlider->setMaximumWidth(m_toolBar->iconSize() - offset);
        m_volumeSlider->setMaximumHeight(volumeMax);

        m_trackPositionSlider->setMaximumWidth(m_toolBar->iconSize() - offset);
        m_trackPositionSlider->setMaximumHeight(absoluteMax);
    }
    else {
        m_volumeSlider->setMaximumHeight(m_toolBar->iconSize() - offset);
        m_volumeSlider->setMaximumWidth(volumeMax);

        m_trackPositionSlider->setMaximumHeight(m_toolBar->iconSize() - offset);
        m_trackPositionSlider->setMaximumWidth(absoluteMax);
    }
}

void SliderAction::slotSliderPressed()
{
    m_dragging = true;
}

void SliderAction::slotSliderReleased()
{
    m_dragging = false;
    emit signalPositionChanged(m_trackPositionSlider->value());
}

void SliderAction::slotVolumeSliderPressed()
{
    m_volumeDragging = true;
}

void SliderAction::slotVolumeSliderReleased()
{
    m_volumeDragging = false;
    emit signalVolumeChanged(m_volumeSlider->value());
}

void SliderAction::slotToolbarDestroyed()
{
    int index = findContainer(m_toolBar);
    if(index != -1)
        removeContainer(index);

    m_toolBar = 0;

    // This is probably a leak, but this code path hardly ever occurs, and it's
    // too hard to debug correctly.

    m_trackPositionSlider = 0;
    m_volumeSlider = 0;
}

#include "slideraction.moc"
