/*
    This file is part of libkcal.

    Copyright (c) 2001-2003 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
#ifndef KCAL_TODO_H
#define KCAL_TODO_H

#include "incidence.h"

namespace KCal {

/**
  This class provides a Todo in the sense of RFC2445.
*/
class LIBKCAL_EXPORT Todo : public Incidence
{
  public:
    typedef ListBase<Todo> List;

    Todo();
    Todo( const Todo & );
    ~Todo();
    Todo& operator=( const Todo& );
    bool operator==( const Todo& ) const;

    TQCString type() const { return "Todo"; }

    /**
      Returns an exact copy of this todo. The returned object is owned by the
      caller.
    */
    Todo *clone();

    /**
      Sets due date and time.

      @param dtDue The due date/time.
      @param first Set the date of the first occurrence (if the todo is recurrent).
    */
    void setDtDue(const TQDateTime &dtDue, bool first = false);
    /**
      Returns due date and time.

      @param first If true and the todo recurs, the due date of the first
      occurrence will be returned.If false and recurrent, the date of the
      current occurrence will be returned. If non-recurrent, the normal due date
      will be returned.
    */
    TQDateTime dtDue( bool first = false ) const;
    /**
      Returns due time as string formatted according to the users locale
      settings.
      @deprecated use IncidenceFormatter::timeToString()
    */
    KDE_DEPRECATED TQString dtDueTimeStr() const;

    /**
      Returns due date as string formatted according to the users locale
      settings.

      @param shortfmt If set to true, use short date format, if set to false use
                      long format.
      @deprecated use IncidenceFormatter::dateToString()
    */
    KDE_DEPRECATED TQString dtDueDateStr( bool shortfmt = true ) const;

    /**
      Returns due date and time as string formatted according to the users locale
      settings.
      @deprecated use IncidenceFormatter::dateTimeToString()
    */
    KDE_DEPRECATED TQString dtDueStr() const;

    /**
      Returns true if the todo has a due date, otherwise return false.
    */
    bool hasDueDate() const;
    /**
      Set if the todo has a due date.

      @param hasDueDate true if todo has a due date, otherwise false
    */
    void setHasDueDate( bool hasDueDate );

    /**
      Returns true if the todo has a start date, otherwise return false.
    */
    bool hasStartDate() const;
    /**
      Set if the todo has a start date.

      @param hasStartDate true if todo has a start date, otherwise false
    */
    void setHasStartDate( bool hasStartDate );

    /**
      Returns the startdate of the todo.
      @param first If true, the startdate of the todo will be returned. If the
      todo recurs, the startdate of the first occurrence will be returned.
      If false and the todo recurs, the relative startdate will be returned,
      based on the date returned by dtRecurrence().
    */
    TQDateTime dtStart( bool first = false ) const;

    /**
      Sets the startdate of the todo.
    */
    void setDtStart( const TQDateTime &dtStart );

    /** Returns an todo's starting time as a string formatted according to the
     users locale settings.
     @param first If true, the startdate of the todo will be returned. If the
     todo recurs, the startdate of the first occurrence will be returned.
     If false and the todo recurs, the relative startdate will be returned,
     based on the date returned by dtRecurrence().
    */
    TQString dtStartTimeStr( bool first = false ) const;
    /** Returns an todo's starting date as a string formatted according to the
     users locale settings.
     @param shortfmt If true, use short date format, if set to false use
     long format.
     @param first If true, the startdate of the todo will be returned. If the
     todo recurs, the startdate of the first occurrence will be returned.
     If false and the todo recurs, the relative startdate will be returned,
     based on the date returned by dtRecurrence().
    */
    TQString dtStartDateStr( bool shortfmt = true, bool first = false ) const;
    /** Returns an todo's starting date and time as a string formatted according
     to the users locale settings.
     @param first If true, the startdate of the todo will be returned. If the
     todo recurs, the startdate of the first occurrence will be returned.
     If false and the todo recurs, the relative startdate will be returned,
     based on the date returned by dtRecurrence().
    */
    TQString dtStartStr( bool first = false ) const;

    /**
      Returns true if the todo is 100% completed, otherwise return false.
    */
    bool isCompleted() const;
    /**
      Set completed state.

      @param completed If true set completed state to 100%, if false set
                       completed state to 0%.
    */
    void setCompleted( bool completed );

    /**
      Returns how many percent of the task are completed. Returns a value
      between 0 and 100.
    */
    int percentComplete() const;
    /**
      Set how many percent of the task are completed. Valid values are in the
      range from 0 to 100.
    */
    void setPercentComplete( int );

    /**
      Returns date and time when todo was completed.
    */
    TQDateTime completed() const;
    /**
      Returns string contaiting date and time when the todo was completed
      formatted according to the users locale settings.
    */
    TQString completedStr() const;
    /**
      Set date and time of completion.
    */
    void setCompleted( const TQDateTime &completed );

    /**
      Returns true, if todo has a date associated with completion, otherwise
      return false.
    */
    bool hasCompletedDate() const;

    /**
      Sets the due date/time of the current occurrence if recurrent.
    */
    void setDtRecurrence( const TQDateTime &dt );

    /**
      Returns the due date/time of the current occurrence if recurrent.
    */
    TQDateTime dtRecurrence() const;

    /**
      Returns true if the date specified is one on which the todo will
      recur. Todo's are a special case, hence the overload. It adds an extra
      check, which make it return false if there's an occurrence between
      the recurstart and today.
    */
    virtual bool recursOn( const TQDate &date ) const;

    /**
      Returns true if this todo is overdue (e.g. due date is lower than today
      and not completed), else false.
     */
      bool isOverdue() const;

  protected:
    /** Return the end date/time of the base incidence. */
    virtual TQDateTime endDateRecurrenceBase() const { return dtDue(); }

  private:
    bool accept(Visitor &v) { return v.visit( this ); }

    /**
      * If the todo recurs, mDtRecurrence is set to the next occurrence
      * that's after today, mPercentComplete is set to 0 and true is returned.
      *
      * If the todo doesn't recur or if there aren't anymore occurrences
      * it just returns false.
      */
    bool recurTodo();

    TQDateTime mDtDue;                    // due date of todo
                                         // (first occurrence if recurrent)
    TQDateTime mDtRecurrence;             // due date of recurrence

    bool mHasDueDate;                    // if todo has associated due date
    bool mHasStartDate;                  // if todo has associated start date

    TQDateTime mCompleted;
    bool mHasCompletedDate;

    int mPercentComplete;

    class Private;
    Private *d;
};

}

#endif
