#include <stdlib.h>             // abs()

#include <tqlabel.h>
#include <tqlayout.h>
#include <tqlineedit.h>
#include <tqstring.h>
#include <tqvalidator.h>
#include <tqwidget.h>

#include <tdelocale.h>            // i18n
#include <tdeglobal.h>
#include "ktimewidget.h"

enum ValidatorType { HOUR, MINUTE };

class TimeValidator : public TQValidator
{
  public:
    TimeValidator( ValidatorType tp, TQWidget *parent=0, const char *name=0)
      : TQValidator(TQT_TQOBJECT(parent), name)
    {
      _tp = tp;
    }
    State validate(TQString &str, int &) const
    {
      if (str.isEmpty())
        return Acceptable;

      bool ok;
      int val = str.toInt( &ok );
      if ( ! ok )
        return Invalid;

      if ( _tp==MINUTE && val >= 60  )
        return Invalid;
      else
        return Acceptable;
    }

  public:
    ValidatorType _tp;
};


class KarmLineEdit : public TQLineEdit
{

  public:
    KarmLineEdit( TQWidget* parent, const char* name = 0 )
      : TQLineEdit( parent, name ) {}

protected:

  virtual void keyPressEvent( TQKeyEvent *event )
  {
    TQLineEdit::keyPressEvent( event );
    if ( text().length() == 2 && !event->text().isEmpty() )
      focusNextPrevChild(true);
  }
};


KArmTimeWidget::KArmTimeWidget( TQWidget* parent, const char* name )
  : TQWidget(parent, name)
{
  TQHBoxLayout *layout = new TQHBoxLayout(this);

  _hourLE = new TQLineEdit( this);
  // 9999 hours > 1 year!
  // 999 hours = 41 days  (That should be enough ...)
  _hourLE->setFixedWidth( fontMetrics().maxWidth() * 3
                          + 2 * _hourLE->frameWidth() + 2);
  layout->addWidget(_hourLE);
  TimeValidator *validator = new TimeValidator( HOUR, _hourLE,
                                                "Validator for _hourLE");
  _hourLE->setValidator( validator );
  _hourLE->setAlignment( TQt::AlignRight );


  TQLabel *hr = new TQLabel( i18n( "abbreviation for hours", " hr. " ), this );
  layout->addWidget( hr );

  _minuteLE = new KarmLineEdit(this);

  // Minutes lineedit: Make room for 2 digits
  _minuteLE->setFixedWidth( fontMetrics().maxWidth() * 2
                            + 2 * _minuteLE->frameWidth() + 2);
  layout->addWidget(_minuteLE);
  validator = new TimeValidator( MINUTE, _minuteLE, "Validator for _minuteLE");
  _minuteLE->setValidator( validator );
  _minuteLE->setMaxLength(2);
  _minuteLE->setAlignment( TQt::AlignRight );

  TQLabel *min = new TQLabel( i18n( "abbreviation for minutes", " min. " ), this );
  layout->addWidget( min );

  layout->addStretch(1);
  setFocusProxy( _hourLE );
}

void KArmTimeWidget::setTime( long minutes )
{
  TQString dummy;
  long hourpart = labs(minutes) / 60;
  long minutepart = labs(minutes) % 60;

  dummy.setNum( hourpart );
  if (minutes < 0)
    dummy = TDEGlobal::locale()->negativeSign() + dummy;
  _hourLE->setText( dummy );

  dummy.setNum( minutepart );
  if (minutepart < 10 ) {
    dummy = TQString::fromLatin1( "0" ) + dummy;
  }
  _minuteLE->setText( dummy );
}

long KArmTimeWidget::time() const
{
  bool ok, isNegative;
  int h, m;

  h = abs(_hourLE->text().toInt( &ok ));
  m = _minuteLE->text().toInt( &ok );
  isNegative = _hourLE->text().startsWith(TDEGlobal::locale()->negativeSign());

  return (h * 60 + m) * ((isNegative) ? -1 : 1);
}
