/**
 * kbookmarkmerger.cpp - Copyright (C) 2005 Frerich Raabe <raabe@kde.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <tdeaboutdata.h>
#include <tdeapplication.h>
#include <kbookmarkmanager.h>
#include <tdecmdlineargs.h>
#include <kdebug.h>
#include <kstandarddirs.h>

#include <dcopclient.h>

#include <tqdir.h>
#include <tqdom.h>
#include <tqfile.h>

#include <X11/Xlib.h>

static const TDECmdLineOptions cmdLineOptions[] =
{
	{ "+directory", I18N_NOOP( "Directory to scan for extra bookmarks" ), 0 },
	TDECmdLineLastOption
};

// The code for this function was taken from kdesktop/kcheckrunning.cpp
static bool kdeIsRunning()
{
	Display *dpy = XOpenDisplay( NULL );
	if ( !dpy ) {
		return false;
	}

	Atom atom = XInternAtom( dpy, "_KDE_RUNNING", False );
	return XGetSelectionOwner( dpy, atom ) != None;
}

int main( int argc, char**argv )
{
	const bool kdeRunning = kdeIsRunning();

	TDEAboutData aboutData( "kbookmarkmerger", I18N_NOOP( "KBookmarkMerger" ),
	                      "1.0", I18N_NOOP( "Merges bookmarks installed by 3rd parties into the user's bookmarks" ),
	                      TDEAboutData::License_BSD,
	                      I18N_NOOP(  "Copyright © 2005 Frerich Raabe" ) );
	aboutData.addAuthor( "Frerich Raabe", I18N_NOOP( "Original author" ),
	                     "raabe@kde.org" );

	TDECmdLineArgs::init( argc, argv, &aboutData );
	TDECmdLineArgs::addCmdLineOptions( cmdLineOptions );

	if ( !kdeRunning ) {
		TDEApplication::disableAutoDcopRegistration();
	}
	TDEApplication app( false, false );
	app.disableSessionManagement();

	TDECmdLineArgs *args = TDECmdLineArgs::parsedArgs();
	if ( args->count() != 1 ) {
		kdError() << "No directory to scan for bookmarks specified." << endl;
		return 1;
	}

	KBookmarkManager *konqBookmarks = KBookmarkManager::userBookmarksManager();
	TQStringList mergedFiles;
	{
		KBookmarkGroup root = konqBookmarks->root();
		for ( KBookmark bm = root.first(); !bm.isNull(); bm = root.next( bm ) ) {
			if ( bm.isGroup() ) {
				continue;
			}

			TQString mergedFrom = bm.metaDataItem( "merged_from" );
			if ( !mergedFrom.isNull() ) {
				mergedFiles << mergedFrom;
			}
		}
	}

	bool didMergeBookmark = false;

	TQString extraBookmarksDirName = TQFile::decodeName( args->arg( 0 ) );
	TQDir extraBookmarksDir( extraBookmarksDirName, "*.xml" );
	if ( !extraBookmarksDir.isReadable() ) {
		kdError() << "Failed to read files in directory " << extraBookmarksDirName << endl;
		return 1;
	}

	for ( unsigned int i = 0; i < extraBookmarksDir.count(); ++i ) {
		const TQString fileName = extraBookmarksDir[ i ];
		if ( mergedFiles.find( fileName ) != mergedFiles.end() ) {
			continue;
		}

		const TQString absPath = extraBookmarksDir.filePath( fileName );
		KBookmarkManager *mgr = KBookmarkManager::managerForFile( absPath, false );
		KBookmarkGroup root = mgr->root();
		for ( KBookmark bm = root.first(); !bm.isNull(); bm = root.next( bm ) ) {
			if ( bm.isGroup() ) {
				continue;
			}
			bm.setMetaDataItem( "merged_from", fileName );
			konqBookmarks->root().addBookmark( konqBookmarks, bm , false );
			didMergeBookmark = true;
		}
	}

	if ( didMergeBookmark ) {
		if ( !konqBookmarks->save() ) {
			kdError() << "Failed to write merged bookmarks." << endl;
			return 1;
		}
		if ( kdeRunning ) {
			konqBookmarks->notifyChanged( "" );
		}
	}
}

