/*
 *  kmacctcachedimap.h
 *
 *  Copyright (c) 2002-2004 Bo Thorsen <bo@sonofthor.dk>
 *  Copyright (c) 2002-2003 Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of this program with any edition of
 *  the TQt library by Trolltech AS, Norway (or with modified versions
 *  of TQt that use the same license as TQt), and distribute linked
 *  combinations including the two.  You must obey the GNU General
 *  Public License in all respects for all of the code used other than
 *  TQt.  If you modify this file, you may extend this exception to
 *  your version of the file, but you are not obligated to do so.  If
 *  you do not wish to do so, delete this exception statement from
 *  your version.
 */

#ifndef KMAcctCachedImap_h
#define KMAcctCachedImap_h

#include "imapaccountbase.h"

#include <tqguardedptr.h>

class KMFolderCachedImap;
class KMFolderTreeItem;
class KMFolder;
namespace KMail {
  class FolderJob;
  class ImapJob;
  class CachedImapJob;
}
using KMail::ImapJob;
using KMail::CachedImapJob;

namespace TDEIO {
  class Job;
}

class KMAcctCachedImap: public KMail::ImapAccountBase
{
  Q_OBJECT
  
  friend class ::KMail::ImapJob;
  friend class ::KMail::CachedImapJob;

public:
  virtual ~KMAcctCachedImap();
  virtual void init();

  /** A weak assignment operator */
  virtual void pseudoAssign( const KMAccount * a );

  /**
   * Overloaded to make sure it's never set for cached IMAP.
   */
  virtual void setAutoExpunge(bool);

  /**
   * Inherited methods.
   */
  virtual TQString type() const;
  virtual void processNewMail( bool interactive );

  /**
   * Kill the slave if any jobs are active
   */
  virtual void killAllJobs( bool disconnectSlave=false );

  /**
   * Abort running mail checks
   */
  virtual void cancelMailCheck();

  /**
   * Set the top level pseudo folder
   */
  virtual void setImapFolder(KMFolderCachedImap *);
  KMFolderCachedImap* imapFolder() const { return mFolder; }

  virtual void readConfig( /*const*/ TDEConfig/*Base*/ & config );
  virtual void writeConfig( TDEConfig/*Base*/ & config ) /*const*/;

  /**
   * Invalidate the local cache.
   */
  virtual void invalidateIMAPFolders();
  virtual void invalidateIMAPFolders( KMFolderCachedImap* );

  /**
   * Remember that a folder got explicitely deleted - including all child folders
   */
  void addDeletedFolder( KMFolder* folder );

  /**
   * Remember that a folder got explicitely deleted - NOT including all child folders
   * This is used when renaming a folder.
   */
  void addDeletedFolder( const TQString& imapPath );

  /**
   * Ask if a folder was explicitely deleted in this session
   */
  bool isDeletedFolder( const TQString& subFolderPath ) const;

  /**
   * Ask if a folder was explicitely deleted in a previous session
   */
  bool isPreviouslyDeletedFolder( const TQString& subFolderPath ) const;

  /**
   * return the imap path to the deleted folder, as well as the paths for any child folders
   */
  TQStringList deletedFolderPaths( const TQString& subFolderPath ) const;

  /**
   * Remove folder from the "deleted folders" list
   */
  void removeDeletedFolder( const TQString& subFolderPath );

  /**
   * Remember that a folder was renamed
   */
  void addRenamedFolder( const TQString& subFolderPath,
                         const TQString& oldLabel, const TQString& newName );

  /**
   * Remove folder from "renamed folders" list
   * Warning: @p subFolderPath is the OLD path
   */
  void removeRenamedFolder( const TQString& subFolderPath );

  struct RenamedFolder {
    RenamedFolder() {} // for TQMap
    RenamedFolder( const TQString& oldLabel, const TQString& newName )
      : mOldLabel( oldLabel ), mNewName( newName ) {}
    TQString mOldLabel;
    TQString mNewName;
  };

  /**
   * Returns new name for folder if it was renamed
   */
  TQString renamedFolder( const TQString& imapPath ) const;
  /**
   * Returns the list of folders that were renamed
   */
  const TQMap<TQString, RenamedFolder>& renamedFolders() const { return mRenamedFolders; }

  /**
   * Add a folder's unread count to the new "unread messages count", done during a sync after getting new mail
   */
  void addUnreadMsgCount( const KMFolderCachedImap *folder, int countUnread );

  /**
   * Add a folder's unread count to the last "unread messages count", i.e. the counts before getting new mail
   */
  void addLastUnreadMsgCount( const KMFolderCachedImap *folder,
                              int countLastUnread );

  /**
   * Returns the root folder of this account
   */
  virtual FolderStorage* rootFolder() const;
  
  /** return if the account passed the annotation test  */
  bool annotationCheckPassed(){ return mAnnotationCheckPassed;};
  void setAnnotationCheckPassed( bool a ){ mAnnotationCheckPassed = a; };

  /** Describes whether the account is a groupware account. */
  enum GroupwareType
  {
    GroupwareNone,     ///< Normal IMAP account
    GroupwareKolab,    ///< A Kolab groupware account
    GroupwareScalix    ///< A Scalix groupware account
  };

  void setGroupwareType( GroupwareType type ){ mGroupwareType = type; }
  GroupwareType groupwareType() const { return mGroupwareType; }

  void setSentCustomLoginCommand( bool value ){ mSentCustomLoginCommand = value; }
  bool sentCustomLoginCommand() const { return mSentCustomLoginCommand; }

protected:
  friend class ::AccountManager;
  KMAcctCachedImap(AccountManager* owner, const TQString& accountName, uint id);

protected slots:
  /** new-mail-notification for the current folder (is called via folderComplete) */
  void postProcessNewMail(KMFolderCachedImap*, bool);

  void slotProgressItemCanceled( KPIM::ProgressItem* );

  virtual void slotCheckQueuedFolders();

private:
  TQValueList<KMFolderCachedImap*> killAllJobsInternal( bool disconnectSlave );
  void processNewMail( KMFolderCachedImap* folder, bool recurse );

private:
  TQPtrList<CachedImapJob> mJobList;
  KMFolderCachedImap *mFolder;
  TQStringList mDeletedFolders; // folders deleted in this session
  TQStringList mPreviouslyDeletedFolders; // folders deleted in a previous session
  TQMap<TQString, RenamedFolder> mRenamedFolders;
  bool mAnnotationCheckPassed;

  GroupwareType mGroupwareType;
  bool mSentCustomLoginCommand;
};

#endif /*KMAcctCachedImap_h*/
