/*
    This file is part of KMail.
    Copyright (c) 2003 Andreas Gungl <a.gungl@gmx.de>

    KMail is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License, version 2, as
    published by the Free Software Foundation.

    KMail is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

    In addition, as a special exception, the copyright holders give
    permission to link the code of this program with any edition of
    the TQt library by Trolltech AS, Norway (or with modified versions
    of TQt that use the same license as TQt), and distribute linked
    combinations including the two.  You must obey the GNU General
    Public License in all respects for all of the code used other than
    TQt.  If you modify this file, you may extend this exception to
    your version of the file, but you are not obligated to do so.  If
    you do not wish to do so, delete this exception statement from
    your version.
*/

#include "filterlogdlg.h"
#include "filterlog.h"

#include <kdebug.h>
#include <tdeversion.h>
#include <tdefiledialog.h>
#include <tdelocale.h>
#include <tdemessagebox.h>

#include <tqcheckbox.h>
#include <tqlabel.h>
#include <tqspinbox.h>
#include <tqstringlist.h>
#include <tqtextedit.h>
#include <tqvbox.h>
#include <tqwhatsthis.h>
#include <tqvgroupbox.h>

#include <errno.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

using namespace KMail;


FilterLogDialog::FilterLogDialog( TQWidget * parent )
: KDialogBase( parent, "FilterLogDlg", false, i18n( "Filter Log Viewer" ),
              User1|User2|Close, Close, true, KStdGuiItem::clear(), KStdGuiItem::saveAs() )
{
  setWFlags( WDestructiveClose );
  TQVBox *page = makeVBoxMainWidget();

  mTextEdit = new TQTextEdit( page );
  mTextEdit->setReadOnly( true );
  mTextEdit->setWordWrap( TQTextEdit::NoWrap );
  mTextEdit->setTextFormat( TQTextEdit::LogText );

  TQStringList logEntries = FilterLog::instance()->getLogEntries();
  for ( TQStringList::Iterator it = logEntries.begin();
        it != logEntries.end(); ++it )
  {
    mTextEdit->append( *it );
  }

  mLogActiveBox = new TQCheckBox( i18n("&Log filter activities"), page );
  mLogActiveBox->setChecked( FilterLog::instance()->isLogging() );
  connect( mLogActiveBox, TQT_SIGNAL(clicked()),
           this, TQT_SLOT(slotSwitchLogState(void)) );
  TQWhatsThis::add( mLogActiveBox,
      i18n( "You can turn logging of filter activities on and off here. "
            "Of course, log data is collected and shown only when logging "
            "is turned on. " ) );

  mLogDetailsBox = new TQVGroupBox( i18n( "Logging Details" ), page );
  mLogDetailsBox->setEnabled( mLogActiveBox->isChecked() );
  connect( mLogActiveBox, TQT_SIGNAL( toggled( bool ) ),
           mLogDetailsBox, TQT_SLOT( setEnabled( bool ) ) );

  mLogPatternDescBox = new TQCheckBox( i18n("Log pattern description"),
                                      mLogDetailsBox );
  mLogPatternDescBox->setChecked(
      FilterLog::instance()->isContentTypeEnabled( FilterLog::patternDesc ) );
  connect( mLogPatternDescBox, TQT_SIGNAL(clicked()),
           this, TQT_SLOT(slotChangeLogDetail(void)) );
  // TODO
  //TQWhatsThis::add( mLogPatternDescBox,
  //    i18n( "" ) );

  mLogRuleEvaluationBox = new TQCheckBox( i18n("Log filter &rule evaluation"),
                                         mLogDetailsBox );
  mLogRuleEvaluationBox->setChecked(
      FilterLog::instance()->isContentTypeEnabled( FilterLog::ruleResult ) );
  connect( mLogRuleEvaluationBox, TQT_SIGNAL(clicked()),
           this, TQT_SLOT(slotChangeLogDetail(void)) );
  TQWhatsThis::add( mLogRuleEvaluationBox,
      i18n( "You can control the feedback in the log concerning the "
            "evaluation of the filter rules of applied filters: "
            "having this option checked will give detailed feedback "
            "for each single filter rule; alternatively, only "
            "feedback about the result of the evaluation of all rules "
            "of a single filter will be given." ) );

  mLogPatternResultBox = new TQCheckBox( i18n("Log filter pattern evaluation"),
                                        mLogDetailsBox );
  mLogPatternResultBox->setChecked(
      FilterLog::instance()->isContentTypeEnabled( FilterLog::patternResult ) );
  connect( mLogPatternResultBox, TQT_SIGNAL(clicked()),
           this, TQT_SLOT(slotChangeLogDetail(void)) );
  // TODO
  //TQWhatsThis::add( mLogPatternResultBox,
  //    i18n( "" ) );

  mLogFilterActionBox = new TQCheckBox( i18n("Log filter actions"),
                                       mLogDetailsBox );
  mLogFilterActionBox->setChecked(
      FilterLog::instance()->isContentTypeEnabled( FilterLog::appliedAction ) );
  connect( mLogFilterActionBox, TQT_SIGNAL(clicked()),
           this, TQT_SLOT(slotChangeLogDetail(void)) );
  // TODO
  //TQWhatsThis::add( mLogFilterActionBox,
  //    i18n( "" ) );

  TQHBox * hbox = new TQHBox( page );
  new TQLabel( i18n("Log size limit:"), hbox );
  mLogMemLimitSpin = new TQSpinBox( hbox );
  mLogMemLimitSpin->setMinValue( 1 );
  mLogMemLimitSpin->setMaxValue( 1024 * 256 ); // 256 MB
  // value in the TQSpinBox is in KB while it's in Byte in the FilterLog
  mLogMemLimitSpin->setValue( FilterLog::instance()->getMaxLogSize() / 1024 );
  mLogMemLimitSpin->setSuffix( " KB" );
  mLogMemLimitSpin->setSpecialValueText( i18n("unlimited") );
  connect( mLogMemLimitSpin, TQT_SIGNAL(valueChanged(int)),
           this, TQT_SLOT(slotChangeLogMemLimit(int)) );
  TQWhatsThis::add( mLogMemLimitSpin,
      i18n( "Collecting log data uses memory to temporarily store the "
	    "log data; here you can limit the maximum amount of memory "
	    "to be used: if the size of the collected log data exceeds "
	    "this limit then the oldest data will be discarded until "
	    "the limit is no longer exceeded. " ) );

  connect(FilterLog::instance(), TQT_SIGNAL(logEntryAdded(TQString)),
          this, TQT_SLOT(slotLogEntryAdded(TQString)));
  connect(FilterLog::instance(), TQT_SIGNAL(logShrinked(void)),
          this, TQT_SLOT(slotLogShrinked(void)));
  connect(FilterLog::instance(), TQT_SIGNAL(logStateChanged(void)),
          this, TQT_SLOT(slotLogStateChanged(void)));

  setInitialSize( TQSize( 500, 500 ) );
#if !KDE_IS_VERSION( 3, 2, 91 )
  // HACK - KWin keeps all dialogs on top of their mainwindows, but that's probably
  // wrong (#76026), and should be done only for modals. CVS HEAD should get
  // proper fix in KWin (see also searchwindow.cpp)
  XDeleteProperty( tqt_xdisplay(), winId(), XA_WM_TRANSIENT_FOR );
#endif
}


void FilterLogDialog::slotLogEntryAdded( TQString logEntry )
{
  mTextEdit->append( logEntry );
}


void FilterLogDialog::slotLogShrinked()
{
  // limit the size of the shown log lines as soon as
  // the log has reached it's memory limit
  if ( mTextEdit->maxLogLines() == -1 )
    mTextEdit->setMaxLogLines( mTextEdit->lines() );
}


void FilterLogDialog::slotLogStateChanged()
{
  mLogActiveBox->setChecked( FilterLog::instance()->isLogging() );
  mLogPatternDescBox->setChecked(
      FilterLog::instance()->isContentTypeEnabled( FilterLog::patternDesc ) );
  mLogRuleEvaluationBox->setChecked(
      FilterLog::instance()->isContentTypeEnabled( FilterLog::ruleResult ) );
  mLogPatternResultBox->setChecked(
      FilterLog::instance()->isContentTypeEnabled( FilterLog::patternResult ) );
  mLogFilterActionBox->setChecked(
      FilterLog::instance()->isContentTypeEnabled( FilterLog::appliedAction ) );

  // value in the TQSpinBox is in KB while it's in Byte in the FilterLog
  int newLogSize = FilterLog::instance()->getMaxLogSize() / 1024;
  if ( mLogMemLimitSpin->value() != newLogSize )
    mLogMemLimitSpin->setValue( newLogSize );
}


void FilterLogDialog::slotChangeLogDetail()
{
  if ( mLogPatternDescBox->isChecked() !=
       FilterLog::instance()->isContentTypeEnabled( FilterLog::patternDesc ) )
    FilterLog::instance()->setContentTypeEnabled( FilterLog::patternDesc,
                                                  mLogPatternDescBox->isChecked() );

  if ( mLogRuleEvaluationBox->isChecked() !=
       FilterLog::instance()->isContentTypeEnabled( FilterLog::ruleResult ) )
    FilterLog::instance()->setContentTypeEnabled( FilterLog::ruleResult,
                                                  mLogRuleEvaluationBox->isChecked() );

  if ( mLogPatternResultBox->isChecked() !=
       FilterLog::instance()->isContentTypeEnabled( FilterLog::patternResult ) )
    FilterLog::instance()->setContentTypeEnabled( FilterLog::patternResult,
                                                  mLogPatternResultBox->isChecked() );

  if ( mLogFilterActionBox->isChecked() !=
       FilterLog::instance()->isContentTypeEnabled( FilterLog::appliedAction ) )
    FilterLog::instance()->setContentTypeEnabled( FilterLog::appliedAction,
                                                  mLogFilterActionBox->isChecked() );
}


void FilterLogDialog::slotSwitchLogState()
{
  FilterLog::instance()->setLogging( mLogActiveBox->isChecked() );
}


void FilterLogDialog::slotChangeLogMemLimit( int value )
{
  FilterLog::instance()->setMaxLogSize( value * 1024 );
}


void FilterLogDialog::slotUser1()
{
  FilterLog::instance()->clear();
  mTextEdit->clear();
}


void FilterLogDialog::slotUser2()
{
  TQString fileName;
  KFileDialog fdlg( TQString(), TQString(), this, 0, true );

  fdlg.setMode( KFile::File );
  fdlg.setSelection( "kmail-filter.log" );
  fdlg.setOperationMode( KFileDialog::Saving );
  if ( fdlg.exec() )
  {
    fileName = fdlg.selectedFile();
    if ( !FilterLog::instance()->saveToFile( fileName ) )
    {
      KMessageBox::error( this,
                          i18n( "Could not write the file %1:\n"
                                "\"%2\" is the detailed error description." )
                          .arg( fileName,
                                TQString::fromLocal8Bit( strerror( errno ) ) ),
                          i18n( "KMail Error" ) );
    }
  }
}


#include "filterlogdlg.moc"
