/*
    dirservconfigpage.cpp

    This file is part of kleopatra
    Copyright (c) 2004 Klarlvdalens Datakonsult AB

    Libkleopatra is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License,
    version 2, as published by the Free Software Foundation.

    Libkleopatra is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

    In addition, as a special exception, the copyright holders give
    permission to link the code of this program with any edition of
    the TQt library by Trolltech AS, Norway (or with modified versions
    of TQt that use the same license as TQt), and distribute linked
    combinations including the two.  You must obey the GNU General
    Public License in all respects for all of the code used other than
    TQt.  If you modify this file, you may extend this exception to
    your version of the file, but you are not obligated to do so.  If
    you do not wish to do so, delete this exception statement from
    your version.
*/

#include "dirservconfigpage.h"
#include "directoryserviceswidget.h"

#include <kleo/cryptobackendfactory.h>

#include <tdemessagebox.h>
#include <tdelocale.h>
#include <kdebug.h>
#include <tdeconfig.h>
#include <knuminput.h>
#include <kdialog.h>

#include <tqhbox.h>
#include <tqlabel.h>
#include <tqdatetimeedit.h>
#include <tqcheckbox.h>
#include <tqlayout.h>

#include <kdemacros.h>

#if 0 // disabled, since it is apparently confusing
// For sync'ing kabldaprc
class KABSynchronizer
{
public:
  KABSynchronizer()
    : mConfig( "kabldaprc" ) {
    mConfig.setGroup( "LDAP" );
  }

  KURL::List readCurrentList() const {

    KURL::List lst;
    // stolen from tdeabc/ldapclient.cpp
    const uint numHosts = mConfig.readUnsignedNumEntry( "NumSelectedHosts" );
    for ( uint j = 0; j < numHosts; j++ ) {
      const TQString num = TQString::number( j );

      KURL url;
      url.setProtocol( "ldap" );
      url.setPath( "/" ); // workaround KURL parsing bug
      const TQString host = mConfig.readEntry( TQString( "SelectedHost" ) + num ).stripWhiteSpace();
      url.setHost( host );

      const int port = mConfig.readUnsignedNumEntry( TQString( "SelectedPort" ) + num );
      if ( port != 0 )
        url.setPort( port );

      const TQString base = mConfig.readEntry( TQString( "SelectedBase" ) + num ).stripWhiteSpace();
      url.setQuery( base );

      const TQString bindDN = mConfig.readEntry( TQString( "SelectedBind" ) + num ).stripWhiteSpace();
      url.setUser( bindDN );

      const TQString pwdBindDN = mConfig.readEntry( TQString( "SelectedPwdBind" ) + num ).stripWhiteSpace();
      url.setPass( pwdBindDN );
      lst.append( url );
    }
    return lst;
  }

  void writeList( const KURL::List& lst ) {

    mConfig.writeEntry( "NumSelectedHosts", lst.count() );

    KURL::List::const_iterator it = lst.begin();
    KURL::List::const_iterator end = lst.end();
    unsigned j = 0;
    for( ; it != end; ++it, ++j ) {
      const TQString num = TQString::number( j );
      KURL url = *it;

      Q_ASSERT( url.protocol() == "ldap" );
      mConfig.writeEntry( TQString( "SelectedHost" ) + num, url.host() );
      mConfig.writeEntry( TQString( "SelectedPort" ) + num, url.port() );

      // KURL automatically encoded the query (e.g. for spaces inside it),
      // so decode it before writing it out
      const TQString base = KURL::decode_string( url.query().mid(1) );
      mConfig.writeEntry( TQString( "SelectedBase" ) + num, base );
      mConfig.writeEntry( TQString( "SelectedBind" ) + num, url.user() );
      mConfig.writeEntry( TQString( "SelectedPwdBind" ) + num, url.pass() );
    }
    mConfig.sync();
  }

private:
  TDEConfig mConfig;
};

#endif

static const char s_dirserv_componentName[] = "dirmngr";
static const char s_dirserv_groupName[] = "LDAP";
static const char s_dirserv_entryName[] = "LDAP Server";

static const char s_timeout_componentName[] = "dirmngr";
static const char s_timeout_groupName[] = "LDAP";
static const char s_timeout_entryName[] = "ldaptimeout";

static const char s_maxitems_componentName[] = "dirmngr";
static const char s_maxitems_groupName[] = "LDAP";
static const char s_maxitems_entryName[] = "max-replies";

static const char s_addnewservers_componentName[] = "dirmngr";
static const char s_addnewservers_groupName[] = "LDAP";
static const char s_addnewservers_entryName[] = "add-servers";

DirectoryServicesConfigurationPage::DirectoryServicesConfigurationPage( TQWidget * parent, const char * name )
    : TDECModule( parent, name )
{
  mConfig = Kleo::CryptoBackendFactory::instance()->config();
  TQVBoxLayout* lay = new TQVBoxLayout( this, 0, KDialog::spacingHint() );
  Kleo::CryptoConfigEntry* entry = configEntry( s_dirserv_componentName, s_dirserv_groupName, s_dirserv_entryName,
                                                Kleo::CryptoConfigEntry::ArgType_LDAPURL, true );
  mWidget = new Kleo::DirectoryServicesWidget( entry, this );
  lay->addWidget( mWidget );
  connect( mWidget, TQT_SIGNAL( changed() ), this, TQT_SLOT( slotChanged() ) );

  // LDAP timeout
  TQHBox* box = new TQHBox( this );
  box->setSpacing( KDialog::spacingHint() );
  lay->addWidget( box );
  TQLabel* label = new TQLabel( i18n( "LDAP &timeout (minutes:seconds)" ), box );
  mTimeout = new TQTimeEdit( box );
  mTimeout->setDisplay( TQTimeEdit::Minutes | TQTimeEdit::Seconds );
  connect( mTimeout, TQT_SIGNAL( valueChanged( const TQTime& ) ), this, TQT_SLOT( slotChanged() ) );
  label->setBuddy( mTimeout );
  TQWidget* stretch = new TQWidget( box );
  box->setStretchFactor( stretch, 2 );

  // Max number of items returned by queries
  box = new TQHBox( this );
  box->setSpacing( KDialog::spacingHint() );
  lay->addWidget( box );
  mMaxItems = new KIntNumInput( box );
  mMaxItems->setLabel( i18n( "&Maximum number of items returned by query" ), TQt::AlignLeft | TQt::AlignVCenter );
  mMaxItems->setMinValue( 0 );
  connect( mMaxItems, TQT_SIGNAL( valueChanged(int) ), this, TQT_SLOT( slotChanged() ) );
  stretch = new TQWidget( box );
  box->setStretchFactor( stretch, 2 );

#ifdef NOT_USEFUL_CURRENTLY
  mAddNewServersCB = new TQCheckBox( i18n( "Automatically add &new servers discovered in CRL distribution points" ), this );
  connect( mAddNewServersCB, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotChanged() ) );
  lay->addWidget( mAddNewServersCB );
#endif

#ifndef HAVE_UNBROKEN_KCMULTIDIALOG
  load();
#endif
}

void DirectoryServicesConfigurationPage::load()
{
  mWidget->load();

  mTimeoutConfigEntry = configEntry( s_timeout_componentName, s_timeout_groupName, s_timeout_entryName, Kleo::CryptoConfigEntry::ArgType_UInt, false );
  if ( mTimeoutConfigEntry ) {
    TQTime time = TQTime().addSecs( mTimeoutConfigEntry->uintValue() );
    //kdDebug() << "timeout:" << mTimeoutConfigEntry->uintValue() << "  -> " << time << endl;
    mTimeout->setTime( time );
  }

  mMaxItemsConfigEntry = configEntry( s_maxitems_componentName, s_maxitems_groupName, s_maxitems_entryName, Kleo::CryptoConfigEntry::ArgType_UInt, false );
  if ( mMaxItemsConfigEntry ) {
    mMaxItems->blockSignals( true ); // KNumInput emits valueChanged from setValue!
    mMaxItems->setValue( mMaxItemsConfigEntry->uintValue() );
    mMaxItems->blockSignals( false );
  }

#ifdef NOT_USEFUL_CURRENTLY
  mAddNewServersConfigEntry = configEntry( s_addnewservers_componentName, s_addnewservers_groupName, s_addnewservers_entryName, Kleo::CryptoConfigEntry::ArgType_None, false );
  if ( mAddNewServersConfigEntry ) {
    mAddNewServersCB->setChecked( mAddNewServersConfigEntry->boolValue() );
  }
#endif
}

void DirectoryServicesConfigurationPage::save()
{
  mWidget->save();

  TQTime time( mTimeout->time() );
  unsigned int timeout = time.minute() * 60 + time.second();
  if ( mTimeoutConfigEntry && mTimeoutConfigEntry->uintValue() != timeout )
    mTimeoutConfigEntry->setUIntValue( timeout );
  if ( mMaxItemsConfigEntry && mMaxItemsConfigEntry->uintValue() != (uint)mMaxItems->value() )
    mMaxItemsConfigEntry->setUIntValue( mMaxItems->value() );
#ifdef NOT_USEFUL_CURRENTLY
  if ( mAddNewServersConfigEntry && mAddNewServersConfigEntry->boolValue() != mAddNewServersCB->isChecked() )
    mAddNewServersConfigEntry->setBoolValue( mAddNewServersCB->isChecked() );
#endif

  mConfig->sync( true );

#if 0
  // Also write the LDAP URLs to kabldaprc so that they are used by kaddressbook
  KABSynchronizer sync;
  const KURL::List toAdd = mWidget->urlList();
  KURL::List currentList = sync.readCurrentList();

  KURL::List::const_iterator it = toAdd.begin();
  KURL::List::const_iterator end = toAdd.end();
  for( ; it != end; ++it ) {
    // check if the URL is already in currentList
    if ( currentList.find( *it ) == currentList.end() )
      // if not, add it
      currentList.append( *it );
  }
  sync.writeList( currentList );
#endif
}

void DirectoryServicesConfigurationPage::defaults()
{
  mWidget->defaults();
  if ( mTimeoutConfigEntry )
    mTimeoutConfigEntry->resetToDefault();
  if ( mMaxItemsConfigEntry )
    mMaxItemsConfigEntry->resetToDefault();
#ifdef NOT_USEFUL_CURRENTLY
  if ( mAddNewServersConfigEntry )
    mAddNewServersConfigEntry->resetToDefault();
#endif
  load();
}

extern "C"
{
  KDE_EXPORT TDECModule *create_kleopatra_config_dirserv( TQWidget *parent, const char * )
  {
    DirectoryServicesConfigurationPage *page =
      new DirectoryServicesConfigurationPage( parent, "kleopatra_config_dirserv" );
    return page;
  }
}

// tdelibs-3.2 didn't have the changed signal in TDECModule...
void DirectoryServicesConfigurationPage::slotChanged()
{
  emit changed(true);
}


// Find config entry for ldap servers. Implements runtime checks on the configuration option.
Kleo::CryptoConfigEntry* DirectoryServicesConfigurationPage::configEntry( const char* componentName,
                                                                          const char* groupName,
                                                                          const char* entryName,
                                                                          Kleo::CryptoConfigEntry::ArgType argType,
                                                                          bool isList )
{
    Kleo::CryptoConfigEntry* entry = mConfig->entry( componentName, groupName, entryName );
    if ( !entry ) {
        KMessageBox::error( this, i18n( "Backend error: gpgconf does not seem to know the entry for %1/%2/%3" ).arg( componentName, groupName, entryName ) );
        return 0;
    }
    if( entry->argType() != argType || entry->isList() != isList ) {
        KMessageBox::error( this, i18n( "Backend error: gpgconf has wrong type for %1/%2/%3: %4 %5" ).arg( componentName, groupName, entryName ).arg( entry->argType() ).arg( entry->isList() ) );
        return 0;
    }
    return entry;
}

#include "dirservconfigpage.moc"
