/***************************************************************************
 *   Copyright (C) 2004 by Paulo Moura Guedes                              *
 *   moura@kdewebdev.org                                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "resultssearchbar.h"

#include <kcombobox.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <tdelocale.h>
#include <kstandarddirs.h>
#include <kdebug.h>

#include <tqapplication.h>
#include <tqhbox.h>
#include <tqlabel.h>
#include <tqpixmap.h>
#include <tqstring.h>
#include <tqtimer.h>
#include <tqtoolbutton.h>
#include <tqtooltip.h>
#include <tqlayout.h>


class ResultsSearchBar::ResultsSearchBarPrivate
{
public:
    ResultsSearchBarPrivate()
    : layout(0), searchLine(0), searchCombo(0), delay(400), m_lastComboIndex(0)
    {}

    TQString searchText;
    TQTimer timer;
    TQHBoxLayout* layout;
    KLineEdit* searchLine;
    KComboBox* searchCombo;
    int delay;
    int m_lastComboIndex;
};

ResultsSearchBar::ResultsSearchBar(TQWidget* parent, const char* name)
        : TQWidget(parent, name), d(new ResultsSearchBar::ResultsSearchBarPrivate)
{
    setSizePolicy(TQSizePolicy(TQSizePolicy::Minimum, TQSizePolicy::Fixed));
    
    d->layout = new TQHBoxLayout(this);    
    d->layout->setMargin(2);
    d->layout->setSpacing(5);
    
    TQToolButton* clearButton = new TQToolButton(this);
    clearButton->setIconSet(SmallIconSet(TQApplication::reverseLayout() ? "clear_left" : "locationbar_erase"));
    clearButton->setAutoRaise(true);
    d->layout->addWidget(clearButton);

    TQLabel* searchLabel = new TQLabel(this);
    searchLabel->setText(i18n("S&earch:"));
    d->layout->addWidget(searchLabel);

    d->searchLine = new KLineEdit(this, "searchline");
    connect(d->searchLine, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SLOT(slotSearchStringChanged(const TQString &)));

    searchLabel->setBuddy(d->searchLine);
    d->layout->addWidget(d->searchLine);

    TQLabel* statusLabel = new TQLabel(this);
    statusLabel->setText( i18n("Status:") );
    d->layout->addWidget(statusLabel);

    d->searchCombo = new KComboBox(this, "searchcombo");
    TQPixmap iconAll = TDEGlobal::iconLoader()->loadIcon("application-x-executable", TDEIcon::Small);
    TQPixmap iconGood = TDEGlobal::iconLoader()->loadIcon("ok", TDEIcon::Small);
    TQPixmap iconBroken = TDEGlobal::iconLoader()->loadIcon("no", TDEIcon::Small);
    TQPixmap iconMalformed = TDEGlobal::iconLoader()->loadIcon("bug", TDEIcon::Small);
    TQPixmap iconUndetermined = TDEGlobal::iconLoader()->loadIcon("help", TDEIcon::Small);

    d->searchCombo->insertItem(iconAll, i18n("All Links"));
    d->searchCombo->insertItem(iconGood, i18n("Good Links"));
    d->searchCombo->insertItem(iconBroken, i18n("Broken Links"));
    d->searchCombo->insertItem(iconMalformed, i18n("Malformed Links"));
    d->searchCombo->insertItem(iconUndetermined, i18n("Undetermined Links"));
    d->layout->addWidget(d->searchCombo);

    TQToolTip::add(clearButton, i18n("Clear filter"));
    TQToolTip::add( d->searchLine, i18n("Enter the terms to filter the result link list"));
    TQToolTip::add( d->searchCombo, i18n("Choose what kind of link status to show in result list"));

    connect(clearButton, TQT_SIGNAL( clicked() ),
            this, TQT_SLOT(slotClearSearch()) );

    connect(d->searchCombo, TQT_SIGNAL(activated(int)),
            this, TQT_SLOT(slotSearchComboChanged(int)));

    connect(&(d->timer), TQT_SIGNAL(timeout()), this, TQT_SLOT(slotActivateSearch()));
}

ResultsSearchBar::~ResultsSearchBar()
{
    delete d;
    d = 0;
}

TQString const& ResultsSearchBar::text() const
{
    return d->searchText;
}

int ResultsSearchBar::status() const
{
    return d->searchCombo->currentItem();
}

void ResultsSearchBar::setDelay(int ms)
{
    d->delay = ms;
}

int ResultsSearchBar::delay() const
{
    return d->delay;
}

void ResultsSearchBar::slotClearSearch()
{
    if(status() != 0 || !d->searchLine->text().isEmpty())
    {
        d->searchLine->clear();
        d->searchCombo->setCurrentItem(0);
        d->timer.stop();
        slotActivateSearch();
    }
}

void ResultsSearchBar::slotSetStatus(int status)
{
    d->searchCombo->setCurrentItem(status);
}

void ResultsSearchBar::slotSetText(const TQString& text)
{
    d->searchLine->setText(text);
}

void ResultsSearchBar::slotSearchComboChanged(int index)
{
    if(d->timer.isActive())
        d->timer.stop();

    if(d->m_lastComboIndex == index)
        return;
    
    d->m_lastComboIndex = index;
    
    d->timer.start(200, true);
}

void ResultsSearchBar::slotSearchStringChanged(const TQString& search)
{
    if(d->timer.isActive())
        d->timer.stop();

    if(d->searchText == search) 
        return;
    
    d->searchText = search;
    
    d->timer.start(200, true);
}

void ResultsSearchBar::slotActivateSearch()
{
    kdDebug(23100) << "ResultsSearchBar::slotActivateSearch" << endl;
    
    ResultView::Status status = selectedStatus();
    
    emit signalSearch(LinkMatcher(d->searchLine->text(), status));
}

LinkMatcher ResultsSearchBar::currentLinkMatcher() const
{
    return LinkMatcher(d->searchLine->text(), selectedStatus());
}

ResultView::Status ResultsSearchBar::selectedStatus() const
{
    ResultView::Status status = ResultView::none;
    
    if(d->searchCombo->currentItem())
    {
        switch(d->searchCombo->currentItem())
        {
            case 1: // Good
            {
                status = ResultView::good;
                break;
            }
            case 2: // Broken
            {
                status = ResultView::bad;
                break;
            }
            case 3: // Malformed
            {
                status = ResultView::malformed;
                break;
            }
            case 4: // Undetermined
            {
                status = ResultView::undetermined;
                break;
            }
            default:
                break;
        }
    }
    return status;
}


#include "resultssearchbar.moc"
