/* This file is part of the KDE project
   Copyright (C) 2001 Matthias Hoelzer-Kluepfel <hoelzer@kde.org>
   Copyright (C) 2002-2003 Roberto Raggi <roberto@kdevelop.org>
   Copyright (C) 2002 Simon Hausmann <hausmann@kde.org>
   Copyright (C) 2003 Jens Dagerbo <jens.dagerbo@swipnet.se>
   Copyright (C) 2003 Mario Scalas <mario.scalas@libero.it>
   Copyright (C) 2003-2004 Alexander Dymo <adymo@kdevelop.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <kdebug.h>

#include "kdevproject.h"
#include <urlutil.h>
#include <tqfileinfo.h>
#include <tqtimer.h>
#include "kdevprojectiface.h"

struct KDevProject::Private {
    TQMap<TQString, TQString> m_absToRel;
    TQStringList m_symlinkList;
    TQTimer *m_timer;
    KDevProjectIface *m_iface;
};

KDevProject::KDevProject(const KDevPluginInfo *info, TQObject *parent, const char *name)
    : KDevPlugin(info, parent, name), d(new KDevProject::Private())
{
    connect( this, TQT_SIGNAL(addedFilesToProject(const TQStringList& )), this, TQT_SLOT(buildFileMap()) );
    connect( this, TQT_SIGNAL(removedFilesFromProject(const TQStringList& )), this, TQT_SLOT(buildFileMap()) );

    connect( this, TQT_SIGNAL(addedFilesToProject(const TQStringList& )), this, TQT_SLOT(slotAddFilesToFileMap(const TQStringList& )) );
    connect( this, TQT_SIGNAL(removedFilesFromProject(const TQStringList& )), this, TQT_SLOT(slotRemoveFilesFromFileMap(const TQStringList& )) );

    d->m_timer = new TQTimer(this);
    connect(d->m_timer, TQT_SIGNAL(timeout()), this, TQT_SLOT(slotBuildFileMap()));
    d->m_iface = new KDevProjectIface(this);
}

KDevProject::~KDevProject()
{
    d->m_timer->stop();
    delete d->m_iface;
    delete d->m_timer;
    delete d;
}

void KDevProject::changedFile( const TQString & fileName )
{
    TQStringList fileList;
    fileList.append ( fileName );

    emit changedFilesInProject( fileList );

}

void KDevProject::changedFiles( const TQStringList & fileList )
{
    emit changedFilesInProject( fileList );
}

KDevProject::Options KDevProject::options() const
{
    return (KDevProject::Options)0;
}

bool KDevProject::isProjectFile( const TQString & absFileName )
{
    return d->m_absToRel.contains( absFileName );
}

TQString KDevProject::relativeProjectFile( const TQString & absFileName )
{
    if( isProjectFile(absFileName) )
	return d->m_absToRel[ absFileName ];
    return TQString();
}

void KDevProject::buildFileMap()
{
    d->m_timer->stop();
    d->m_timer->start(0, true);
}

void KDevProject::slotBuildFileMap()
{
    kdDebug(9000) << k_funcinfo << endl;

    d->m_absToRel.clear();
    d->m_symlinkList.clear();
    const TQStringList fileList = allFiles();
    for( TQStringList::ConstIterator it=fileList.begin(); it!=fileList.end(); ++it )
    {
	TQFileInfo fileInfo( projectDirectory() + "/" + *it );
	d->m_absToRel[ URLUtil::canonicalPath(fileInfo.absFilePath()) ] = *it;

        if ( URLUtil::canonicalPath( fileInfo.absFilePath() ) != fileInfo.absFilePath() )
        {
            d->m_symlinkList << *it;
        }
    }
}

void KDevProject::openProject( const TQString & /*dirName*/, const TQString & /*projectName*/ )
{
    buildFileMap();
}

TQStringList KDevProject::symlinkProjectFiles( )
{
    return d->m_symlinkList;
}

TQString KDevProject::defaultRunDirectory(const TQString& projectPluginName) const
{
    return DomUtil::readEntry(*projectDom(), "/" + projectPluginName + "/run/globalcwd");
}

void KDevProject::slotAddFilesToFileMap( const TQStringList & fileList )
{
	TQStringList::ConstIterator it = fileList.begin();
	while( it != fileList.end() )
	{
		TQFileInfo fileInfo( projectDirectory() + "/" + *it );
		d->m_absToRel[ URLUtil::canonicalPath(fileInfo.absFilePath()) ] = *it;

		if ( URLUtil::canonicalPath( fileInfo.absFilePath() ) != fileInfo.absFilePath() )
		{
			d->m_symlinkList << *it;
		}

		++it;
	}
}

void KDevProject::slotRemoveFilesFromFileMap( const TQStringList & fileList )
{
	TQStringList::ConstIterator it = fileList.begin();
	while( it != fileList.end() )
	{
		TQFileInfo fileInfo( projectDirectory() + "/" + *it );
		d->m_absToRel.remove( URLUtil::canonicalPath(fileInfo.absFilePath()) );

		d->m_symlinkList.remove( *it );

		++it;
	}
}

#include "kdevproject.moc"
