/***************************************************************************
    begin                : Tue Nov 9 2004
    copyright            : (C) 2004 by Scott Wheeler
    email                : wheeler@kde.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef NOWPLAYING_H
#define NOWPLAYING_H

#include <kactivelabel.h>

#include <tqhbox.h>
#include <tqlabel.h>
#include <tqguardedptr.h>

#include "filehandle.h"
#include "playlist.h"

class TQTimer;
class TQPoint;

class NowPlayingItem;
class PlaylistCollection;
class Playlist;

/**
 * This is the widget that holds all of the other items and handles updating them
 * when the playing item changes.
 */

class NowPlaying : public TQHBox
{
    Q_OBJECT
  

public:
    NowPlaying(TQWidget *parent, PlaylistCollection *collection,
               const char *name = 0);
    void addItem(NowPlayingItem *item);
    PlaylistCollection *collection() const;

private slots:
    void slotUpdate();

private:
    struct Observer : public PlaylistObserver
    {
        Observer(NowPlaying *parent, PlaylistInterface *playlist) :
            PlaylistObserver(playlist),
            m_parent(parent) {}
        virtual void updateCurrent() {}
        virtual void updateData() { m_parent->slotUpdate(); }
        NowPlaying *m_parent;
    };
    friend struct Observer;

    Observer m_observer;
    PlaylistCollection *m_collection;
    TQValueList<NowPlayingItem *> m_items;
};

/**
 * Abstract base for the other NowPlaying items.
 */

class NowPlayingItem
{
public:
    virtual void update(const FileHandle &file) = 0;
    NowPlaying *parent() const { return m_parent; }
protected:
    NowPlayingItem(NowPlaying *parent) : m_parent(parent) { parent->addItem(this); }
private:
    NowPlaying *m_parent;
};

/**
 * Displays the cover of the currently playing file if available, or hides
 * itself if not.
 */

class CoverItem : public TQLabel, public NowPlayingItem
{
public:
    CoverItem(NowPlaying *parent);
    virtual void update(const FileHandle &file);
    virtual void mouseReleaseEvent(TQMouseEvent *event);

protected:
    virtual void dragEnterEvent(TQDragEnterEvent *e);
    virtual void dropEvent(TQDropEvent *e);

    virtual void mousePressEvent(TQMouseEvent *e);
    virtual void mouseMoveEvent(TQMouseEvent *e);

private:
    FileHandle m_file;
    bool m_dragging;
    TQPoint m_dragStart;
};

/**
 * A link label that doesn't automatically open Konqueror.
 */

class LinkLabel : public KActiveLabel
{
public:
    LinkLabel(TQWidget *parent, const char *name = 0) : KActiveLabel(parent, name) {}
    virtual void openLink(const TQString &) {}
};

/**
 * Show the text information on the current track and provides links to the
 * album and artist of the currently playing item.
 */

class TrackItem : public TQWidget, public NowPlayingItem
{
    Q_OBJECT
  

public:
    TrackItem(NowPlaying *parent);
    virtual void update(const FileHandle &file);

private slots:
    void slotOpenLink(const TQString &link);
    void slotUpdate();

private:
    FileHandle m_file;
    LinkLabel *m_label;
};

/**
 * Shows up to 10 items of history and links to those items.
 */

class HistoryItem : public LinkLabel, public NowPlayingItem
{
    Q_OBJECT
  

public:
    HistoryItem(NowPlaying *parent);
    virtual void update(const FileHandle &file);
    virtual void openLink(const TQString &link);

private slots:
    void slotAddPlaying();

private:
    struct Item
    {
        Item() {}
        Item(const TQString &a, const FileHandle &f, Playlist *p)
            : anchor(a), file(f), playlist(p) {}

        TQString anchor;
        FileHandle file;
        TQGuardedPtr<Playlist> playlist;
    };

    TQValueList<Item> m_history;
    LinkLabel *m_label;
    TQTimer *m_timer;
    FileHandle m_file;
};

#endif
