/* This file is part of the KDE libraries
    Copyright (C) 2001 Ellis Whitehead <ellis@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef _TDEACCELBASE_H
#define _TDEACCELBASE_H

#include <tqmap.h>
#include <tqptrvector.h>
#include <tqstring.h>
#include <tqvaluevector.h>
#include <tqvaluelist.h>

#include "tdeaccelaction.h"
#include "kkeyserver_x11.h"

class TQPopupMenu;
class TQWidget;

//----------------------------------------------------

/**
 * @internal
 * Handle keyboard accelerators.
 *
 * Allow an user to configure
 * key bindings through application configuration files or through the
 * @ref KKeyChooser GUI.
 *
 * A @ref TDEAccel contains a list of accelerator items. Each accelerator item
 * consists of an action name and a keyboard code combined with modifiers
 * (Shift, Ctrl and Alt.)
 *
 * For example, "Ctrl+P" could be a shortcut for printing a document. The key
 * codes are listed in ckey.h. "Print" could be the action name for printing.
 * The action name identifies the key binding in configuration files and the
 * @ref KKeyChooser GUI.
 *
 * When pressed, an accelerator key calls the slot to which it has been
 * connected. Accelerator items can be connected so that a key will activate
 * two different slots.
 *
 * A TDEAccel object handles key events sent to its parent widget and to all
 * children of this parent widget.
 *
 * Key binding reconfiguration during run time can be prevented by specifying
 * that an accelerator item is not configurable when it is inserted. A special
 * group of non-configurable key bindings are known as the
 * standard accelerators.
 *
 * The standard accelerators appear repeatedly in applications for
 * standard document actions such as printing and saving. Convenience methods are
 * available to insert and connect these accelerators which are configurable on
 * a desktop-wide basis.
 *
 * It is possible for a user to choose to have no key associated with
 * an action.
 *
 * The translated first argument for @ref insertItem() is used only
 * in the configuration dialog.
 *<pre>
 * TDEAccel *a = new TDEAccel( myWindow );
 * // Insert an action "Scroll Up" which is associated with the "Up" key:
 * a->insertItem( i18n("Scroll up"), "Scroll Up", "Up" );
 * // Insert an action "Scroll Down" which is not associated with any key:
 * a->insertItem( i18n("Scroll down"), "Scroll Down", 0);
 * a->connectItem( "Scroll up", myWindow, SLOT( scrollUp() ) );
 * // a->insertStdItem( TDEStdAccel::Print ); //not necessary, since it
 *	// is done automatially with the
 *	// connect below!
 * a->connectItem(TDEStdAccel::Print, myWindow, SLOT( printDoc() ) );
 *
 * a->readSettings();
 *</pre>
 *
 * If a shortcut has a menu entry as well, you could insert them like
 * this. The example is again the @ref TDEStdAccel::Print from above.
 *
 * <pre>
 * int id;
 * id = popup->insertItem("&Print",this, SLOT(printDoc()));
 * a->changeMenuAccel(popup, id, TDEStdAccel::Print );
 * </pre>
 *
 * If you want a somewhat "exotic" name for your standard print action, like
 *   id = popup->insertItem(i18n("Print &Document"),this, SLOT(printDoc()));
 * it might be a good idea to insert the standard action before as
 *          a->insertStdItem( TDEStdAccel::Print, i18n("Print Document") )
 * as well, so that the user can easily find the corresponding function.
 *
 * This technique works for other actions as well.  Your "scroll up" function
 * in a menu could be done with
 *
 * <pre>
 *    id = popup->insertItem(i18n"Scroll &up",this, SLOT(scrollUp()));
 *    a->changeMenuAccel(popup, id, "Scroll Up" );
 * </pre>
 *
 * Please keep the order right:  First insert all functions in the
 * acceleratior, then call a -> @ref readSettings() and @em then build your
 * menu structure.
 *
 * @short Configurable key binding support.
 * @version $Id: tdeaccelbase.h,v 1.22 2002/10/06 18:19:39 ellis Exp $
 */

class TDEAccelBase
{
 public:
	enum Init { QT_KEYS = 0x00, NATIVE_KEYS = 0x01 };
	enum Signal { KEYCODE_CHANGED };

	TDEAccelBase( int fInitCode );
	virtual ~TDEAccelBase();

	uint actionCount() const;
	TDEAccelActions& actions();
	bool isEnabled() const;

	TDEAccelAction* actionPtr( const TQString& sAction );
	const TDEAccelAction* actionPtr( const TQString& sAction ) const;
	TDEAccelAction* actionPtr( const KKey& key );
	TDEAccelAction* actionPtr( const KKeyServer::Key& key );

	const TQString& configGroup() const { return m_sConfigGroup; }
	void setConfigGroup( const TQString& group );
	void setConfigGlobal( bool global );
	virtual void setEnabled( bool bEnabled ) = 0;
	bool getAutoUpdate() { return m_bAutoUpdate; }
	// return value of AutoUpdate flag before this call.
	bool setAutoUpdate( bool bAuto );

// Procedures for manipulating Actions.
	//void clearActions();

	TDEAccelAction* insert( const TQString& sName, const TQString& sDesc );
	TDEAccelAction* insert(
	                 const TQString& sAction, const TQString& sDesc, const TQString& sHelp,
	                 const TDEShortcut& rgCutDefaults3, const TDEShortcut& rgCutDefaults4,
	                 const TQObject* pObjSlot, const char* psMethodSlot,
			 bool bConfigurable = true, bool bEnabled = true );
	bool remove( const TQString& sAction );
	bool setActionSlot( const TQString& sAction, const TQObject* pObjSlot, const char* psMethodSlot );

	bool updateConnections();

	bool setShortcut( const TQString& sAction, const TDEShortcut& cut );

// Modify individual Action sub-items
	bool setActionEnabled( const TQString& sAction, bool bEnable );

	/**
	 * Read all key associations from @p config, or (if @p config
	 * is zero) from the application's configuration file
	 * @ref TDEGlobal::config().
	 *
	 * The group in which the configuration is stored can be
	 * set with @ref setConfigGroup().
	 */
	void readSettings( TDEConfigBase* pConfig = 0 );

	/**
	 * Write the current configurable associations to @p config,
         * or (if @p config is zero) to the application's
	 * configuration file.
	 */
	void writeSettings( TDEConfigBase* pConfig = 0 ) const;

	TQPopupMenu* createPopupMenu( TQWidget* pParent, const KKeySequence& );

 // Protected methods
 protected:
	void slotRemoveAction( TDEAccelAction* );

	void createKeyList( TQValueVector<struct X>& rgKeys );
	bool insertConnection( TDEAccelAction* );
	bool removeConnection( TDEAccelAction* );

	virtual bool emitSignal( Signal ) = 0;
	virtual bool connectKey( TDEAccelAction&, const KKeyServer::Key& ) = 0;
	virtual bool connectKey( const KKeyServer::Key& ) = 0;
	virtual bool disconnectKey( TDEAccelAction&, const KKeyServer::Key& ) = 0;
	virtual bool disconnectKey( const KKeyServer::Key& ) = 0;

 protected:
	struct ActionInfo
	{
		TDEAccelAction* pAction;
		uint iSeq, iVariation;
		//ActionInfo* pInfoNext; // nil if only one action uses this key.

		ActionInfo() { pAction = 0; iSeq = 0xffff; iVariation = 0xffff; }
		ActionInfo( TDEAccelAction* _pAction, uint _iSeq, uint _iVariation )
			{ pAction = _pAction; iSeq = _iSeq; iVariation = _iVariation; }
	};
	typedef TQMap<KKeyServer::Key, ActionInfo> KKeyToActionMap;

	TDEAccelActions m_rgActions;
	KKeyToActionMap m_mapKeyToAction;
	TQValueList<TDEAccelAction*> m_rgActionsNonUnique;
	bool m_bNativeKeys; // Use native key codes instead of TQt codes
	bool m_bEnabled;
	bool m_bConfigIsGlobal;
	TQString m_sConfigGroup;
	bool m_bAutoUpdate;
	TDEAccelAction* mtemp_pActionRemoving;

 private:
	TDEAccelBase& operator =( const TDEAccelBase& );

	friend class TDEAccelActions;
};

#endif // _TDEACCELBASE_H
