/* This file is part of the Keep project
   Copyright (C) 2005 Jean-Rémy Falleri <jr.falleri@laposte.net>

   Keep is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   Keep is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Keep; if not, write to the
   Free Software Foundation, Inc.,
   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           */

#include "restorebackupwizard.h"

#include <tqpushbutton.h>
#include <kiconloader.h>
#include <tdelistview.h>
#include <tdemessagebox.h>
#include <tdelocale.h>
#include <kactivelabel.h>
#include <tqcheckbox.h>
#include <tqradiobutton.h>
#include <kurlrequester.h>
#include <tqvaluelist.h>
#include <tqdir.h>
#include <tdemessagebox.h>

#include "backuplistviewitem.h"
#include "backupconfig.h"

#include "listviewtooltip.h"
#include "incrementlistviewitem.h"

RestoreBackupWizard::RestoreBackupWizard(TQWidget *parent,const char* name): KWizard( parent, name, true)
{
	m_manager = new RDBManager();
	connect( m_manager, TQT_SIGNAL(backupError(Backup,TQString)), this, TQT_SLOT(slotRestoreError(Backup,TQString)) );

	TDEIconLoader icons;
	this->setIcon( icons.loadIcon( "wizard",TDEIcon::Small ));
	this->setCaption("Restore a backup");

	m_popup = new TDEPopupMenu(this);
	m_popup->insertTitle(i18n("Menu"));
	m_popup->insertItem(i18n("Show differences destination/backup"), this, TQT_SLOT(slotShowDiff()));
	m_popup->insertItem(i18n("List changed files"), this, TQT_SLOT(slotShowChanged()));
	m_popup->insertItem(i18n("List files"), this, TQT_SLOT(slotShowList()));

	setupPage1();
	setupPage2();
	setupPage3();

	initConnections();

	setNextEnabled(page1,false);
	setNextEnabled(page2,true);
	setFinishEnabled(page3, false);

	resize( minimumSizeHint() );
}

RestoreBackupWizard::~RestoreBackupWizard()
{
	delete m_manager;
}

void RestoreBackupWizard::initConnections()
{
	// If custom source setted
	connect(restoreBackupWizard1View->btnCustomSource, TQT_SIGNAL( toggled(bool)),this,TQT_SLOT(slotCustomSourceChecked(bool)));

	// Date selection popup menu
	connect(restoreBackupWizard3View->incrementList, TQT_SIGNAL(contextMenu (TDEListView *, TQListViewItem *, const TQPoint &)), this, TQT_SLOT(slotContextMenu(TDEListView *,TQListViewItem*,const TQPoint&)));

	// Page 1 changes
	connect(restoreBackupWizard1View->m_lstBackup,TQT_SIGNAL(selectionChanged()),this,TQT_SLOT(slotPage1Changed()));
	connect(restoreBackupWizard1View->btnCustomSource,TQT_SIGNAL(toggled(bool)),this,TQT_SLOT(slotPage1Changed()));
	connect(restoreBackupWizard1View->customSource,TQT_SIGNAL(textChanged(const TQString &)),this,TQT_SLOT(slotPage1Changed()));

	// Page 2 changes
	connect(restoreBackupWizard2View->btnCustomDest,TQT_SIGNAL(toggled(bool)),this,TQT_SLOT(slotPage2Changed()));
	connect(restoreBackupWizard2View->customDest,TQT_SIGNAL(textChanged(const TQString &)),this,TQT_SLOT(slotPage2Changed()));
	
	// Page 3 Changes
	connect(restoreBackupWizard3View->incrementList,TQT_SIGNAL(selectionChanged()),this,TQT_SLOT(slotPage3Changed()));
	
	// Finish button
	connect(finishButton(),TQT_SIGNAL(clicked()),this,TQT_SLOT(slotRestoreBackup()));
}

void RestoreBackupWizard::slotPage1Changed()
{
	setNextEnabled(page1,false);
	Backup b;
	// Checks if a standard backup is selected.
	if ( !restoreBackupWizard1View->btnCustomSource->isChecked() )
	{
		TQListViewItem *item = restoreBackupWizard1View->m_lstBackup->selectedItem();
		if ( item )
		{
			BackupListViewItem *backupItem = static_cast<BackupListViewItem*>(item);
			b = backupItem->backup();
		}
	}
	// If not, checks if a custom source
	else
	{
		if ( restoreBackupWizard1View->customSource->url() != "" )
			b.setDest(restoreBackupWizard1View->customSource->url());
	}
	if ( b.dest() != "" )
	{		
		RDBManager manager;
		TQDateTime lastBackup = manager.lastIncrement(b);
		restoreBackupWizard3View->lblLastBackup->setText(lastBackup.toString(Qt::LocalDate));
		TQValueList<TQDateTime> increments = manager.incrementList(b);
		if ( increments.size() != 0 )
		{
			restoreBackupWizard3View->incrementList->clear();
			TQValueList<TQDateTime>::iterator it;
			for ( it = increments.begin(); it != increments.end(); ++it )
			{
				IncrementListViewItem *item = new IncrementListViewItem(restoreBackupWizard3View->incrementList,(*it));
			}
			setBackup(b);
			setNextEnabled(page1,true);
		}
	}
}

void RestoreBackupWizard::slotPage2Changed()
{
	setNextEnabled(page2,false);
	if ( !restoreBackupWizard2View->btnCustomDest->isChecked() )
	{
		TQListViewItem *item = restoreBackupWizard1View->m_lstBackup->selectedItem();
		BackupListViewItem *backupItem = static_cast<BackupListViewItem*>(item);
		setBackup(backupItem->backup());
		setNextEnabled(page2,true);
	}
	else
	{
		TQDir dir( restoreBackupWizard2View->customDest->url() );
		if ( dir.exists() && restoreBackupWizard2View->customDest->url() != "" )
		{
			Backup b = backup();
			b.setSource(restoreBackupWizard2View->customDest->url());
			setBackup(b);
			setNextEnabled(page2,true);	
		}
	}
}

void RestoreBackupWizard::slotPage3Changed()
{
	setFinishEnabled(page3,false);
	TQListViewItem *item = restoreBackupWizard3View->incrementList->selectedItem();
	if ( item )
		setFinishEnabled(page3,true);	
}

void RestoreBackupWizard::slotCustomSourceChecked(bool value)
{
	if ( value )
	{
		restoreBackupWizard2View->btnCustomDest->setChecked(true);
		restoreBackupWizard2View->btnCustomDest->setEnabled(false);
	}
	else
		restoreBackupWizard2View->btnCustomDest->setEnabled(true);
}

void RestoreBackupWizard::setupPage1()
{
	page1 = new TQHBox( this );
	page1->setSpacing(0);
	page1->setMargin(0);

	restoreBackupWizard1View = new RestoreBackupWizard1View(page1,"restoreBackupWizard1View");
	BackupConfig *backupConfig = new BackupConfig();
	TQValueList<Backup> backupList = backupConfig->backupList();
	TQValueList<Backup>::iterator it;
	for ( it = backupList.begin(); it != backupList.end(); ++it )
	{
		new BackupListViewItem(restoreBackupWizard1View->m_lstBackup,*it);
	}
	
	TQToolTip::remove(restoreBackupWizard1View->m_lstBackup);
	new ListViewToolTip(restoreBackupWizard1View->m_lstBackup);

	restoreBackupWizard1View->customSource->setMode(KFile::Directory);

	addPage( page1, "Directory to restore" );
	delete backupConfig;
}

void RestoreBackupWizard::setupPage2()
{
	page2 = new TQHBox( this );
	page2->setSpacing(0);
	page2->setMargin(0);
	
	restoreBackupWizard2View = new RestoreBackupWizard2View(page2,"restoreBackupWizard2View");

	restoreBackupWizard2View->customDest->setMode(KFile::Directory);

	addPage(page2, "Destination directory" );
}

void RestoreBackupWizard::setupPage3()
{
	page3 = new TQHBox( this );
	page3->setSpacing(0);
	page3->setMargin(0);
	
	restoreBackupWizard3View = new RestoreBackupWizard3View(page3,"restoreBackupWizard3View");

	restoreBackupWizard3View->incrementList->setSorting(0,false);
	restoreBackupWizard3View->lblDate->setText(TQDateTime::currentDateTime().toString(Qt::LocalDate));
	addPage( page3, "Date to restore" );
}

void RestoreBackupWizard::slotContextMenu(TDEListView *list,TQListViewItem *item,const TQPoint&p)
{
	m_popup->popup(p, 1);
}

Backup RestoreBackupWizard::backup()
{
	return m_backup;
}

void RestoreBackupWizard::setBackup(Backup backup)
{
	m_backup = backup;
}

void RestoreBackupWizard::slotShowDiff()
{
	Backup b = backup();
	TQListViewItem *item2 = restoreBackupWizard3View->incrementList->selectedItem();
	IncrementListViewItem *incrementItem = static_cast<IncrementListViewItem*>(item2);
	TQDateTime date = incrementItem->date();
	TQString diff = m_manager->compareAtTime(b,date);
	TQString message = "<p><b>" + i18n("List of modifications since the selected increment:") + "</b></p><p>" + diff + "</p>";
	KMessageBox::information(this,message,i18n("Modifications"));
}

void RestoreBackupWizard::slotShowChanged()
{
	Backup b = backup();
	TQListViewItem *item2 = restoreBackupWizard3View->incrementList->selectedItem();
	IncrementListViewItem *incrementItem = static_cast<IncrementListViewItem*>(item2);
	TQDateTime date = incrementItem->date();
	TQString changed = m_manager->listChangedSince(b,date);
	if ( changed == "" )
		changed = i18n("Nothing.");
	TQString message = "<p><b>" + i18n("List of changed files since the selected increment:") + "</b></p><p>" + changed + "</p>";
	KMessageBox::information(this,message,i18n("Changed files"));
}

void RestoreBackupWizard::slotShowList()
{
	Backup b = backup();
	TQListViewItem *item2 = restoreBackupWizard3View->incrementList->selectedItem();
	IncrementListViewItem *incrementItem = static_cast<IncrementListViewItem*>(item2);
	TQDateTime date = incrementItem->date();
	TQString list = m_manager->listAtTime(b,date);
	TQString message = "<p><b>" + i18n("List of files in the selected increment:") + "</b></p><p>" + list + "</p>";
	KMessageBox::information(this,message,i18n("List of files"));
}

void RestoreBackupWizard::slotRestoreBackup()
{
	Backup b = backup();
	TQListViewItem *item2 = restoreBackupWizard3View->incrementList->selectedItem();
	IncrementListViewItem *incrementItem = static_cast<IncrementListViewItem*>(item2);
	TQDateTime date = incrementItem->date();
	m_manager->slotRestoreBackup(b,date);
}

void RestoreBackupWizard::slotRestoreError(Backup backup,TQString errorMessage)
{
	KMessageBox::error(this,i18n("<p><b>An error occured restoring %1 backup:</b></p><p>%2</p>").arg(backup.dest()).arg(errorMessage));
}

#include "restorebackupwizard.moc"
