/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz (peter.penz@gmx.at) and              *
 *   and Patrice Tremblay                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "generalsettingspage.h"

#include <tqlayout.h>
#include <kdialog.h>
#include <tqlabel.h>
#include <tqlineedit.h>
#include <tqvbox.h>
#include <tqgrid.h>
#include <tqgroupbox.h>
#include <tdelocale.h>
#include <tqcheckbox.h>
#include <tqbuttongroup.h>
#include <tqpushbutton.h>
#include <tdefiledialog.h>
#include <tqradiobutton.h>

#include "dolphinsettings.h"
#include "dolphin.h"
#include "dolphinview.h"

GeneralSettingsPage::GeneralSettingsPage(TQWidget* parent) :
    SettingsPageBase(parent),
    m_homeURL(0),
    m_startSplit(0),
    m_startEditable(0)
{
    TQVBoxLayout* topLayout = new TQVBoxLayout(parent, 2, KDialog::spacingHint());

    const int spacing = KDialog::spacingHint();
    const int margin = KDialog::marginHint();
    const TQSizePolicy sizePolicy(TQSizePolicy::Preferred, TQSizePolicy::Fixed);

    DolphinSettings& settings = DolphinSettings::instance();

    TQVBox* vBox = new TQVBox(parent);
    vBox->setSizePolicy(sizePolicy);
    vBox->setSpacing(spacing);
    vBox->setMargin(margin);
    vBox->setSizePolicy(TQSizePolicy::Preferred, TQSizePolicy::Ignored);

    // create 'Home URL' editor
    TQGroupBox* homeGroup = new TQGroupBox(1, Qt::Horizontal, i18n("Home URL"), vBox);
    homeGroup->setSizePolicy(sizePolicy);
    homeGroup->setMargin(margin);

    TQHBox* homeURLBox = new TQHBox(homeGroup);
    homeURLBox->setSizePolicy(sizePolicy);
    homeURLBox->setSpacing(spacing);

    new TQLabel(i18n("Location:"), homeURLBox);
    m_homeURL = new TQLineEdit(settings.homeURL().prettyURL(), homeURLBox);

    TQPushButton* selectHomeURLButton = new TQPushButton(SmallIcon("folder"), TQString(), homeURLBox);
    connect(selectHomeURLButton, TQT_SIGNAL(clicked()),
            this, TQT_SLOT(selectHomeURL()));

    TQHBox* buttonBox = new TQHBox(homeGroup);
    buttonBox->setSizePolicy(sizePolicy);
    buttonBox->setSpacing(spacing);
    TQPushButton* useCurrentButton = new TQPushButton(i18n("Use current location"), buttonBox);
    connect(useCurrentButton, TQT_SIGNAL(clicked()),
            this, TQT_SLOT(useCurrentLocation()));
    TQPushButton* useDefaultButton = new TQPushButton(i18n("Use default location"), buttonBox);
    connect(useDefaultButton, TQT_SIGNAL(clicked()),
            this, TQT_SLOT(useDefaulLocation()));

    // create 'Default View Mode' group
    TQButtonGroup* buttonGroup = new TQButtonGroup(3, Qt::Vertical, i18n("Default View Mode"), vBox);
    buttonGroup->setSizePolicy(sizePolicy);
    buttonGroup->setMargin(margin);

    m_iconsView = new TQRadioButton(i18n("Icons"), buttonGroup);
    m_detailsView = new TQRadioButton(i18n("Details"), buttonGroup);
    m_previewsView = new TQRadioButton(i18n("Previews"), buttonGroup);

    switch (settings.defaultViewMode()) {
        case DolphinView::IconsView:    m_iconsView->setChecked(true); break;
        case DolphinView::DetailsView:  m_detailsView->setChecked(true); break;
        case DolphinView::PreviewsView: m_previewsView->setChecked(true); break;
    }

    // create 'Start with split view' checkbox
    m_startSplit = new TQCheckBox(i18n("Start with split view"), vBox);
    m_startSplit->setChecked(settings.isViewSplit());

    // create 'Start with editable navigation bar' checkbox
    m_startEditable = new TQCheckBox(i18n("Start with editable navigation bar"), vBox);
    m_startEditable->setChecked(settings.isURLEditable());

    // create 'Save view properties for each folder' checkbox
    m_saveView = new TQCheckBox(i18n("Save view properties for each folder"), vBox);
    m_saveView->setChecked(settings.isSaveView());

    // Add a dummy widget with no restriction regarding
    // a vertical resizing. This assures that the dialog layout
    // is not stretched vertically.
    new TQWidget(vBox);

    topLayout->addWidget(vBox);
}


GeneralSettingsPage::~GeneralSettingsPage()
{
}

void GeneralSettingsPage::applySettings()
{
    DolphinSettings& settings = DolphinSettings::instance();

    const KURL url(m_homeURL->text());
    KFileItem fileItem(S_IFDIR, KFileItem::Unknown, url);
    if (url.isValid() && fileItem.isDir()) {
        settings.setHomeURL(url);
    }

    DolphinView::Mode viewMode = DolphinView::IconsView;
    if (m_detailsView->isChecked()) {
        viewMode = DolphinView::DetailsView;
    }
    else if (m_previewsView->isChecked()) {
        viewMode = DolphinView::PreviewsView;
    }
    settings.setDefaultViewMode(viewMode);

    settings.setViewSplit(m_startSplit->isChecked());
    settings.setSaveView(m_saveView->isChecked());
    settings.setURLEditable(m_startEditable->isChecked());
}

void GeneralSettingsPage::selectHomeURL()
{
    const TQString homeURL(m_homeURL->text());
    KURL url(KFileDialog::getExistingURL(homeURL));
    if (!url.isEmpty()) {
        m_homeURL->setText(url.prettyURL());
    }
}

void GeneralSettingsPage::useCurrentLocation()
{
    const DolphinView* view = Dolphin::mainWin().activeView();
    m_homeURL->setText(view->url().prettyURL());
}

void GeneralSettingsPage::useDefaulLocation()
{
    m_homeURL->setText("file://" + TQDir::homeDirPath());
}

#include "generalsettingspage.moc"
