/* -*- C++ -*- adept/taglist.cpp
   written by Peter Rockai <me@mornfall.net> */

#include <tqlabel.h>
#include <tqtimer.h>
#include <kdebug.h>
#include <tqdragobject.h>
#include <tqevent.h>
#include <tdelocale.h>

#include <apt-front/cache/component/tags.h>
#include <adept/taglist.h>
#include <adept/utils.h>

using namespace adept;

TagLabel::TagLabel( Tag t, TagList *l, TQWidget *p, const char *n )
    : TQHBox( p, n ), m_tag( t ), m_list( l )
{
    if ( t == Tag() ) {
        m_description = new TQLabel( TQString( " " ) + i18n( "[none]" ), this );
    } else {
        m_remove = new TQLabel( this );
        m_remove->setPixmap( SmallIcon( u8( "cancel" ) ) );
        m_remove->setSizePolicy( TQSizePolicy::Fixed, TQSizePolicy::Fixed );
        m_description = new TQLabel(  TQString( " [" ) + t.fullname( "n/a" ) + "] "
                                     + t.shortDescription( "n/a" ), this );
    }
}

void TagLabel::mouseReleaseEvent( TQMouseEvent *e ) {
    if ( e->button() == Qt::LeftButton &&
         dynamic_cast< TQLabel * >( childAt( e->pos() ) ) == m_remove )
        m_list->setTags( m_list->tags() - m_tag );
}

TagList::TagList( TQWidget *p, const char *n )
    : TQVBox( p, n )
{
    m_name = new TQLabel( this );
    m_tagBox = new TQVBox( this );
    m_tagBox->setFrameShape( TQFrame::Panel );
    m_tagBox->setFrameShadow( TQFrame::Sunken );
    m_updateScheduled = false;
    setAcceptDrops( true );
    scheduleUpdateList();
    m_tagSpacer = new TQSpacerItem( 0, 0, TQSizePolicy::Minimum, TQSizePolicy::Expanding );
}

void TagList::setTags( Tag::Set t )
{
    m_tags = t;
    scheduleUpdateList();
    emit tagsChanged( m_tags );
}

void TagList::addTag( Tag t )
{
    if ( t == Tag() )
        return;
    m_tags += t;
    scheduleUpdateList();
    emit tagsChanged( m_tags );
}

void TagList::setName( TQString n )
{
    m_name->setText( n );
}

void TagList::scheduleUpdateList()
{
    if (! m_updateScheduled) {
        kdDebug() << "TagList: scheduling update" << endl;
        TQTimer::singleShot( 0, this, TQT_SLOT( updateList() ) );
        m_updateScheduled = true;
    }
}

void TagList::updateList()
{
    kdDebug() << "TagList (" + m_name->text() + "): updating list" << endl;
    clearList();
    if ( m_tags.empty() ) {
        appendLabel( new TagLabel( Tag(), this, m_tagBox ) );
    } else {
        for ( Tag::Set::iterator i = m_tags.begin(); i != m_tags.end(); ++i ) {
            appendLabel( new TagLabel( *i, this, m_tagBox ) );
        }
    }
    m_tagBox->layout()->addItem( m_tagSpacer );
    update();
    parentWidget()->adjustSize();
    m_updateScheduled = false;
}

void TagList::appendLabel( TagLabel *l )
{
    m_list.push_back( l );
    l->show();
}

void TagList::mouseMoveEvent( TQMouseEvent *e ) {
    TagLabel *child = dynamic_cast< TagLabel * >( childAt( e->pos() )->parentWidget() );
    if ( !child )
        return;
    TQDragObject *d = new TQTextDrag( child->tag().fullname( "" ), this );
    d->dragCopy();
}

void TagList::dragEnterEvent( TQDragEnterEvent *e ) {
    kdDebug() << "TagList::dragEnterEvent" << endl;
    e->accept( TQTextDrag::canDecode( e ) );
}

void TagList::dropEvent( TQDropEvent* e ) {
    TQString tag;
    kdDebug() << "TagList: drop event" << endl;
    TQTextDrag::decode( e, tag );
    try {
        addTag( cache::Global::get().tags().tagByName(
                    static_cast< const char * >( tag.local8Bit() ) ) );
    } catch (...) {} // not a tag, ignore
    scheduleUpdateList();
}

void TagList::clearList()
{
    for (List::iterator i = m_list.begin(); i != m_list.end(); ++i ) {
        delete *i;
    }
    m_list.clear();
    m_tagBox->layout()->removeItem( m_tagSpacer );
}
