/** -*- C++ -*-
    @file adept/application.cpp
    @author Peter Rockai <me@mornfall.net>
*/

#include <tdemessagebox.h>
#include <tdelocale.h>
#include <tdeapplication.h>
#include <kstatusbar.h>

#include <apt-front/init.h>
#include <apt-front/cache/component/packages.h>
#include <apt-front/cache/component/state.h>

#include <adept/commitprogress.h>
#include <adept/application.h>
#include <adept/packageinfo.h>
#include <adept/progress.h>
#include <adept/utils.h>

namespace adept {
using namespace cache;

/* void WaitForLister::waiting()
{
    kdDebug() << "WaitForLister::waiting()" << endl;
    bool done = true;
    for ( Vector::iterator i = listers.begin(); i != listers.end(); ++i ) {
        if ( (*i)->busy() ) {
            done = false;
            break;
        }
    }
    if ( listers.empty() ) done = false;
    if ( done ) {
        TQTimer::singleShot( 0, app, slot );
        deleteLater();
    } else {
        TQTimer::singleShot( 100, this, TQT_SLOT( waiting() ) );
    }
    } */

Application::Application()
    : m_acceptReadOnly( false ), m_main( 0 ), m_history( 0 ), m_statusBar( 0 )
{
}

void Application::openCache( unsigned flags )
{
    bool ro = m_acceptReadOnly;
    bool root = ::getuid() == 0 || ::geteuid() == 0;
    try {
        cache::Global::get().open( flags );
    } catch (...) {
        try {
            cache::Global::get().open( flags | Cache::OpenReadOnly );
            if ( ro && root ) {
                KMessageBox::information(
                    m_main, i18n(
                        "You will not be able to change your system settings "
                        "in any way (install, remove or upgrade software), "
                        "because another process is using the packaging system database "
                        "(probably some other Adept application or apt-get or "
                        "aptitude). Please close the other application before "
                        "using this one." ),
                    i18n( "Read Only mode: Database Locked" ) );
            } else if ( !root && ro ) {
                KMessageBox::information(
                    m_main, i18n(
                        "You will not be able to change your system settings "
                        "in any way (install, remove or upgrade software), "
                        "because this application needs special administrator "
                        "(root) privileges. Please run it as root or "
                        "through tdesu or sudo programs to be able to perform "
                        "these actions" ),
                    i18n( "Read Only mode: Need root privileges" ) );
            } else if ( root && !ro ) {
                KMessageBox::information(
                    m_main, i18n(
                        "Another process is using the packaging system database "
                        "(probably some other Adept application or apt-get or "
                        "aptitude). Please close the other application before "
                        "using this one." ),
                    i18n( "Database Locked" ) );
            } else if ( !root && !ro ) {
                KMessageBox::information(
                    m_main, i18n( "This application needs special administrator "
                                  "(root) privileges. Please run it as root or "
                                  "through tdesu or sudo programs" ),
                    i18n( "Need root privileges" ) );
            }
            if ( !ro ) {
                kdDebug() << "cannot continue, exiting" << endl;
                exit( 1 );
            }
        } catch (...) {
            KMessageBox::sorry(
                m_main, i18n(
                    "The APT Database could not be opened!"
                    " This may be caused by incorrect APT configuration"
                    " or some similar problem. Try running apt-setup and"
                    " apt-get update in terminal and see if it helps"
                    " to resolve the problem." ), i18n( "Could not open cache" ) );
            exit( 1 );
        }
    }
}

void Application::initHistory() {
    cache::Global::get().addComponent(
        m_history = new History() );
}

void Application::initKDEDebconf()
{
    // xxx unhardcode the package name somehow?
    if (cache::Global::get().packages()
        .packageByName( "libqt-perl" ).isInstalled())
        putenv( "DEBIAN_FRONTEND=kde" );
}

void Application::initialize()
{
    CommitProgress::initSystem();
    aptFront::init();
    openCache();
    initKDEDebconf();
    initHistory();
    observeComponent< component::State >();
}

void Application::checkpoint() {
    if ( !history() ) return;
    setHistoryEnabled( false );
    history()->checkpoint();
    setHistoryEnabled( true );
}

void Application::undo() {
    if ( !history() ) return;
    setHistoryEnabled( false );
    history()->undo();
    setHistoryEnabled( true );
}

void Application::redo() {
    if ( !history() ) return;
    setHistoryEnabled( false );
    history()->redo();
    setHistoryEnabled( true );
}

TQString Application::changeString() {
    component::State &s = cache().state();
    return i18n( " Install %1, upgrade %2, remove %3 " )
        .arg( s.newInstallCount() ).arg( s.upgradeCount() )
        .arg( s.removeCount() );
}

TQString Application::statusString() {
    component::State &s = cache().state();
    return i18n( " %1 installed, %2 upgradable, %3 available " )
        .arg( s.installedCount() ).arg( s.upgradableCount() )
        .arg( s.availableCount() );
}

TQString Application::sizesString() {
    TQString dl = cache().state().downloadSizeString();
    TQString inst = cache().state().installSizeString();
    return i18n( " download: %1, installation: %2 " ).arg( dl ).arg( inst );
}

void Application::setStatusBar( KStatusBar *s ) {
    m_statusBar = s;
    if ( s ) {
        s->message( i18n( "Initializing..." ) );
        s->insertItem( u8( "" ), 0 );
        s->insertItem( u8( "" ), 1 );
        s->insertItem( u8( "" ), 2 );
        adjustFontSize( s, -1 );
        
        adept::Progress *pr = new adept::Progress();
        pr->setStatusBar( s );
        cache::Global::get().setProgress( pr );
    }
}

void Application::notifyPostChange( component::Base * )
{
    if ( m_statusBar ) {
        m_statusBar->changeItem( changeString(), 0 );
        m_statusBar->changeItem( statusString(), 1 );
        m_statusBar->changeItem( sizesString(), 2 );
    }
}

}
