/***************************************************************************
    copyright            : (C) 2005-2007 by Robby Stephenson
    email                : robby@periapsis.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License as  *
 *   published by the Free Software Foundation;                            *
 *                                                                         *
 ***************************************************************************/

#include "detailedentryitem.h"
#include "detailedlistview.h"
#include "collection.h"
#include "entry.h"
#include "tellico_utils.h"
#include "latin1literal.h"

#include <tdelocale.h>
#include <kstringhandler.h>

#include <tqpainter.h>
#include <tqheader.h>
#include <tqdatetime.h>
#include <tqtimer.h>

namespace {
  static const short ENTRY_MAX_MINUTES_MESSAGE = 5;
}

using Tellico::DetailedEntryItem;

DetailedEntryItem::DetailedEntryItem(DetailedListView* parent_, Data::EntryPtr entry_)
    : EntryItem(parent_, entry_), m_state(Normal), m_time(0), m_timer(0) {
}

DetailedEntryItem::~DetailedEntryItem() {
  delete m_time;
  m_time = 0;
  delete m_timer;
  m_timer = 0;
}

void DetailedEntryItem::setState(State state_) {
  if(m_state == state_) {
    return;
  }
  m_state = state_;

  if(m_state == Normal) {
    delete m_time;
    m_time = 0;
    delete m_timer;
    m_timer = 0;
  } else {
    if(!m_time) {
      m_time = new TQTime;
    }
    m_time->start();

    if(!m_timer) {
      m_timer = new TQTimer();
      m_timer->connect(m_timer, TQT_SIGNAL(timeout()), listView(), TQT_SLOT(triggerUpdate()));
    }
    m_timer->start(30 * 1000); // every 30 seconds
  }

  // have to put this in a timer, or it doesn't update properly
  TQTimer::singleShot(500, listView(), TQT_SLOT(triggerUpdate()));
}

void DetailedEntryItem::paintCell(TQPainter* p_, const TQColorGroup& cg_,
                                  int column_, int width_, int align_) {
  if(m_state == Normal) {
    EntryItem::paintCell(p_, cg_, column_, width_, align_);
    return;
  }

  int t = m_time->elapsed()/(60 * 1000);
  if(t > ENTRY_MAX_MINUTES_MESSAGE) {
    setState(Normal);
    t = 0;
  }

  TQFont f = p_->font();
  f.setBold(true);
  if(m_state == New) {
    f.setItalic(true);
  }
  p_->setFont(f);

  // taken from ListViewItem, but without line drawn to right of cell
  TQColorGroup cg = cg_;
  const TQPixmap* pm = listView()->viewport()->backgroundPixmap();
  if(pm && !pm->isNull()) {
    cg.setBrush(TQColorGroup::Base, TQBrush(backgroundColor(column_), *pm));
    TQPoint o = p_->brushOrigin();
    p_->setBrushOrigin(o.x()-listView()->contentsX(), o.y()-listView()->contentsY());
  } else {
    cg.setColor(listView()->viewport()->backgroundMode() == TQt::FixedColor ?
                TQColorGroup::Background : TQColorGroup::Base,
                backgroundColor(column_));
  }
  // don't call TDEListViewItem::paintCell() since that also does alternate painting, etc...
  TQListViewItem::paintCell(p_, cg, column_, width_, align_);
}

TQColor DetailedEntryItem::backgroundColor(int column_) {
  GUI::ListView* lv = listView();
  if(!lv || m_state == Normal || isSelected()) {
    return EntryItem::backgroundColor(column_);
  }
  int t = m_time->elapsed()/(60 * 1000);
  if(t > ENTRY_MAX_MINUTES_MESSAGE) {
    return EntryItem::backgroundColor(column_);
  }
  return blendColors(lv->colorGroup().highlight(),
                     lv->colorGroup().base(),
                     80 + 20*t/ENTRY_MAX_MINUTES_MESSAGE /* percent */);
                     // no more than 20% of highlight color
}

void DetailedEntryItem::paintFocus(TQPainter*, const TQColorGroup&, const TQRect&) {
// don't paint anything
}
