/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Reginald Stadlbauer <reggie@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "KoCustomVariablesDia.h"
#include "KoCustomVariablesDia.moc"

#include <tdelocale.h>
#include <kbuttonbox.h>

#include <tqcombobox.h>
#include <tqvbox.h>
#include <tqlabel.h>
#include <tqpushbutton.h>
#include <tqheader.h>
#include <klineedit.h>
#include <kdebug.h>

/******************************************************************
 *
 * Class: KoVariableNameDia
 *
 ******************************************************************/

KoVariableNameDia::KoVariableNameDia( TQWidget *parent )
    : KDialogBase( parent, "", TRUE,i18n( "Entry Name" ),Ok|Cancel )
{
    init();
}


KoVariableNameDia::KoVariableNameDia( TQWidget *parent, const TQPtrList<KoVariable>& vars )
    : KDialogBase( parent, "", TRUE, i18n( "Variable Name" ), Ok|Cancel )
{

    init();
    enableButtonOK(false);
    TQPtrListIterator<KoVariable> it( vars );
     for ( ; it.current() ; ++it ) {
        KoVariable *var = it.current();
        if ( var->type() == VT_CUSTOM )
            names->insertItem( ( (KoCustomVariable*) var )->name(), -1 );
    }

}

void KoVariableNameDia::init()
{
    back = makeVBoxMainWidget();

    TQHBox *row1 = new TQHBox( back );
    row1->setSpacing( KDialog::spacingHint() );

    TQLabel *l = new TQLabel( i18n( "Name:" ), row1 );
    l->setFixedSize( l->sizeHint() );
    names = new TQComboBox( TRUE, row1 );
    names->setFocus();

    connect( names, TQT_SIGNAL( textChanged ( const TQString & )),
             this, TQT_SLOT( textChanged ( const TQString & )));
    connect( this, TQT_SIGNAL( okClicked() ),
             this, TQT_SLOT( accept() ) );
    connect( this, TQT_SIGNAL( cancelClicked() ),
             this, TQT_SLOT( reject() ) );
    enableButtonOK( !names->currentText().isEmpty() );
    resize( 350, 100 );
}

TQString KoVariableNameDia::getName() const
{
    return names->currentText();
}

void KoVariableNameDia::textChanged ( const TQString &_text )
{
    enableButtonOK(!_text.isEmpty());
}

/******************************************************************
 *
 * Class: KoCustomVariablesListItem
 *
 ******************************************************************/

KoCustomVariablesListItem::KoCustomVariablesListItem( TQListView *parent )
    : TQListViewItem( parent )
{
    editWidget = new KLineEdit( listView()->viewport() );
    listView()->addChild( editWidget );
}

void KoCustomVariablesListItem::setup()
{
    TQListViewItem::setup();
    setHeight( TQMAX( listView()->fontMetrics().height(),
                     editWidget->sizeHint().height() ) );
    //if ( listView()->columnWidth( 1 ) < editWidget->sizeHint().width() )
    //    listView()->setColumnWidth( 1, editWidget->sizeHint().width() );
}

void KoCustomVariablesListItem::update()
{
    editWidget->resize( listView()->header()->cellSize( 1 ), height() );
    listView()->moveChild( editWidget, listView()->header()->cellPos( 1 ),
                           listView()->itemPos( this ) + listView()->contentsY() );
    editWidget->show();
}

void KoCustomVariablesListItem::setVariable( KoCustomVariable *v )
{
    var = v;
    editWidget->setText( var->value() );
    setText( 0, v->name() );
}

KoCustomVariable *KoCustomVariablesListItem::getVariable() const
{
    return var;
}

void KoCustomVariablesListItem::applyValue()
{
    TQString newVal=editWidget->text();
    if(var->value()!=newVal)
        var->setValue( newVal );
}

int KoCustomVariablesListItem::width( const TQFontMetrics & fm, const TQListView *lv, int c ) const
{
    // The text of the 2nd column isn't known to TQListViewItem, only we know it
    // (it's in our lineedit)
    if ( c == 1 ) {
        TQString val = editWidget->text();
        int w = fm.width( val );
        return w;
    } else
        return TQListViewItem::width( fm, lv, c );
}

/******************************************************************
 *
 * Class: KoCustomVariablesList
 *
 ******************************************************************/

KoCustomVariablesList::KoCustomVariablesList( TQWidget *parent )
    : TDEListView( parent )
{
    header()->setMovingEnabled( FALSE );
    addColumn( i18n( "Variable" ) );
    addColumn( i18n( "Value" ) );
    connect( header(), TQT_SIGNAL( sizeChange( int, int, int ) ),
             this, TQT_SLOT( columnSizeChange( int, int, int ) ) );
    connect( header(), TQT_SIGNAL( sectionClicked( int ) ),
             this, TQT_SLOT( sectionClicked( int ) ) );

    setResizeMode(TQListView::LastColumn);
    setSorting( -1 );
}

void KoCustomVariablesList::setValues()
{
    TQListViewItemIterator it( this );
    for ( ; it.current(); ++it )
        ( (KoCustomVariablesListItem *)it.current() )->applyValue();
}

void KoCustomVariablesList::columnSizeChange( int c, int, int )
{
    if ( c == 0 || c == 1 )
        updateItems();
}

void KoCustomVariablesList::sectionClicked( int )
{
    updateItems();
}

void KoCustomVariablesList::updateItems()
{
    TQListViewItemIterator it( this );
    for ( ; it.current(); ++it )
        ( (KoCustomVariablesListItem*)it.current() )->update();
}

/******************************************************************
 *
 * Class: KoCustomVariablesDia
 *
 ******************************************************************/

KoCustomVariablesDia::KoCustomVariablesDia( TQWidget *parent, const TQPtrList<KoVariable> &variables )
    : KDialogBase( parent, "", TRUE,i18n( "Variable Value Editor" ), Ok|Cancel )
{

    back = makeVBoxMainWidget();

    list = new KoCustomVariablesList( back );

    TQStringList lst;
    TQPtrListIterator<KoVariable> it( variables );
    for ( ; it.current() ; ++it ) {
        KoVariable *var = it.current();
        if ( var->type() == VT_CUSTOM ) {
            KoCustomVariable *v = (KoCustomVariable*)var;
            if ( !lst.contains( v->name() ) ) {
                lst.append( v->name() );
                KoCustomVariablesListItem *item = new KoCustomVariablesListItem( list );
                item->setVariable( v );
            }
        }
    }


    connect( this, TQT_SIGNAL( okClicked() ),
             this, TQT_SLOT( slotOk() ) );
    connect( this, TQT_SIGNAL( cancelClicked() ),
             this, TQT_SLOT( reject() ) );
    showButtonOK(lst.count()>0);

    resize( 600, 400 );
}

void KoCustomVariablesDia::slotOk()
{
    list->setValues();
    accept();
}

/******************************************************************
 *
 * Class: KoCustomVarDialog
 *
 ******************************************************************/

KoCustomVarDialog::KoCustomVarDialog( TQWidget *parent )
    : KDialogBase( parent, "", TRUE,i18n( "Add Variable" ), Ok|Cancel )
{
    init();
    m_name->setFocus();


    connect( this, TQT_SIGNAL( okClicked() ),
             this, TQT_SLOT( slotAddOk() ) );
    connect( this, TQT_SIGNAL( cancelClicked() ),
             this, TQT_SLOT( reject() ) );

    connect( m_name, TQT_SIGNAL( textChanged(const TQString&) ),
             this, TQT_SLOT( slotTextChanged(const TQString&) ) );

    enableButtonOK( false );
    resize( 350, 100 );

}
// edit existing variable
KoCustomVarDialog::KoCustomVarDialog( TQWidget *parent, KoCustomVariable *var )
    : KDialogBase( parent, "", TRUE,i18n( "Edit Variable" ), Ok|Cancel )
{
    m_var = var;
    init();
    m_name->setText( m_var->name() );
    m_value->setText( m_var->value() );
    m_name->setReadOnly(true);
    m_value->setFocus();


    connect( this, TQT_SIGNAL( okClicked() ),
             this, TQT_SLOT( slotEditOk() ) );
    connect( this, TQT_SIGNAL( cancelClicked() ),
             this, TQT_SLOT( reject() ) );

    connect( m_value, TQT_SIGNAL( textChanged(const TQString&) ),
             this, TQT_SLOT( slotTextChanged(const TQString&) ) );

    enableButtonOK( true );
    resize( 350, 100 );
}

void KoCustomVarDialog::init()
{
    back = makeVBoxMainWidget();
    TQHBox *row1 = new TQHBox( back );
    row1->setSpacing( KDialog::spacingHint() );
    TQLabel *ln = new TQLabel( i18n( "Name:" ), row1 );
    ln->setFixedSize( ln->sizeHint() );
    m_name = new KLineEdit( row1 );

    TQHBox *row2 = new TQHBox( back );
    row2->setSpacing( KDialog::spacingHint() );
    TQLabel *lv = new TQLabel( i18n( "Value:" ), row2 );
    lv->setFixedSize( lv->sizeHint() );
    m_value = new KLineEdit( row2 );
}

void KoCustomVarDialog::slotAddOk()
{
    accept();
}
void KoCustomVarDialog::slotEditOk()
{
    m_var->setValue( m_value->text() );
    accept();
}

void KoCustomVarDialog::slotTextChanged(const TQString&text)
{
    enableButtonOK( !text.isEmpty() );
}
TQString KoCustomVarDialog::name()
{
    if ( m_name->text().isEmpty() )
        return TQString( "No name" );
    return m_name->text();
}

TQString KoCustomVarDialog::value()
{
    if ( m_value->text().isEmpty() )
        return TQString( "No value" );
    return m_value->text();
}
