/*
   This file is part of the KDE project
   Copyright (C) 2003 Tobias Koenig <tokoe@kde.org>
   Copyright (C) 2004 Tobias Koenig <tokoe@kde.org>
                      Dirk Schmidt <fs@dirk.schmidt.net>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef _KWMAILMERGE_KABC_H_
#define _KWMAILMERGE_KABC_H_

#include <tqdom.h>
#include <tqguardedptr.h>

#include <tdeabc/addressbook.h>

#include "KWMailMergeDataSource.h"

class KWMailMergeTDEABC: public KWMailMergeDataSource
{
    Q_OBJECT
  

public:
    KWMailMergeTDEABC( TDEInstance *inst, TQObject *parent );
    ~KWMailMergeTDEABC();

    /**
       Saves the mail merge list to the kword document.
     */
    virtual void save( TQDomDocument&, TQDomElement& );

    /**
       Loads the mail merge list stored in the kword document.
     */
    virtual void load( TQDomElement& );

    /**
       @param name	The name of the value e.g. "Family name".
       @param record	The position of the the entry in mail merge list.
       @return		The value of the mail merge variable.
       
       If @p record equals -1, @p name is returned.
     */
    virtual class TQString getValue( const class TQString &name, int record = -1 ) const;

    /**
       @return	The number of available contacts in mail merge list.
     */
    virtual int getNumRecords() const;

    /**
       Only for compatability reasons.
       
       @param force	Hasn't any effect.
     */
    virtual void refresh( bool force );

    /**
       Shows a KWMailMergeTDEABCConfig dialog for selecting entries from KAddressbook.
     */
    virtual bool showConfigDialog( TQWidget*, int action);

protected:
    friend class KWMailMergeTDEABCConfig;

    /**
       Adds an entry from TDEABC::StdAddressBook::self()
       to the mail merge list.
      
       To be called by KWMailMergeTDEABC::load() and 
       KWMailMergeTDEABCConfig::acceptSelection() only.
       
       @param uid	The entry's TDEABC::Addressee::uid().
     */
    void addEntry( const TQString &uid );

    /**
       Adds a distribution list to the mail merge list.
       
       To be called by KWMailMergeTDEABC::load() and 
       KWMailMergeTDEABCConfig::acceptSelection() only.
       
       @param id	The DistributionList::name().
     */
    void addList( const TQString &id );

    /**
       Removes all entries and distribution lists from the mail merge list.
     */
    void clear();

    /**
       @return	All selected DistributionList::name().
       
       To be called by KWMailMergeTDEABCConfig::initSelectedLists()
     */
    virtual TQStringList lists() const;

    /**
       @return	The TDEABC::Addressee::uid() of all individually selected 
       entries in mail merge list.
       
       To be called by KWMailMergeTDEABCConfig::initSelectedAddressees()
    */
    virtual TQStringList singleRecords() const;

private:
    /**
       The TDEABC::StdAddressBook::self().
     */
    TDEABC::AddressBook* _addressBook;

    /**
       Just an Iterator.
     */
    mutable TDEABC::AddressBook::ConstIterator _iterator;

    /**
       Just an Iterator.
     */
    mutable TQStringList::ConstIterator _UIDIterator;

    /**
       The "real" mail merge list. A list of TQStrings. Each represents 
       the TDEABC::Addressee::uid() of a KAdressbook entry.
       There is no UID twice in this list.
      
       Needed because selected contacts may appear in a selected 
       distribution list, too. And we don't want to print it multiple.
     */
    TQStringList _exclusiveUIDs;

    /**
       This list contains all the TDEABC::Addressee::uid() selected 
       individually with the KWMailMergeTDEABCConfig dialog.
     */
    TQStringList _individualUIDs;

    /**
       This list contains all the TDEABC::Addressee::uid() from the distribution 
       lists selected with the KWMailMergeTDEABCConfig dialog.
     */
    TQStringList _listUIDs;

    /**
       This list contains all the DistributionList::name() selected with the
       KWMailMergeTDEABCConfig dialog.
     */
    TQStringList _lists;


    /**
       Appends all TDEABC::Addressee::uid() of a distribution list to _listUIDs
       and updates the mail merge list.
       
       To be used by KWMailMergeTDEABCConfig::addList( const TQString &id )
       only.
       
       @param listName	The DistributionList::name() of the distribution list.
     */
    void parseList( const TQString& listName );

    /**
       Removes duplicate entries in the mail merge list. 
     */
    void makeUIDsExclusive();

};

#endif

