/***************************************************************************
 * This file is part of the KDE project
 * copyright (C) 2005 by Sebastian Sauer (mail@dipe.org)
 * copyright (C) 2005 by Tobi Krebs (tobi.krebs@gmail.com)
 * copyright (C) 2006 by Sascha Kupper (kusato@kfnv.de)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 ***************************************************************************/

#ifndef KOMACRO_ACTION_H
#define KOMACRO_ACTION_H

#include "manager.h"
#include "context.h"
#include "variable.h"

#include <tqobject.h>
#include <ksharedptr.h>
#include <tqstringlist.h>

namespace KoMacro {

	/**
	 * The Action class extendes TDEAction to implement some additional
	 * functionality TDEAction doesn't provide.
	 */
	class KOMACRO_EXPORT Action
		: public TQObject // TQt functionality like signals and slots
		, public TDEShared // shared reference-counting
	{
			Q_OBJECT
  

			/// Property to get/set the name.
			TQ_PROPERTY(TQString name READ name WRITE setName)

			/// Property to get/set the text.
			TQ_PROPERTY(TQString text READ text WRITE setText)

			/// Property to get/set the comment.
			TQ_PROPERTY(TQString comment READ comment WRITE setComment)

		public:

			/**
			* Shared pointer to implement reference-counting.
			*/
			typedef TQMap<TQString, TDESharedPtr<Action> > Map;

			/**
			* Constructor.
			* 
			* @param name The unique name this @a Action has.
			* @param text The i18n-caption text this @a Action has.
			*/
			explicit Action(const TQString& name, const TQString& text = TQString());

			/**
			* Destructor.
			*/
			virtual ~Action();

			/**
			* @return a string representation of the functionality
			* this action provides.
			*/
			virtual const TQString toString() const;

			/**
			* The name this @a Action has.
			*/
			const TQString name() const;
			
			/**
			* Set the name of the @a Action to @p name .
			*/
			void setName(const TQString& name); 

			/**
			* @return the i18n-caption text this @a Action has.
			*/
			const TQString text() const;

			/**
			* Set the i18n-caption text this @a Action has.
			*/
			void setText(const TQString& text);

			/**
			* @return the comment associated with this action.
			*/
			const TQString comment() const;

			/**
			* Set the @p comment associated with this action.
			*/
			void setComment(const TQString& comment);

			/**
			* @return true if there exists a variable with the
			* name @p name else false is returned.
			*/
			bool hasVariable(const TQString& name) const;

			/**
			* @return the variable @a Variable defined for the
			* name @p name . If there exists no @a Variable with
			* such a name, NULL is returned.
			*/
			TDESharedPtr<Variable> variable(const TQString& name) const;

			/**
			* @return the map of variables this @a Action provides.
			*/
			Variable::Map variables() const;

			/**
			* @return a list of variablenames this @a Action provides.s
			*/
			TQStringList variableNames() const;

			/**
			 * Append the @a Variable @p variable to list of variables
			 * this @a Action provides.
			 */
			void setVariable(TDESharedPtr<Variable> variable);

			/**
			 * Set the variable.
			 *
			 * @param name The name the variable should have.
			 * @param text The i18n-caption used for display.
			 * @param variant The TQVariant value.
			 */
			void setVariable(const TQString& name, const TQString& text, const TQVariant& variant);

			/**
			* Remove the variable defined with @p name . If there exists
			* no such variable, nothing is done.
			*/
			void removeVariable(const TQString& name);

			/**
			* This function is called, when the @a KoMacro::Variable
			* with name @p name used within the @a KoMacro::MacroItem
			* @p macroitem got changed.
			*
			* @param macroitem The @a KoMacro::MacroItem instance where
			* the variable defined with @p name is located in.
			* @param name The name the @a KoMacro::Variable has.
			* @return true if the update was successfully else false
			* is returned.
			*/
			virtual bool notifyUpdated(const TDESharedPtr<MacroItem> &macroitem, const TQString& name) {
				Q_UNUSED(macroitem);
				Q_UNUSED(name);
				return true; // The default implementation does nothing.
			}

		public slots:

			/**
			* Called if the @a Action should be executed within the
			* defined @p context .
			*/
			virtual void activate(TDESharedPtr<Context> context) = 0;

		private:
			/// @internal d-pointer class.
			class Private;
			/// @internal d-pointer instance.
			Private* const d;
	};

}

#endif
