/*
 *  Copyright (c) 1999-2000 Matthias Elter  <me@kde.org>
 *  Copyright (c) 2001 Toshitaka Fujioka  <fujioka@kde.org>
 *  Copyright (c) 2002 Patrick Julien <freak@codepimps.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#ifndef KIS_DOC_H_
#define KIS_DOC_H_

#include <kdebug.h>

#include <KoDocument.h>

#include "kis_types.h"
#include "kis_undo_adapter.h"

#include <koffice_export.h>

class TQImage;
class TQString;

class DCOPObject;
class KCommand;

class KoCommandHistory;
class KMacroCommand;

class KisProfile;
class KisView;
class KisNameServer;
class KisChildDoc;
class KisColorSpace;
class KisColor;
class KisCompositeOp;

class KRITACORE_EXPORT KisDoc : public KoDocument, private KisUndoAdapter {

    typedef KoDocument super;
    Q_OBJECT
  

public:
    KisDoc(TQWidget *parentWidget = 0, const char *widgetName = 0, TQObject* parent = 0, const char* name = 0, bool singleViewMode = false);
    virtual ~KisDoc();

public:
    // Overide KoDocument
    virtual bool wantExportConfirmation() const { return false; };
    virtual bool completeLoading(KoStore *store);
    virtual bool completeSaving(KoStore*);
    virtual DCOPObject* dcopObject();
    virtual bool initDoc(InitDocFlags flags, TQWidget* parentWidget=0);
    virtual bool loadOasis( const TQDomDocument&, KoOasisStyles&, const TQDomDocument&, KoStore* );
    virtual bool saveOasis( KoStore*, KoXmlWriter* );
    virtual bool loadChildren( KoStore* store);
    virtual bool loadXML(TQIODevice *, const TQDomDocument& doc);
    virtual TQCString mimeType() const;
    virtual TQWidget* createCustomDocumentWidget(TQWidget *parent);
    virtual KoDocument* hitTest(const TQPoint &pos, const TQWMatrix& matrix = TQWMatrix());

    /**
     * Draw the image embedded in another KOffice document
     *
     * XXX: Use of transparent, zoomX and zoomY is not supported
     *      by Chalk because we appear to be doing our zooming
     *      elsewhere. This may affect KOffice compatibility.
     */
    virtual void paintContent(TQPainter& painter, const TQRect& rect, bool /*transparent*/, double /*zoomX*/, double /*zoomY*/);

    virtual TQDomDocument saveXML();

public slots:
            
        
    /**
     * Initialize an empty document using default values
     * @since 1.5
     */
     virtual void initEmpty();

private: // Undo adapter
    
    virtual void setCommandHistoryListener(const KisCommandHistoryListener *);
    virtual void removeCommandHistoryListener(const KisCommandHistoryListener *);

    virtual KCommand * presentCommand();
    virtual void addCommand(KCommand *cmd);
    virtual void setUndo(bool undo);
    virtual bool undo() const;
    virtual void beginMacro(const TQString& macroName);
    virtual void endMacro();


public:


    TQ_INT32 undoLimit() const;
    void setUndoLimit(TQ_INT32 limit);

    TQ_INT32 redoLimit() const;
    void setRedoLimit(TQ_INT32 limit);

    /**
     * Create a new image that has this document as a parent and
     * replace the current image with this image.
     */
    bool newImage(const TQString& name, TQ_INT32 width, TQ_INT32 height, KisColorSpace * cs, const KisColor &bgColor, const TQString &imgDescription, const double imgResolution);

    /**
     * Create a new image that has this document as a parent and
     * replace the current image with this image.
     */
    KisImageSP newImage(const TQString& name, TQ_INT32 width, TQ_INT32 height, KisColorSpace * colorstrategy);

    void renameImage(const TQString& oldName, const TQString& newName);


    /**
     * Adds the specified child document to this document; this
     * is not done with KoDocument::insertChild() because that
     * is protected and cannot be called from KisView.
     */
    KisChildDoc * createChildDoc( const TQRect& rect, KoDocument* childDoc );

    /**
     * Makes an otherwise empty document ready for import/export
     */
    void prepareForImport();

    KisImageSP currentImage();

    /**
     * Set the current image to the specified image and turn undo on.
     */
    void setCurrentImage(KisImageSP image);

    KisUndoAdapter * undoAdapter() { return this; }

public slots:
    void slotImageUpdated();
    void slotImageUpdated(const TQRect& rect);
    void slotDocumentRestored();
    void slotCommandExecuted(KCommand *command);

signals:
    void docUpdated();
    void docUpdated(TQRect rect);
    void loadingFinished();

    /*
     * Emitted every time a command is added to the undo history, or executed
     * due to an undo or redo action.
     */
    void sigCommandExecuted();

protected:
    // Overide KoDocument
    virtual KoView* createViewInstance(TQWidget *parent, const char *name);

protected slots:
    // Overide KoDocument
    virtual void openExistingFile(const TQString& file);
    virtual void openTemplate(const TQString& file);

private slots:
    void slotUpdate(KisImageSP img, TQ_UINT32 x, TQ_UINT32 y, TQ_UINT32 w, TQ_UINT32 h);
    void slotIOProgress(TQ_INT8 percentage);
    
private:
    
    TQDomElement saveImage(TQDomDocument& doc, KisImageSP img);
    KisImageSP loadImage(const TQDomElement& elem);
    void loadLayers(const TQDomElement& element, KisImageSP img, KisGroupLayerSP parent);
    KisLayerSP loadLayer(const TQDomElement& elem, KisImageSP img);
    KisLayerSP loadPaintLayer(const TQDomElement& elem, KisImageSP img,
                TQString name, TQ_INT32 x, TQ_INT32 y, TQ_INT32 opacity, bool visible, bool locked,
                KisCompositeOp compositeOp);
    KisGroupLayerSP loadGroupLayer(const TQDomElement& elem, KisImageSP img,
                TQString name, TQ_INT32 x, TQ_INT32 y, TQ_INT32 opacity, bool visible, bool locked,
                KisCompositeOp compositeOp);
    KisAdjustmentLayerSP loadAdjustmentLayer(const TQDomElement& elem, KisImageSP img,
                                             TQString name, TQ_INT32 x, TQ_INT32 y, TQ_INT32 opacity, bool visible, bool locked,
                                             KisCompositeOp compositeOp);
    KisPartLayerSP loadPartLayer(const TQDomElement& elem, KisImageSP img,
                                  TQString name, TQ_INT32 x, TQ_INT32 y, TQ_INT32 opacity,
                                  bool visible, bool locked, KisCompositeOp compositeOp);
    bool init();

    void setIOSteps(TQ_INT32 nsteps);
    void IOCompletedStep();
    void IODone();

private:

    bool m_undo;
    KoCommandHistory *m_cmdHistory;
    TQPtrList<KisCommandHistoryListener> m_undoListeners;
    KisImageSP m_currentImage;
    DCOPObject *m_dcop;
    KisNameServer *m_nserver;
    KMacroCommand *m_currentMacro;
    TQ_INT32 m_macroNestDepth;
    TQ_INT32 m_conversionDepth;
    int m_ioProgressTotalSteps;
    int m_ioProgressBase;
    TQMap<KisLayerSP, TQString> m_layerFilenames; // temp storage during load
    
};

#endif // KIS_DOC_H_

