/*
 *  Copyright (c) 2006 Boudewijn Rempt <boud@valdyas.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#ifndef _KIS_FILTER_CONFIGURATION_H_
#define _KIS_FILTER_CONFIGURATION_H_

#include <tqstring.h>
#include <tqmap.h>
#include <tqvariant.h>
#include <kdebug.h>
#include "koffice_export.h"

class KisPreviewDialog;
class KisProgressDisplayInterface;
class KisFilterConfigWidget;
class TQWidget;

/**
 * A KisFilterConfiguration is the serializable representation of
 * the filter parameters. Filters can subclass this class to implement
 * direct accessors to properties, but properties not in the map will
 * not be serialized.
 */
class KRITACORE_EXPORT KisFilterConfiguration {

public:

    /**
     * Create a new filter config.
     */
    KisFilterConfiguration(const TQString & name, TQ_INT32 version)
        : m_name(name)
        , m_version(version) {}

    /**
     * Deep copy the filter configFile
     */
    KisFilterConfiguration(const KisFilterConfiguration & rhs);
            
public:

    /**
     * Fill the filter configuration object from the XML encoded representation in s.
     */
    virtual void fromXML(const TQString &);

    /**
     * Create a serialized version of this filter config
     */
    virtual TQString toString();

    /**
     * Get the unique, language independent name of the filter.
     */
    const TQString & name() const;

    /**
     * Get the version of the filter that has created this config
     */
    TQ_INT32 version() const;

    /**
     * Set the property with name to value.
     */
    virtual void setProperty(const TQString & name, const TQVariant & value);

    /**
     * Set value to the value associated with property name
     * @return false if the specified property did not exist.
     */
    virtual bool getProperty(const TQString & name, TQVariant & value);

    virtual TQVariant getProperty(const TQString & name);

    int getInt(const TQString & name, int def = 0);
    double getDouble(const TQString & name, double def = 0.0);
    bool getBool(const TQString & name, bool def = false);
    TQString getString(const TQString & name, TQString def = TQString());
    
private:
        void dump();
    
protected:

    TQString m_name;
    TQ_INT32 m_version;
    TQMap<TQString, TQVariant> m_properties;
    
};

#endif // _KIS_FILTER_CONFIGURATION_H_
