/***************************************************************************
*   Copyright (C) 2004 by                                                 *
*   Unai Garro (ugarro@users.sourceforge.net)                             *
*   Cyril Bosselut (bosselut@b1project.com)                               *
*   Jason Kivlighn (jkivlighn@gmail.com)                                  *
*                                                                         *
*   Copyright (C) 2006 Jason Kivlighn (jkivlighn@gmail.com)               *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/


#ifndef TQSQLRECIPEDB_H
#define TQSQLRECIPEDB_H

#include "backends/recipedb.h"

#include <ntqglobal.h>
#include <ntqobject.h>
#include <ntqsqldatabase.h>
#include <ntqimage.h>
#include <ntqfileinfo.h>
#include <ntqregexp.h>
#include <ntqstring.h>

#include "datablocks/recipe.h"
#include "datablocks/elementlist.h"
#include "datablocks/ingredientpropertylist.h"
#include "datablocks/unitratiolist.h"

/**
@author Unai Garro, Jason Kivlighn
*/
class TQSqlRecipeDB : public RecipeDB
{

	Q_OBJECT

protected:
	virtual TQString qsqlDriverPlugin() const { return TQString::null; }
	virtual TQSqlDriver *qsqlDriver() const { return 0; }
	virtual void createDB( void ) = 0;

	virtual void portOldDatabases( float version );
	virtual void storePhoto( int recipeID, const TQByteArray &data );
	virtual void loadPhoto( int recipeID, TQPixmap &photo );
	void loadRecipeMetadata( Recipe *recipe );

	void search( RecipeList *list, int items, const RecipeSearchParameters & );

	/** Return the next id for the given table and column.
	  * If the database supports getting this afterwards,
	  * leave the default implementation which returns -1.
	  *
	  * Note: Only call when an insert is actually going to take place.
	  *       This function will increment the sequence counter.
	  */
	virtual int getNextInsertID( const TQString & /*table*/, const TQString & /*column*/ )
	{
		return -1;
	}

	TQSqlDatabase *database;
	TQSqlQuery m_query;
	TQString DBuser;
	TQString DBpass;
	TQString DBhost;
	int DBport;

public:
	TQSqlRecipeDB( const TQString &host, const TQString &user = TQString::null, const TQString &pass = TQString::null, const TQString &DBName = DEFAULT_DB_NAME, int port = 0 );
	~TQSqlRecipeDB( void );

	void connect( bool create_db, bool create_tables );

	void addIngredientWeight( const Weight & );
	void addProperty( const TQString &name, const TQString &units );
	void addPropertyToIngredient( int ingredientID, int propertyID, double amount, int perUnitsID );
	void addUnitToIngredient( int ingredientID, int unitID );

	void categorizeRecipe( int recipeID, const ElementList &categoryList );
	void changePropertyAmountToIngredient( int ingredientID, int propertyID, double amount, int per_units );

	void createNewAuthor( const TQString &authorName );
	void createNewCategory( const TQString &categoryName, int parent_id = -1 );
	void createNewIngGroup( const TQString &name );
	void createNewIngredient( const TQString &ingredientName );
	void createNewPrepMethod( const TQString &prepMethodName );
	void createNewRating( const TQString &name );
	void createNewUnit( const Unit &unit );
	void createNewYieldType( const TQString &type );

	void emptyData( void );
	void empty( void );

	int findExistingAuthorByName( const TQString& name );
	int findExistingCategoryByName( const TQString& name );
	int findExistingIngredientGroupByName( const TQString& name );
	int findExistingIngredientByName( const TQString& name );
	int findExistingPrepByName( const TQString& name );
	int findExistingRecipeByName( const TQString& name );
	int findExistingRatingByName( const TQString& name );
	int findExistingUnitByName( const TQString& name );
	int findExistingPropertyByName( const TQString& name );
	int findExistingYieldTypeByName( const TQString& name );
	void findIngredientUnitDependancies( int ingredientID, int unitID, ElementList *recipes, ElementList *ingredientInfo );
	void findIngredientDependancies( int ingredientID, ElementList *recipes );
	void findPrepMethodDependancies( int prepMethodID, ElementList *recipes );
	void findUnitDependancies( int unitID, ElementList *properties, ElementList *recipes, ElementList *weights );
	void findUseOfIngGroupInRecipes( ElementList *results, int groupID );
	void findUseOfCategoryInRecipes( ElementList *results, int catID );
	void findUseOfAuthorInRecipes( ElementList *results, int authorID );

	TQString getUniqueRecipeTitle( const TQString &recipe_title );

	bool ingredientContainsProperty( int ingredientID, int propertyID, int perUnitsID );
	bool ingredientContainsUnit( int ingredientID, int unitID );

	void loadAuthors( ElementList *list, int limit = -1, int offset = 0 );
	void loadCategories( CategoryTree *list, int limit = -1, int offset = 0, int parent_id = -1, bool recurse = true );
	void loadCategories( ElementList *list, int limit = -1, int offset = 0 );
	void loadIngredientGroups( ElementList *list );
	void loadIngredients( ElementList *list, int limit = -1, int offset = 0 );
	void loadPossibleUnits( int ingredientID, UnitList *list );
	void loadPrepMethods( ElementList *list, int limit = -1, int offset = 0 );
	void loadProperties( IngredientPropertyList *list, int ingredientID = -2 ); // Loads the list of possible properties by default, all the ingredient properties with -1, and the ingredients of given property if id>=0
	void loadRatingCriterion( ElementList *list, int limit = -1, int offset = 0 );
	void loadRecipes( RecipeList *, int items = All, TQValueList<int> ids = TQValueList<int>() );
	void loadRecipeList( ElementList *list, int categoryID = -1, bool recursive = false );
	void loadUncategorizedRecipes( ElementList *list );
	void loadUnits( UnitList *list, Unit::Type = Unit::All, int limit = -1, int offset = 0 );
	void loadUnitRatios( UnitRatioList *ratioList, Unit::Type );
	void loadYieldTypes( ElementList *list, int limit, int offset );

	void mergeAuthors( int id1, int id2 );
	void mergeCategories( int id1, int id2 );
	void mergeIngredientGroups( int id1, int id2 );
	void mergeIngredients( int id1, int id2 );
	void mergeUnits( int id1, int id2 );
	void mergePrepMethods( int id1, int id2 );
	void mergeProperties( int id1, int id2 );

	void modIngredientGroup( int ingredientID, const TQString &newLabel );
	/**
	* set newLabel for ingredientID
	*/
	void modIngredient( int ingredientID, const TQString &newLabel );
	/**
	* set newLabel for unitID
	*/
	void modUnit( const Unit &unit );
	/**
	* set newLabel for categoryID
	*/
	void modCategory( int categoryID, const TQString &newLabel );
	void modCategory( int categoryID, int new_parent_id );
	/**
	* set newLabel for authorID
	*/
	void modAuthor( int authorID, const TQString &newLabel );

	void modPrepMethod( int prepMethodID, const TQString &newLabel );

	void modProperty( int propertyID, const TQString &newLabel );

	TQString recipeTitle( int recipeID );

	void removeAuthor( int categoryID );
	void removeCategory( int categoryID );
	void removeIngredientGroup( int groupID );
	void removeIngredient( int ingredientID );
	void removeIngredientWeight( int id );
	void removePrepMethod( int prepMethodID );
	void removeProperty( int propertyID );
	void removePropertyFromIngredient( int ingredientID, int propertyID, int perUnitID );
	void removeRecipe( int id );
	void removeRecipeFromCategory( int ingredientID, int categoryID );
	void removeUnit( int unitID );
	void removeUnitFromIngredient( int ingredientID, int unitID );
	void removeUnitRatio( int unitID1, int unitID2 );

	void saveRecipe( Recipe *recipe );
	void saveUnitRatio( const UnitRatio *ratio );

	double unitRatio( int unitID1, int unitID2 );
	double ingredientWeight( const Ingredient &ing, bool *wasApproximated = 0 );
	WeightList ingredientWeightUnits( int ingID );

	TQString escapeAndEncode( const TQString &s ) const;
	TQString unescapeAndDecode( const TQCString &s ) const;

	TQString categoryName( int ID );
	TQString prepMethodName( int ID );
	TQString ingredientName( int ID );
	IngredientProperty propertyName( int ID );
	Unit unitName( int ID );

	int getCount( const TQString &table_name );
	int categoryTopLevelCount();

	bool checkIntegrity( void );

	void splitCommands( TQString& s, TQStringList& sl );

	float databaseVersion( void );

protected:
	void execSQL( const TQString &command );

private:
	void loadElementList( ElementList *elList, TQSqlQuery *query );
	void loadPropertyElementList( ElementList *elList, TQSqlQuery *query );
	TQString getNextInsertIDStr( const TQString &table, const TQString &column );

	TQString DBname;
	const TQString connectionName;
	TQString m_command;

	static int m_refCount;
};




#endif
