/***************************************************************************
 *   Copyright (C) 2005-2006 Nicolas Hadacek <hadacek@kde.org>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#ifndef PURL_H
#define PURL_H

#include "common/global/global.h"
#  include <tqhttp.h>
#include "common/global/log.h"
#include "common/common/purl_base.h"

namespace PURL
{
//----------------------------------------------------------------------------
class Http : public TQHttp
{
Q_OBJECT
  
public:
  Http(const TQString &hostname);
  TQHttpResponseHeader _header;

private slots:
  void responseHeaderReceivedSlot(const TQHttpResponseHeader &rh) { _header = rh; }
};

class Url;
class Directory;
class Private;

enum SeparatorType { UnixSeparator, WindowsSeparator };

//----------------------------------------------------------------------------
class Base
{
public:
  Base(const TQString &filepath = TQString());
  Base(const KURL &url);
  bool operator <(const Base &url) const { return _url<url._url; }
  bool operator ==(const Base &url) const;
  bool operator !=(const Base &url) const { return !(_url==url._url); }
  const KURL &kurl() const { return _url; }
  TQString pretty() const;
  bool isEmpty() const { return _url.isEmpty(); }
  bool isLocal() const;
  TQString path(SeparatorType type = UnixSeparator) const;             // with ending '/' unless empty path
  TQString unterminatedPath(SeparatorType type = UnixSeparator) const; // no ending '/'
  Directory directory() const;
  bool isInto(const Directory &dir) const;
  bool isRelative() const { return _relative; }
  bool exists(TQDateTime *lastModified = 0) const;

protected:
  bool _relative;
  KURL _url;
  static Private *_private;

private:
  bool httpUrlExists(bool *ok = 0) const;
};

//----------------------------------------------------------------------------
class Url : public Base
{
public:
  Url() {}
  Url(const KURL &url) : Base(url) {}
  // add correct extension if filename has no extension
  Url(const Directory &path, const TQString &filename, FileType type);
  Url(const Directory &path, const TQString &filepath);
  static Url fromPathOrUrl(const TQString &s);

  Url toFileType(FileType type) const { return toExtension(type.data().extensions[0]); }
  Url toExtension(const TQString &extension) const;
  Url appendExtension(const TQString &extension) const;

  const FileType::Data &data() const { return fileType().data(); }
  FileType fileType() const;
  TQString basename() const; // filename without extension
  TQString filename() const; // filename without path
  TQString filepath(SeparatorType type = UnixSeparator) const; // filename with path
  TQString relativeTo(const Directory &dir, SeparatorType type = UnixSeparator) const;
  Url toAbsolute(const Directory &dir) const;
#if !defined(NO_KDE)
  bool isDosFile() const;
  bool create(Log::Generic &log) const; // do not overwrite
  bool write(const TQString &text, Log::Generic &log) const;
  bool copyTo(const Url &destination, Log::Generic &log) const; // do not overwrite
  bool del(Log::Generic &log) const;
#endif

private:
  Url(const TQString &filepath) : Base(filepath) {}
};

extern bool findExistingUrl(Url &url); // may transform extension's case if needed

//----------------------------------------------------------------------------
class UrlList : public TQValueList<Url>
{
public:
  UrlList() {}
  UrlList(const Url &url) { append(url); }
  UrlList(const TQValueList<Url> &list) : TQValueList<Url>(list) {}
#if !defined(NO_KDE)
  UrlList(const KURL::List &list);
#endif
};

//----------------------------------------------------------------------------
class Directory : public Base
{
public:
  Directory(const TQString &path = TQString());
  TQStringList files(const TQString &filter) const;
  Url findMatchingFilename(const TQString &filename) const;
  Directory up() const;
  Directory down(const TQString &path) const;
  static Directory current();
#if !defined(NO_KDE)
  bool create(Log::Generic &log) const;
#endif
};

} // namespace

#endif
