/*
    Kopete Yahoo Protocol

    Copyright (c) 2005-2006 André Duffeck <duffeck@kde.org>
    Copyright (c) 2004 Duncan Mac-Vicar P. <duncan@kde.org>
    Copyright (c) 2004 Matt Rogers <matt.rogers@kdemail.net>
    Copyright (c) 2004 SuSE Linux AG <http://www.suse.com>
    Copyright (C) 2003  Justin Karneges <justin@affinix.com>

    Kopete (c) 2002-2006 by the Kopete developers <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This library is free software; you can redistribute it and/or         *
    * modify it under the terms of the GNU Lesser General Public            *
    * License as published by the Free Software Foundation; either          *
    * version 2 of the License, or (at your option) any later version.      *
    *                                                                       *
    *************************************************************************
*/

#include <tqtimer.h>
#include <tqpixmap.h>

#include <kdebug.h>
#include <tdesocketbase.h>

#include "yahooclientstream.h"
#include "yahooconnector.h"
#include "task.h"
#include "logintask.h"
#include "listtask.h"
#include "statusnotifiertask.h"
#include "mailnotifiertask.h"
#include "messagereceivertask.h"
#include "sendnotifytask.h"
#include "sendmessagetask.h"
#include "logofftask.h"
#include "changestatustask.h"
#include "modifybuddytask.h"
#include "picturenotifiertask.h"
#include "requestpicturetask.h"
#include "stealthtask.h"
#include "sendpicturetask.h"
#include "webcamtask.h"
#include "conferencetask.h"
#include "sendauthresptask.h"
#include "pingtask.h"
#include "yabtask.h"
#include "modifyyabtask.h"
#include "chatsessiontask.h"
#include "sendfiletask.h"
#include "filetransfernotifiertask.h"
#include "receivefiletask.h"
#include "yahoochattask.h"
#include "client.h"
#include "yahootypes.h"
#include "yahoobuddyiconloader.h"

using namespace KNetwork;

class Client::ClientPrivate
{
public:
	ClientPrivate() {}

	ClientStream *stream;
	int id_seed;
	Task *root;
	TQString host, user, pass;
	uint port;
	bool active;
	YahooBuddyIconLoader *iconLoader;
	int error;
	TQString errorString;
	TQString errorInformation;

	// tasks
	bool tasksInitialized;
	LoginTask * loginTask;
	ListTask *listTask;
	StatusNotifierTask *statusTask;
	MailNotifierTask *mailTask;
	MessageReceiverTask *messageReceiverTask;
	PictureNotifierTask *pictureNotifierTask;
	WebcamTask *webcamTask;
	ConferenceTask *conferenceTask;
	YABTask *yabTask;
	FileTransferNotifierTask *fileTransferTask;
	YahooChatTask *yahooChatTask;
	ReceiveFileTask *receiveFileTask;

	// Connection data
	uint sessionID;
	TQString yCookie;
	TQString tCookie;
	TQString cCookie;
	Yahoo::Status status;
	Yahoo::Status statusOnConnect;
	TQString statusMessageOnConnect;
	Yahoo::PictureStatus pictureFlag;
	int pictureChecksum;
	bool buddyListReady;
	TQStringList pictureRequestQueue;
};

Client::Client(TQObject *par) :TQObject(par, "yahooclient")
{
	d = new ClientPrivate;
/*	d->tzoffset = 0;*/
	d->active = false;

	d->root = new Task(this, true);
	d->statusOnConnect = Yahoo::StatusAvailable;
	setStatus( Yahoo::StatusDisconnected );
	d->tasksInitialized = false;
	d->stream = 0L;
	d->iconLoader = 0L;
	d->loginTask = new LoginTask( d->root );
	d->listTask = new ListTask( d->root );
	d->pictureFlag = Yahoo::NoPicture;
	d->buddyListReady = false;
	m_connector = 0L;

	m_pingTimer = new TQTimer( this );
	TQObject::connect( m_pingTimer, TQT_SIGNAL( timeout() ), this, TQT_SLOT( sendPing() ) );

	TQObject::connect( d->loginTask, TQT_SIGNAL( haveSessionID( uint ) ), TQT_SLOT( lt_gotSessionID( uint ) ) );
	TQObject::connect( d->loginTask, TQT_SIGNAL( buddyListReady() ), TQT_SLOT( processPictureQueue() ) );
	TQObject::connect( d->loginTask, TQT_SIGNAL( loginResponse( int, const TQString& ) ),
				TQT_SLOT( slotLoginResponse( int, const TQString& ) ) );
	TQObject::connect( d->loginTask, TQT_SIGNAL( haveCookies() ), TQT_SLOT( slotGotCookies() ) );
	TQObject::connect( d->listTask, TQT_SIGNAL( gotBuddy(const TQString &, const TQString &, const TQString &) ),
					TQT_SIGNAL( gotBuddy(const TQString &, const TQString &, const TQString &) ) );
	TQObject::connect( d->listTask, TQT_SIGNAL( stealthStatusChanged( const TQString&, Yahoo::StealthStatus ) ),
					TQT_SIGNAL( stealthStatusChanged( const TQString&, Yahoo::StealthStatus ) ) );
}

Client::~Client()
{
	close();
	delete d->iconLoader;
	delete d->root;
	delete d;
}

void Client::connect( const TQString &host, const uint port, const TQString &userId, const TQString &pass )
{
	kdDebug(YAHOO_GEN_DEBUG) << k_funcinfo << endl;
	d->host = host;
	d->port = port;
	d->user = userId;
	d->pass = pass;
	setStatus( Yahoo::StatusConnecting );

	m_connector = new KNetworkConnector;
	m_connector->setOptHostPort( host, port );
	d->stream = new ClientStream( m_connector, this );
	TQObject::connect( d->stream, TQT_SIGNAL( connected() ), this, TQT_SLOT( cs_connected() ) );
	TQObject::connect( d->stream, TQT_SIGNAL( error(int) ), this, TQT_SLOT( streamError(int) ) );
	TQObject::connect( d->stream, TQT_SIGNAL( readyRead() ), this, TQT_SLOT( streamReadyRead() ) );
	TQObject::connect( d->stream, TQT_SIGNAL( connectionClosed() ), this, TQT_SLOT( streamDisconnected() ) );

	d->stream->connectToServer( host, false );
}

void Client::cancelConnect()
{
	d->loginTask->reset();
}

void Client::cs_connected()
{
	kdDebug(YAHOO_GEN_DEBUG) << k_funcinfo << endl;
	emit connected();
	kdDebug(YAHOO_RAW_DEBUG) << " starting login task ... " << endl;

	d->loginTask->setStateOnConnect( (d->statusOnConnect == Yahoo::StatusInvisible) ? Yahoo::StatusInvisible : Yahoo::StatusAvailable );
	d->loginTask->go();
	d->active = true;
}

void Client::close()
{
	kdDebug(YAHOO_GEN_DEBUG) << k_funcinfo << endl;
	m_pingTimer->stop();
	if( d->active )
	{
		LogoffTask *lt = new LogoffTask( d->root );
		lt->go( true );
	}
	if( d->tasksInitialized)
		deleteTasks();
	d->loginTask->reset();
	if( d->stream ) {
		TQObject::disconnect( d->stream, TQT_SIGNAL( readyRead() ), this, TQT_SLOT( streamReadyRead() ) );
		d->stream->deleteLater();
	}
	d->stream = 0L;
	if( m_connector )
		m_connector->deleteLater();
	m_connector = 0L;
	d->active = false;
	d->buddyListReady = false;
}

int Client::error()
{
	return d->error;
}

TQString Client::errorString()
{
	return d->errorString;
}

TQString Client::errorInformation()
{
	return d->errorInformation;
}

// SLOTS //
void Client::streamError( int error )
{
	kdDebug(YAHOO_RAW_DEBUG) << "CLIENT ERROR (Error " <<  error << ")" << endl;
	TQString msg;

	d->active = false;

	// Examine error
	if( error == ClientStream::ErrConnection && m_connector )			// Ask Connector in this case
	{
		d->error = m_connector->errorCode();
		d->errorString = TDESocketBase::errorString( (TDESocketBase::SocketError)d->error );
	}
	else if( d->stream )
	{
		d->error = error;
		d->errorString = d->stream->errorText();
	}
	close();
	if( status() == Yahoo::StatusConnecting )
		emit loginFailed();
	else
		emit disconnected();
}

void Client::streamReadyRead()
{
	// take the incoming transfer and distribute it to the task tree
	Transfer * transfer = d->stream->read();
	distribute( transfer );
}

void Client::streamDisconnected()
{
	d->active = false;
	emit disconnected();
}

void Client::lt_loginFinished()
{
	kdDebug(YAHOO_GEN_DEBUG) << k_funcinfo << endl;

	slotLoginResponse( d->loginTask->statusCode(), d->loginTask->statusString() );
}

void Client::slotLoginResponse( int response, const TQString &msg )
{
	if( response == Yahoo::LoginOk )
	{
		if( !(d->statusOnConnect == Yahoo::StatusAvailable ||
				d->statusOnConnect == Yahoo::StatusInvisible) ||
				!d->statusMessageOnConnect.isEmpty() )
			changeStatus( d->statusOnConnect, d->statusMessageOnConnect, Yahoo::StatusTypeAway );
		d->statusMessageOnConnect = TQString();
		setStatus( d->statusOnConnect );
		/* only send a ping every hour. we get disconnected otherwise */
		m_pingTimer->start( 60 * 60 * 1000 );
		initTasks();
	} else {
		d->active = false;
		close();
	}

	kdDebug(YAHOO_RAW_DEBUG) << "Emitting loggedIn" << endl;
	emit loggedIn( response, msg );
}

void Client::lt_gotSessionID( uint id )
{
	kdDebug(YAHOO_RAW_DEBUG) << "Got SessionID: " << id << endl;
	d->sessionID = id;
}

void Client::slotGotCookies()
{
	kdDebug(YAHOO_RAW_DEBUG) << "Y: " << d->loginTask->yCookie()
					<< " T: " << d->loginTask->tCookie()
					<< " C: " << d->loginTask->cCookie() << endl;
	d->yCookie = d->loginTask->yCookie();
	d->tCookie = d->loginTask->tCookie();
	d->cCookie = d->loginTask->cCookie();
}

// INTERNALS //

// ***** Messaging handling *****
void Client::sendTyping( const TQString &who, bool typing )
{
	SendNotifyTask *snt = new SendNotifyTask( d->root );
	snt->setTarget( who );
	snt->setState( typing ? SendNotifyTask::Active : SendNotifyTask::NotActive );
	snt->setType( SendNotifyTask::NotifyTyping );
	snt->go( true );
}

void Client::sendWebcamInvite( const TQString &who )
{
	if( !d->webcamTask->transmitting() )
		d->webcamTask->registerWebcam();

	d->webcamTask->addPendingInvitation( who );
}

void Client::sendMessage( const TQString &to, const TQString &msg )
{
	SendMessageTask *smt = new SendMessageTask( d->root );
	smt->setTarget( to );
	smt->setText( msg );
	smt->setPicureFlag( pictureFlag() );
	smt->go( true );
}

void Client::setChatSessionState( const TQString &to, bool close )
{
	ChatSessionTask *cst = new ChatSessionTask( d->root );
	cst->setTarget( to );
	cst->setType( close ? ChatSessionTask::UnregisterSession : ChatSessionTask::RegisterSession );
	cst->go( true );
}

void Client::sendBuzz( const TQString &to )
{
	SendMessageTask *smt = new SendMessageTask( d->root );
	smt->setTarget( to );
	smt->setText( TQString::fromLatin1( "<ding>" ) );
	smt->setPicureFlag( pictureFlag() );
	smt->go( true );
}

void Client::sendFile( unsigned int transferId, const TQString &to, const TQString &msg, KURL url )
{
	SendFileTask *sft = new SendFileTask( d->root );

	TQObject::connect( sft, TQT_SIGNAL(complete(unsigned int)), TQT_SIGNAL(fileTransferComplete(unsigned int)) );
	TQObject::connect( sft, TQT_SIGNAL(bytesProcessed(unsigned int, unsigned int)), TQT_SIGNAL(fileTransferBytesProcessed(unsigned int, unsigned int)) );
	TQObject::connect( sft, TQT_SIGNAL(error(unsigned int, int, const TQString &)), TQT_SIGNAL(fileTransferError(unsigned int, int, const TQString &)) );

	TQObject::connect( this, TQT_SIGNAL(fileTransferCanceled( unsigned int )), sft, TQT_SLOT(canceled( unsigned int )) );

	sft->setTarget( to );
	sft->setMessage( msg );
	sft->setFileUrl( url );
	sft->setTransferId( transferId );
	sft->go( true );
}

void Client::receiveFile( unsigned int transferId, const TQString &userId, KURL remoteURL, KURL localURL )
{
	ReceiveFileTask *rft = new ReceiveFileTask( d->root );

	TQObject::connect( rft, TQT_SIGNAL(complete(unsigned int)), TQT_SIGNAL(fileTransferComplete(unsigned int)) );
	TQObject::connect( rft, TQT_SIGNAL(bytesProcessed(unsigned int, unsigned int)), TQT_SIGNAL(fileTransferBytesProcessed(unsigned int, unsigned int)) );
	TQObject::connect( rft, TQT_SIGNAL(error(unsigned int, int, const TQString &)), TQT_SIGNAL(fileTransferError(unsigned int, int, const TQString &)) );
	TQObject::connect( this, TQT_SIGNAL(fileTransferCanceled( unsigned int )), rft, TQT_SLOT(canceled( unsigned int )) );

	rft->setRemoteUrl( remoteURL );
	rft->setLocalUrl( localURL );
	rft->setTransferId( transferId );
	rft->setUserId( userId );
	if( remoteURL.url().startsWith( "http://" ) )
		rft->setType( ReceiveFileTask::FileTransferAccept );
	else
		rft->setType( ReceiveFileTask::FileTransfer7Accept );
	rft->go( true );
}

void Client::rejectFile( const TQString &userId, KURL remoteURL )
{
	if( remoteURL.url().startsWith( "http://" ) )
		return;

	ReceiveFileTask *rft = new ReceiveFileTask( d->root );

	rft->setRemoteUrl( remoteURL );
	rft->setUserId( userId );
	rft->setType( ReceiveFileTask::FileTransfer7Reject );
	rft->go( true );
}

void Client::cancelFileTransfer( unsigned int transferId )
{
	emit fileTransferCanceled( transferId );
}

void Client::changeStatus( Yahoo::Status status, const TQString &message, Yahoo::StatusType type )
{
	kdDebug(YAHOO_RAW_DEBUG) << "status: " << status
					<< " message: " << message
					<< " type: " << type << endl;
	ChangeStatusTask *cst = new ChangeStatusTask( d->root );
	cst->setStatus( status );
	cst->setMessage( message );
	cst->setType( type );
	cst->go( true );

	if( status == Yahoo::StatusInvisible )
		stealthContact( TQString(), Yahoo::StealthOnline, Yahoo::StealthClear );

	setStatus( status );
}

void Client::sendAuthReply( const TQString &userId, bool accept, const TQString &msg )
{
	SendAuthRespTask *sarp = new SendAuthRespTask( d->root );
	sarp->setGranted( accept );
	sarp->setTarget( userId );
	sarp->setMessage( msg );
	sarp->go( true );
}

void Client::sendPing()
{
	if( !d->active )
	{
		kdDebug(YAHOO_RAW_DEBUG) << "Disconnected. NOT sending a PING." << endl;
		return;
	}
	kdDebug(YAHOO_RAW_DEBUG) << "Sending a PING." << endl;
	PingTask *pt = new PingTask( d->root );
	pt->go( true );
}

// ***** Contactlist handling *****

void Client::stealthContact(TQString const &userId, Yahoo::StealthMode mode, Yahoo::StealthStatus state)
{
	StealthTask *st = new StealthTask( d->root );
	st->setTarget( userId );
	st->setState( state );
	st->setMode( mode );
	st->go( true );
}

void Client::addBuddy( const TQString &userId, const TQString &group, const TQString &message )
{
	ModifyBuddyTask *mbt = new ModifyBuddyTask( d->root );

	TQObject::connect(mbt, TQT_SIGNAL(buddyAddResult( const TQString &, const TQString &, bool )),
			 TQT_SIGNAL(buddyAddResult( const TQString &, const TQString &, bool)));

	mbt->setType( ModifyBuddyTask::AddBuddy );
	mbt->setTarget( userId );
	mbt->setGroup( group );
	mbt->setMessage( message );
	mbt->go( true );
}

void Client::removeBuddy( const TQString &userId, const TQString &group )
{
	ModifyBuddyTask *mbt = new ModifyBuddyTask( d->root );

	TQObject::connect(mbt, TQT_SIGNAL(buddyRemoveResult( const TQString &, const TQString &, bool )),
			 TQT_SIGNAL(buddyRemoveResult( const TQString &, const TQString &, bool)));

	mbt->setType( ModifyBuddyTask::RemoveBuddy );
	mbt->setTarget( userId );
	mbt->setGroup( group );
	mbt->go( true );
}

void Client::moveBuddy( const TQString &userId, const TQString &oldGroup, const TQString &newGroup )
{
	ModifyBuddyTask *mbt = new ModifyBuddyTask( d->root );

	TQObject::connect(mbt, TQT_SIGNAL(buddyChangeGroupResult( const TQString &, const TQString &, bool )),
			 TQT_SIGNAL(buddyChangeGroupResult( const TQString &, const TQString &, bool)));

	mbt->setType( ModifyBuddyTask::MoveBuddy );
	mbt->setTarget( userId );
	mbt->setOldGroup( oldGroup );
	mbt->setGroup( newGroup );
	mbt->go( true );
}

// ***** Buddyicon handling *****

void Client::processPictureQueue()
{
	kdDebug(YAHOO_RAW_DEBUG) << k_funcinfo << endl;
	d->buddyListReady = true;
	if( d->pictureRequestQueue.isEmpty() )
	{
		return;
	}

	requestPicture( d->pictureRequestQueue.front() );
	d->pictureRequestQueue.pop_front();


	if( !d->pictureRequestQueue.isEmpty() )
	{
		TQTimer::singleShot( 1000, this, TQT_SLOT(processPictureQueue()) );
	}
}

void Client::requestPicture( const TQString &userId )
{
	if( !d->buddyListReady )
	{
		d->pictureRequestQueue << userId;
		return;
	}

	RequestPictureTask *rpt = new RequestPictureTask( d->root );
	rpt->setTarget( userId );
	rpt->go( true );
}

void Client::downloadPicture(  const TQString &userId, KURL url, int checksum )
{
	if( !d->iconLoader )
	{
		d->iconLoader = new YahooBuddyIconLoader( this );
		TQObject::connect( d->iconLoader, TQT_SIGNAL(fetchedBuddyIcon(const TQString&, const TQByteArray &, int )),
				TQT_SIGNAL(pictureDownloaded(const TQString&, const TQByteArray &,  int ) ) );
	}

	d->iconLoader->fetchBuddyIcon( TQString(userId), KURL(url), checksum );
}

void Client::uploadPicture( KURL url )
{
	kdDebug(YAHOO_RAW_DEBUG) << "URL: " << url.url() << endl;
	SendPictureTask *spt = new SendPictureTask( d->root );
	spt->setType( SendPictureTask::UploadPicture );
	spt->setFilename( url.fileName() );
	if ( url.isLocalFile() )
		spt->setPath( url.path() ); // FIXME: to test if is what we want
	else
		spt->setPath( url.url() );
	spt->go( true );
}

void Client::sendPictureChecksum( const TQString &userId, int checksum )
{
	kdDebug(YAHOO_RAW_DEBUG) << "checksum: " << checksum << endl;
	SendPictureTask *spt = new SendPictureTask( d->root );
	spt->setType( SendPictureTask::SendChecksum );
	spt->setChecksum( checksum );
	if( !userId.isEmpty() )
		spt->setTarget( userId );
	spt->go( true );
}

void Client::sendPictureInformation( const TQString &userId, const TQString &url, int checksum )
{
	kdDebug(YAHOO_RAW_DEBUG) << "checksum: " << checksum << endl;
	SendPictureTask *spt = new SendPictureTask( d->root );
	spt->setType( SendPictureTask::SendInformation );
	spt->setChecksum( checksum );
	spt->setUrl( url );
	spt->setTarget( userId );
	spt->go( true );
}

void Client::setPictureStatus( Yahoo::PictureStatus status )
{
	if( d->pictureFlag == status )
		return;

	kdDebug(YAHOO_RAW_DEBUG) << "Setting PictureStatus to: " << status << endl;
	d->pictureFlag = status;
	SendPictureTask *spt = new SendPictureTask( d->root );
	spt->setType( SendPictureTask::SendStatus );
	spt->setStatus( status );
	spt->go( true );
}

// ***** Webcam handling *****

void Client::requestWebcam( const TQString &userId )
{
	d->webcamTask->requestWebcam( userId );
}

void Client::closeWebcam( const TQString &userId )
{
	d->webcamTask->closeWebcam( userId );
}

void Client::sendWebcamImage( const TQByteArray &ar )
{
	d->webcamTask->sendWebcamImage( ar );
}

void Client::closeOutgoingWebcam()
{
	d->webcamTask->closeOutgoingWebcam();
}


void Client::grantWebcamAccess( const TQString &userId )
{
	d->webcamTask->grantAccess( userId );
}

// ***** Conferences *****
void Client::inviteConference( const TQString &room, const TQStringList &members, const TQString &msg )
{
	d->conferenceTask->inviteConference( room, members, msg );
}

void Client::addInviteConference( const TQString &room, const TQStringList &who, const TQStringList &members, const TQString &msg )
{
	d->conferenceTask->addInvite( room, who, members, msg );
}

void Client::joinConference( const TQString &room, const TQStringList &members )
{
	d->conferenceTask->joinConference( room, members );
}

void Client::declineConference( const TQString &room, const TQStringList &members, const TQString &msg )
{
	d->conferenceTask->declineConference( room, members, msg );
}

void Client::leaveConference( const TQString &room, const TQStringList &members )
{
	d->conferenceTask->leaveConference( room, members );
}

void Client::sendConferenceMessage( const TQString &room, const TQStringList &members, const TQString &msg )
{
	d->conferenceTask->sendMessage( room, members, msg );
}

// ***** YAB *****
void Client::getYABEntries( long lastMerge, long lastRemoteRevision )
{
	d->yabTask->getAllEntries( lastMerge, lastRemoteRevision);
}

void Client::saveYABEntry( YABEntry &entry )
{
	ModifyYABTask *myt = new ModifyYABTask( d->root );
	myt->setAction( ModifyYABTask::EditEntry );
	myt->setEntry( entry );
	TQObject::connect( myt, TQT_SIGNAL(gotEntry( YABEntry * )), this, TQT_SIGNAL( gotYABEntry( YABEntry * ) ) );
	TQObject::connect( myt, TQT_SIGNAL(error( YABEntry *, const TQString &)), this, TQT_SIGNAL(modifyYABEntryError( YABEntry *, const TQString & )));
	myt->go(true);
}

void Client::addYABEntry(  YABEntry &entry )
{
	ModifyYABTask *myt = new ModifyYABTask( d->root );
	myt->setAction( ModifyYABTask::AddEntry );
	myt->setEntry( entry );
	TQObject::connect( myt, TQT_SIGNAL(gotEntry( YABEntry * )), this, TQT_SIGNAL( gotYABEntry( YABEntry * ) ) );
	TQObject::connect( myt, TQT_SIGNAL(error( YABEntry *, const TQString &)), this, TQT_SIGNAL(modifyYABEntryError( YABEntry *, const TQString & )));
	myt->go(true);
}

void Client::deleteYABEntry(  YABEntry &entry )
{
	ModifyYABTask *myt = new ModifyYABTask( d->root );
	myt->setAction( ModifyYABTask::DeleteEntry );
	myt->setEntry( entry );
	myt->go(true);
}

// ***** Yahoo Chat *****
void Client::getYahooChatCategories()
{
	d->yahooChatTask->getYahooChatCategories();
}

void Client::getYahooChatRooms( const Yahoo::ChatCategory &category )
{
	d->yahooChatTask->getYahooChatRooms( category );
}

void Client::joinYahooChatRoom( const Yahoo::ChatRoom &room )
{
	d->yahooChatTask->joinRoom( room );
}

void Client::sendYahooChatMessage( const TQString &msg, const TQString &handle )
{
	d->yahooChatTask->sendYahooChatMessage( msg, handle );
}

void Client::leaveChat()
{
	d->yahooChatTask->logout();
}

// ***** other *****
void Client::notifyError( const TQString &info, const TQString & errorString, LogLevel level )
{
	kdDebug(YAHOO_RAW_DEBUG) << TQString::fromLatin1("\nThe following error occurred: %1\n    Reason: %2\n    LogLevel: %3")
		.arg(info).arg(errorString).arg(level) << endl;
	d->errorString = errorString;
	d->errorInformation = info;
	emit error( level );
}

TQString Client::userId()
{
	return d->user;
}

void Client::setUserId( const TQString & userId )
{
	d->user = userId;
}

Yahoo::Status Client::status()
{
	return d->status;
}

void Client::setStatus( Yahoo::Status status )
{
	d->status = status;
}


void Client::setStatusOnConnect( Yahoo::Status status )
{
	d->statusOnConnect = status;
}

void Client::setStatusMessageOnConnect( const TQString &msg )
{
	d->statusMessageOnConnect = msg;
}

void Client::setVerificationWord( const TQString &word )
{
	d->loginTask->setVerificationWord( word );
}

TQString Client::password()
{
	return d->pass;
}

TQString Client::host()
{
	return d->host;
}

int Client::port()
{
	return d->port;
}

uint Client::sessionID()
{
	return d->sessionID;
}

int Client::pictureFlag()
{
	return d->pictureFlag;
}

int Client::pictureChecksum()
{
	return d->pictureChecksum;
}

void Client::setPictureChecksum( int cs )
{
	d->pictureChecksum = cs;
}

TQString Client::yCookie()
{
	return d->yCookie;
}

TQString Client::tCookie()
{
	return d->tCookie;
}

TQString Client::cCookie()
{
	return d->cCookie;
}

void Client::distribute( Transfer * transfer )
{
	kdDebug(YAHOO_GEN_DEBUG) << k_funcinfo << endl;
	if( !rootTask()->take( transfer ) )
		kdDebug(YAHOO_RAW_DEBUG) << "CLIENT: root task refused transfer" << endl;
	delete transfer;
}

void Client::send( Transfer* request )
{
	kdDebug(YAHOO_RAW_DEBUG) << "CLIENT::send()" << endl;
	if( !d->stream )
	{
		kdDebug(YAHOO_RAW_DEBUG) << "CLIENT - NO STREAM TO SEND ON!" << endl;
		return;
	}

	d->stream->write( request );
}

void Client::debug(const TQString &str)
{
       tqDebug( "CLIENT: %s", str.ascii() );
}

Task * Client::rootTask()
{
	return d->root;
}

void Client::initTasks()
{
	if( d->tasksInitialized )
		return;

	d->statusTask = new StatusNotifierTask( d->root );
	TQObject::connect( d->statusTask, TQT_SIGNAL( statusChanged(const TQString&,int,const TQString&,int,int,int) ),
				TQT_SIGNAL( statusChanged(const TQString&,int,const TQString&,int,int,int) ) );
	TQObject::connect( d->statusTask, TQT_SIGNAL( stealthStatusChanged( const TQString&, Yahoo::StealthStatus ) ),
				TQT_SIGNAL( stealthStatusChanged( const TQString&, Yahoo::StealthStatus ) ) );
	TQObject::connect( d->statusTask, TQT_SIGNAL( loginResponse( int, const TQString& ) ),
				TQT_SLOT( slotLoginResponse( int, const TQString& ) ) );
	TQObject::connect( d->statusTask, TQT_SIGNAL( authorizationRejected( const TQString&, const TQString& ) ),
				TQT_SIGNAL( authorizationRejected( const TQString&, const TQString& ) ) );
	TQObject::connect( d->statusTask, TQT_SIGNAL( authorizationAccepted( const TQString& ) ),
				TQT_SIGNAL( authorizationAccepted( const TQString& ) ) );
	TQObject::connect( d->statusTask, TQT_SIGNAL( gotAuthorizationRequest( const TQString &, const TQString &, const TQString & ) ),
				TQT_SIGNAL( gotAuthorizationRequest( const TQString &, const TQString &, const TQString & ) ) );

	d->mailTask = new MailNotifierTask( d->root );
	TQObject::connect( d->mailTask, TQT_SIGNAL( mailNotify(const TQString&, const TQString&, int) ),
				TQT_SIGNAL( mailNotify(const TQString&, const TQString&, int) ) );

	d->messageReceiverTask = new MessageReceiverTask( d->root );
	TQObject::connect( d->messageReceiverTask, TQT_SIGNAL( gotIm(const TQString&, const TQString&, long, int) ),
				TQT_SIGNAL( gotIm(const TQString&, const TQString&, long, int) ) );
	TQObject::connect( d->messageReceiverTask, TQT_SIGNAL( systemMessage(const TQString&) ),
				TQT_SIGNAL( systemMessage(const TQString&) ) );
	TQObject::connect( d->messageReceiverTask, TQT_SIGNAL( gotTypingNotify(const TQString &, int) ),
				TQT_SIGNAL( typingNotify(const TQString &, int) ) );
	TQObject::connect( d->messageReceiverTask, TQT_SIGNAL( gotBuzz( const TQString &, long ) ),
				TQT_SIGNAL( gotBuzz( const TQString &, long ) ) );
	TQObject::connect( d->messageReceiverTask, TQT_SIGNAL( gotWebcamInvite(const TQString &) ),
				TQT_SIGNAL( gotWebcamInvite(const TQString &) ) );

	d->pictureNotifierTask = new PictureNotifierTask( d->root );
	TQObject::connect( d->pictureNotifierTask, TQT_SIGNAL( pictureStatusNotify( const TQString &, int ) ),
				TQT_SIGNAL( pictureStatusNotify( const TQString &, int ) ) );
	TQObject::connect( d->pictureNotifierTask, TQT_SIGNAL( pictureChecksumNotify( const TQString &, int ) ),
				TQT_SIGNAL( pictureChecksumNotify( const TQString &, int ) ) );
	TQObject::connect( d->pictureNotifierTask, TQT_SIGNAL( pictureInfoNotify( const TQString &, KURL, int ) ),
				TQT_SIGNAL( pictureInfoNotify( const TQString &, KURL, int ) ) );
	TQObject::connect( d->pictureNotifierTask, TQT_SIGNAL( pictureRequest( const TQString & ) ),
				TQT_SIGNAL( pictureRequest( const TQString & ) ) );
	TQObject::connect( d->pictureNotifierTask, TQT_SIGNAL( pictureUploaded( const TQString &, int ) ),
				TQT_SIGNAL( pictureUploaded( const TQString &, int ) ) );

	d->webcamTask = new WebcamTask( d->root );
	TQObject::connect( d->webcamTask, TQT_SIGNAL( webcamImageReceived( const TQString &, const TQPixmap &) ),
				TQT_SIGNAL( webcamImageReceived( const TQString &, const TQPixmap &) ) );
	TQObject::connect( d->webcamTask, TQT_SIGNAL( webcamNotAvailable( const TQString & ) ),
				TQT_SIGNAL( webcamNotAvailable( const TQString & ) ) );
	TQObject::connect( d->webcamTask, TQT_SIGNAL( webcamClosed( const TQString &, int ) ),
				TQT_SIGNAL( webcamClosed( const TQString &, int ) ) );
	TQObject::connect( d->webcamTask, TQT_SIGNAL( webcamPaused(const TQString&) ),
				TQT_SIGNAL( webcamPaused(const TQString&) ) );
	TQObject::connect( d->webcamTask, TQT_SIGNAL( readyForTransmission() ),
				TQT_SIGNAL( webcamReadyForTransmission() ) );
	TQObject::connect( d->webcamTask, TQT_SIGNAL( stopTransmission() ),
				TQT_SIGNAL( webcamStopTransmission() ) );
	TQObject::connect( d->webcamTask, TQT_SIGNAL( viewerJoined( const TQString &) ),
				TQT_SIGNAL( webcamViewerJoined( const TQString &) ) );
	TQObject::connect( d->webcamTask, TQT_SIGNAL( viewerLeft( const TQString &) ),
				TQT_SIGNAL( webcamViewerLeft( const TQString &) ) );
	TQObject::connect( d->webcamTask, TQT_SIGNAL( viewerRequest( const TQString &) ),
				TQT_SIGNAL( webcamViewerRequest( const TQString &) ) );

	d->conferenceTask = new ConferenceTask( d->root );
	TQObject::connect( d->conferenceTask, TQT_SIGNAL( gotInvite( const TQString &, const TQString &, const TQString &, const TQStringList & ) ),
				TQT_SIGNAL( gotConferenceInvite( const TQString &, const TQString &, const TQString &, const TQStringList & ) ) );
	TQObject::connect( d->conferenceTask, TQT_SIGNAL( gotMessage( const TQString &, const TQString &, const TQString & ) ),
				TQT_SIGNAL( gotConferenceMessage( const TQString &, const TQString &, const TQString & ) ) );
	TQObject::connect( d->conferenceTask, TQT_SIGNAL( userJoined( const TQString &, const TQString & ) ),
				TQT_SIGNAL( confUserJoined( const TQString &, const TQString & ) ) );
	TQObject::connect( d->conferenceTask, TQT_SIGNAL( userLeft( const TQString &, const TQString & ) ),
				TQT_SIGNAL( confUserLeft( const TQString &, const TQString & ) ) );
	TQObject::connect( d->conferenceTask, TQT_SIGNAL( userDeclined( const TQString &, const TQString &, const TQString & ) ),
				TQT_SIGNAL( confUserDeclined( const TQString &, const TQString &, const TQString & ) ) );

	d->yabTask = new YABTask( d->root );
	TQObject::connect( d->yabTask, TQT_SIGNAL( gotEntry( YABEntry * ) ),
				TQT_SIGNAL( gotYABEntry( YABEntry * ) ) );
	TQObject::connect( d->yabTask, TQT_SIGNAL( gotRevision( long, bool ) ),
				TQT_SIGNAL( gotYABRevision( long, bool ) ) );

	d->fileTransferTask = new FileTransferNotifierTask( d->root );
	TQObject::connect( d->fileTransferTask, TQT_SIGNAL(incomingFileTransfer( const TQString &, const TQString &,
					long, const TQString &, const TQString &, unsigned long, const TQPixmap & )),
				TQT_SIGNAL(incomingFileTransfer( const TQString &, const TQString &,
					long, const TQString &, const TQString &, unsigned long, const TQPixmap & )) );

	d->yahooChatTask = new YahooChatTask( d->root );
	TQObject::connect( d->yahooChatTask, TQT_SIGNAL(gotYahooChatCategories( const TQDomDocument & )),
				TQT_SIGNAL(gotYahooChatCategories( const TQDomDocument & )) );
	TQObject::connect( d->yahooChatTask, TQT_SIGNAL(gotYahooChatRooms( const Yahoo::ChatCategory &, const TQDomDocument & )),
				TQT_SIGNAL(gotYahooChatRooms( const Yahoo::ChatCategory &, const TQDomDocument & )) );
	TQObject::connect( d->yahooChatTask, TQT_SIGNAL(chatRoomJoined( int , int , const TQString &, const TQString & ) ),
				TQT_SIGNAL(chatRoomJoined( int , int , const TQString &, const TQString & ) ) );
	TQObject::connect( d->yahooChatTask, TQT_SIGNAL(chatBuddyHasJoined( const TQString &, const TQString &, bool  ) ),
				TQT_SIGNAL(chatBuddyHasJoined( const TQString &, const TQString &, bool  ) ) );
	TQObject::connect( d->yahooChatTask, TQT_SIGNAL(chatBuddyHasLeft(TQString,TQString) ),
				TQT_SIGNAL(chatBuddyHasLeft(TQString,TQString) ) );
	TQObject::connect( d->yahooChatTask, TQT_SIGNAL(chatMessageReceived( const TQString &, const TQString &, const TQString & ) ),
				TQT_SIGNAL(chatMessageReceived( const TQString &, const TQString &, const TQString & ) ) );
}

void Client::deleteTasks()
{
	d->tasksInitialized = false;
	d->statusTask->deleteLater();
	d->statusTask = 0L;
	d->mailTask->deleteLater();
	d->mailTask = 0L;
	d->messageReceiverTask->deleteLater();
	d->messageReceiverTask = 0L;
	d->pictureNotifierTask->deleteLater();
	d->pictureNotifierTask = 0L;
	d->webcamTask->deleteLater();
	d->webcamTask = 0L;
	d->conferenceTask->deleteLater();
	d->conferenceTask = 0L;
	d->yabTask->deleteLater();
	d->yabTask = 0L;
	d->fileTransferTask->deleteLater();
	d->fileTransferTask = 0;
	d->yahooChatTask->deleteLater();
	d->yahooChatTask = 0;
	d->receiveFileTask->deleteLater();
	d->receiveFileTask = 0;
}

#include "client.moc"
