#include "feedbrowser.h"

#include <tdeaboutdata.h>
#include <tdeapplication.h>
#include <tdecmdlineargs.h>
#include <kdebug.h>
#include <kdialogbase.h>
#include <tdelistview.h>
#include <tdelocale.h>
#include <dcopclient.h>

#include <tqcstring.h>
#include <tqdatastream.h>
#include <tqvbox.h>

CategoryItem::CategoryItem( TDEListView *parent, const TQString &category )
	: TDEListViewItem( parent ),
	m_category( category )
{
	init();
}

CategoryItem::CategoryItem( TDEListViewItem *parent, const TQString &category )
	: TDEListViewItem( parent ),
	m_category( category )
{
	init();
}

void CategoryItem::init()
{
	m_populated = false;
	m_dcopIface = 0;

	setText( 0, m_category.mid( m_category.findRev( '/' ) + 1 ).replace( '_', ' ' ) );
}

void CategoryItem::setOpen( bool open )
{
	if ( open && !m_populated ) {
		populate();
		m_populated = true;
	}
	TDEListViewItem::setOpen( open );
}

void CategoryItem::populate()
{
	m_dcopIface = new DCOPRSSIface( this, "m_dcopIface" );
	connect( m_dcopIface, TQT_SIGNAL( gotCategories( const TQStringList & ) ),
	         this, TQT_SLOT( gotCategories( const TQStringList & ) ) );
	m_dcopIface->getCategories( m_category );
}

void CategoryItem::gotCategories( const TQStringList &categories )
{
	delete m_dcopIface;
	m_dcopIface = 0;

	TQStringList::ConstIterator it = categories.begin();
	TQStringList::ConstIterator end = categories.end();
	for ( ; it != end; ++it )
		new CategoryItem( this, *it );

	if ( !categories.isEmpty() )
		TDEListViewItem::setOpen( true );
}

DCOPRSSIface::DCOPRSSIface( TQObject *parent, const char *name ) :
	TQObject( parent, name ), DCOPObject( "FeedBrowser" )
{
	connectDCOPSignal( "rssservice", "RSSQuery", "gotCategories(TQStringList)",
	                   "slotGotCategories(TQStringList)", false );
}

void DCOPRSSIface::getCategories( const TQString &cat )
{
	TQByteArray data;
	TQDataStream stream( data, IO_WriteOnly );
	stream << cat;
	kapp->dcopClient()->send( "rssservice", "RSSQuery",
	                          "getCategories(TQString)", data );
}

void DCOPRSSIface::slotGotCategories( const TQStringList &categories )
{
	emit gotCategories( categories );
}

FeedBrowserDlg::FeedBrowserDlg( TQWidget *parent, const char *name )
	: KDialogBase( parent, name, true, i18n( "DCOPRSS Feed Browser" ),
	               Close, Close, true )
{
	m_dcopIface = new DCOPRSSIface( TQT_TQOBJECT(this), "m_dcopIface" );
	connect( m_dcopIface, TQT_SIGNAL( gotCategories( const TQStringList & ) ),
	         this, TQT_SLOT( gotTopCategories( const TQStringList & ) ) );

	TQVBox *mainWidget = makeVBoxMainWidget();

	m_feedList = new TDEListView( mainWidget, "m_feedList" );
	m_feedList->setAllColumnsShowFocus( true );
	m_feedList->setRootIsDecorated( true );
	m_feedList->addColumn( i18n( "Name" ) );
	connect( m_feedList, TQT_SIGNAL( executed( TQListViewItem * ) ),
	         this, TQT_SLOT( itemSelected( TQListViewItem * ) ) );
	connect( m_feedList, TQT_SIGNAL( returnPressed( TQListViewItem * ) ),
	         this, TQT_SLOT( itemSelected( TQListViewItem * ) ) );

	resize( 500, 400 );

	getTopCategories();
}

void FeedBrowserDlg::getTopCategories()
{
	m_dcopIface->getCategories( "Top" );
}

void FeedBrowserDlg::gotTopCategories( const TQStringList &categories )
{
	TQStringList::ConstIterator it = categories.begin();
	TQStringList::ConstIterator end = categories.end();
	for ( ; it != end; ++it )
		new CategoryItem( m_feedList, *it );
}

void FeedBrowserDlg::itemSelected( TQListViewItem *item )
{
	item->setOpen( !item->isOpen() );
}

int main( int argc, char **argv )
{
	TDEGlobal::locale()->setMainCatalogue( "dcoprss" );
	TDEAboutData aboutData( "feedbrowser", I18N_NOOP( "Feed Browser" ), "0.1" );
	TDECmdLineArgs::init( argc, argv, &aboutData );
	TDEApplication app;
	FeedBrowserDlg *dlg = new FeedBrowserDlg( 0 );
	app.setMainWidget( dlg );
	dlg->show();
	return app.exec();
}

#include "feedbrowser.moc"
