//===-- llvm/AutoUpgrade.h - AutoUpgrade Helpers ----------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
//  These functions are implemented by lib/VMCore/AutoUpgrade.cpp.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_AUTOUPGRADE_H
#define LLVM_AUTOUPGRADE_H

namespace llvm {
  class Module;
  class GlobalVariable;
  class Function;
  class CallInst;

  /// This is a more granular function that simply checks an intrinsic function 
  /// for upgrading, and returns true if it requires upgrading. It may return
  /// null in NewFn if the all calls to the original intrinsic function
  /// should be transformed to non-function-call instructions.
  bool UpgradeIntrinsicFunction(Function *F, Function *&NewFn);

  /// This is the complement to the above, replacing a specific call to an 
  /// intrinsic function with a call to the specified new function.
  void UpgradeIntrinsicCall(CallInst *CI, Function *NewFn);
  
  /// This is an auto-upgrade hook for any old intrinsic function syntaxes 
  /// which need to have both the function updated as well as all calls updated 
  /// to the new function. This should only be run in a post-processing fashion 
  /// so that it can update all calls to the old function.
  void UpgradeCallsToIntrinsic(Function* F);

  /// This checks for global variables which should be upgraded. It returns true
  /// if it requires upgrading.
  bool UpgradeGlobalVariable(GlobalVariable *GV);

  /// This function checks debug info intrinsics. If an intrinsic is invalid
  /// then this function simply removes the intrinsic. 
  void CheckDebugInfoIntrinsics(Module *M);
  
  /// This function upgrades the old pre-3.0 exception handling system to the
  /// new one. N.B. This will be removed in 3.1.
  void UpgradeExceptionHandling(Module *M);
} // End llvm namespace

#endif
