/**************************************************************************
**
** Copyright (C) 2020 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the Qt Installer Framework.
**
** $QT_BEGIN_LICENSE:GPL-EXCEPT$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
**************************************************************************/
#include "../shared/packagemanager.h"

#include <simplemovefileoperation.h>

#include <packagemanagercore.h>

#include <utils.h>

#include <QObject>
#include <QFile>
#include <QTest>

using namespace KDUpdater;
using namespace QInstaller;

class tst_simplemovefileoperation : public QObject
{
    Q_OBJECT

private:
    void createDummyFile(const QString &source, const QString &destination)
    {
        // Create dummy original file for move destination
        QFile destFile(destination);
        QVERIFY(destFile.open(QIODevice::WriteOnly));
        destFile.close();

        QByteArray testString("Generated by QTest\n");
        QFile testFile(source);
        QVERIFY(testFile.open(QIODevice::WriteOnly | QIODevice::Text));
        QTextStream out(&testFile);
        out << testString;
        testFile.close();
        QVERIFY(QFileInfo(source).exists());
    }

private slots:
    void testMissingArguments()
    {
        SimpleMoveFileOperation op(nullptr);

        op.backup();
        QVERIFY(op.testOperation());
        QVERIFY(!op.performOperation());

        QCOMPARE(UpdateOperation::Error(op.error()), UpdateOperation::InvalidArguments);
        QCOMPARE(op.errorString(), QString("Invalid arguments in SimpleMoveFile: "
                                           "0 arguments given, exactly 2 arguments expected."));

        op.setArguments(QStringList() << "" << "");
        QVERIFY(!op.performOperation());

        QCOMPARE(UpdateOperation::Error(op.error()), UpdateOperation::UserDefinedError);
        QCOMPARE(op.errorString(), QString("None of the arguments can be empty: "
                                           "source \"\", target \"\"."));
    }

    void initTestCase_data()
    {
        QTest::addColumn<QString>("source");
        QTest::addColumn<QString>("destination");
        QTest::newRow("relative") << "test1" << "test2";
        QTest::newRow("absolute")
            << qApp->applicationDirPath() + QDir::toNativeSeparators("/test")
            << generateTemporaryFileName();
    }

    void testMoveFileWithOverwrite()
    {
        QFETCH_GLOBAL(QString, source);
        QFETCH_GLOBAL(QString, destination);

        createDummyFile(source, destination);
        QByteArray testFileHash = QInstaller::calculateHash(source, QCryptographicHash::Sha1);

        SimpleMoveFileOperation op(nullptr);
        op.setArguments(QStringList() << source << destination);

        QVERIFY2(op.performOperation(), op.errorString().toLatin1());

        QVERIFY(!QFileInfo(source).exists());
        QByteArray destinationFileHash = QInstaller::calculateHash(destination, QCryptographicHash::Sha1);
        QVERIFY(testFileHash == destinationFileHash);

        QVERIFY2(op.undoOperation(), op.errorString().toLatin1());

        QVERIFY(!QFileInfo(destination).exists());
        QByteArray sourceFileHash = QInstaller::calculateHash(source, QCryptographicHash::Sha1);
        QVERIFY(testFileHash == sourceFileHash);

        QVERIFY(QFile(source).remove());
    }

    void testMoveFileNoUndo()
    {
        QFETCH_GLOBAL(QString, source);
        QFETCH_GLOBAL(QString, destination);

        createDummyFile(source, destination);

        QByteArray testFileHash = QInstaller::calculateHash(source, QCryptographicHash::Sha1);

        SimpleMoveFileOperation op(nullptr);
        op.setArguments(QStringList() << source << destination << "UNDOOPERATION" << "");

        QVERIFY2(op.performOperation(), op.errorString().toLatin1());
        QVERIFY(!QFileInfo(source).exists());
        QByteArray destinationFileHash = QInstaller::calculateHash(destination, QCryptographicHash::Sha1);
        QVERIFY(testFileHash == destinationFileHash);

        QVERIFY2(op.undoOperation(), op.errorString().toLatin1());
        QVERIFY(QFileInfo(destination).exists());
        QVERIFY(testFileHash == destinationFileHash);
        QVERIFY(QFile(destination).remove());
    }

    void testPerformingFromCLI()
    {
        QString installDir = QInstaller::generateTemporaryFileName();
        QVERIFY(QDir().mkpath(installDir));
        PackageManagerCore *core = PackageManager::getPackageManagerWithInit
                (installDir, ":///data/repository");

        QString destinationDir = installDir + QDir::separator() + "destination";
        QVERIFY(QDir().mkpath(destinationDir));

        // Matches filename in component install script
        QFile file(installDir + QDir::toNativeSeparators("/test"));
        QVERIFY(file.open(QIODevice::ReadWrite));
        file.close();

        core->installDefaultComponentsSilently();
        QVERIFY(!file.exists());

        QFile movedFile(destinationDir + QDir::separator() + "test");
        QVERIFY(movedFile.exists());

        core->setPackageManager();
        core->commitSessionOperations();
        core->uninstallComponentsSilently(QStringList() << "A");
        QVERIFY(!movedFile.exists() && file.exists());

        QDir dir(installDir);
        QVERIFY(dir.removeRecursively());
        core->deleteLater();
    }
};

QTEST_MAIN(tst_simplemovefileoperation)

#include "tst_simplemovefileoperation.moc"
